/*
 * Decompiled with CFR 0.152.
 */
package org.webjars;

import com.fasterxml.jackson.core.JsonFactory;
import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.core.JsonToken;
import java.io.Closeable;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.Reader;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.StandardCopyOption;
import java.nio.file.attribute.PosixFilePermission;
import java.util.Collection;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import org.apache.commons.compress.archivers.zip.ZipArchiveEntry;
import org.apache.commons.compress.archivers.zip.ZipFile;
import org.apache.commons.lang3.SystemUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.webjars.WebJarAssetLocator;

public class WebJarExtractor {
    public static final String PACKAGE_JSON = "package.json";
    public static final String BOWER_JSON = "bower.json";
    private static final String JAR_PATH_DELIMITER = "/";
    private static final Logger log = LoggerFactory.getLogger(WebJarExtractor.class);
    private final Cache cache;
    private final ClassLoader classLoader;
    public static Cache NO_CACHE = new NoCache();

    public WebJarExtractor() {
        this(WebJarExtractor.class.getClassLoader());
    }

    public WebJarExtractor(ClassLoader classLoader) {
        this(NO_CACHE, classLoader);
    }

    public WebJarExtractor(Cache cache, ClassLoader classLoader) {
        this.cache = cache;
        this.classLoader = classLoader;
    }

    public void extractAllWebJarsTo(File to) throws IOException {
        this.extractWebJarsTo(null, null, to);
    }

    public void extractWebJarTo(String name, File to) throws IOException {
        this.extractWebJarsTo(name, null, to);
    }

    public void extractAllNodeModulesTo(File to) throws IOException {
        this.extractWebJarsTo(null, PACKAGE_JSON, to);
    }

    public void extractAllBowerComponentsTo(File to) throws IOException {
        this.extractWebJarsTo(null, BOWER_JSON, to);
    }

    private void extractWebJarsTo(String name, String moduleNameFile, File to) throws IOException {
        boolean useModuleName = moduleNameFile != null;
        String fullPath = "META-INF/resources/webjars/";
        String searchPath = name == null ? fullPath : fullPath + name + JAR_PATH_DELIMITER;
        for (URL url : WebJarAssetLocator.listParentURLsWithResource(new ClassLoader[]{this.classLoader}, searchPath)) {
            if ("jar".equals(url.getProtocol())) {
                String urlPath = url.getPath();
                File file = new File(URI.create(urlPath.substring(0, urlPath.indexOf("!"))));
                log.debug("Loading webjars from {}", (Object)file);
                ZipFile zipFile = new ZipFile(file, StandardCharsets.UTF_8.name());
                Throwable throwable = null;
                try {
                    Collection<JarFileWebJar> webJars = this.findWebJarsInJarFile(zipFile, moduleNameFile);
                    for (JarFileWebJar webJar : webJars) {
                        String webJarId;
                        if (name != null && !webJar.name.equals(name) || (webJarId = useModuleName ? webJar.getWebModuleId() : webJar.name) == null) continue;
                        for (Map.Entry<String, ZipArchiveEntry> entry : webJar.entries.entrySet()) {
                            String entryName = entry.getKey();
                            if (entry.getValue().isDirectory()) continue;
                            String relativeName = webJarId + File.separator + entryName;
                            this.copyZipEntry(zipFile, entry.getValue(), new File(to, relativeName), relativeName);
                        }
                    }
                    continue;
                }
                catch (Throwable throwable2) {
                    throwable = throwable2;
                    throw throwable2;
                }
                finally {
                    if (zipFile == null) continue;
                    if (throwable != null) {
                        try {
                            zipFile.close();
                        }
                        catch (Throwable throwable3) {
                            throwable.addSuppressed(throwable3);
                        }
                        continue;
                    }
                    zipFile.close();
                    continue;
                }
            }
            if ("file".equals(url.getProtocol())) {
                File file;
                try {
                    file = new File(url.toURI());
                }
                catch (URISyntaxException e) {
                    file = new File(url.getPath());
                }
                log.debug("Found file system webjar: {}", (Object)file);
                File[] webjars = name == null ? file.listFiles() : new File[]{file};
                if (webjars != null) {
                    for (File webjar : webjars) {
                        if (webjar.isDirectory()) {
                            File[] versions = webjar.listFiles();
                            if (versions != null) {
                                for (File version : versions) {
                                    if (version.isDirectory()) {
                                        String moduleId = useModuleName ? this.getFileNodeModuleIdEntry(new File(version, moduleNameFile)) : webjar.getName();
                                        if (moduleId == null) continue;
                                        File copyTo = new File(to, moduleId);
                                        this.copyDirectory(version, copyTo, webjar.getName());
                                        continue;
                                    }
                                    log.debug("Filesystem webjar version {} is not a directory", (Object)version);
                                }
                                continue;
                            }
                            log.debug("Filesystem webjar has no versions: {}", (Object)webjar);
                            continue;
                        }
                        log.debug("Filesystem webjar {} is not a directory", (Object)webjar);
                    }
                    continue;
                }
                log.debug("Filesystem webjar has no webjars: {}", (Object)file);
                continue;
            }
            log.debug("Ignoring given unsupported protocol for: {}", (Object)url);
        }
    }

    private void ensureIsDirectory(File dir) {
        boolean created;
        if (dir.exists() && !dir.isDirectory()) {
            log.debug("Destination directory is not a directory, deleting {}", (Object)dir);
            boolean isDeleted = dir.delete();
            if (!isDeleted) {
                log.debug("Destination directory {} wasn't deleted", (Object)dir);
            }
        }
        if (!(created = dir.mkdirs())) {
            log.debug("Destination directory {} didn't need creation", (Object)dir);
        }
    }

    private void copyDirectory(File directory, File to, String key) throws IOException {
        File[] files = directory.listFiles();
        if (files != null) {
            for (File file : files) {
                File copyTo = new File(to, file.getName());
                String relativeName = key.isEmpty() ? file.getName() : key + File.separator + file.getName();
                if (file.isDirectory()) {
                    this.copyDirectory(file, copyTo, relativeName);
                    continue;
                }
                Cacheable forCache = new Cacheable(file.getPath(), file.lastModified());
                log.debug("Checking whether {} is up to date at {}", (Object)relativeName, (Object)copyTo);
                if (copyTo.exists() && this.cache.isUpToDate(relativeName, forCache)) continue;
                log.debug("Up to date check failed, copying {} to {}", (Object)relativeName, (Object)copyTo);
                this.ensureIsDirectory(copyTo.getParentFile());
                Files.copy(file.toPath(), copyTo.toPath(), StandardCopyOption.COPY_ATTRIBUTES, StandardCopyOption.REPLACE_EXISTING);
                this.cache.put(relativeName, forCache);
            }
        }
    }

    private Collection<JarFileWebJar> findWebJarsInJarFile(ZipFile zipFile, String moduleNameFile) throws IOException {
        HashMap<String, JarFileWebJar> webJars = new HashMap<String, JarFileWebJar>();
        Enumeration entries = zipFile.getEntries();
        while (entries.hasMoreElements()) {
            String webJarPath;
            String[] nameVersion;
            ZipArchiveEntry entry = (ZipArchiveEntry)entries.nextElement();
            if (!entry.getName().startsWith("META-INF/resources/webjars/") || (nameVersion = (webJarPath = entry.getName().substring("META-INF/resources/webjars".length() + 1)).split(JAR_PATH_DELIMITER, 3)).length != 3) continue;
            String name = nameVersion[0];
            String version = nameVersion[1];
            String path = nameVersion[2];
            String key = name + JAR_PATH_DELIMITER + version;
            JarFileWebJar webJar = (JarFileWebJar)webJars.get(key);
            if (webJar == null) {
                webJar = new JarFileWebJar(name, version);
                webJars.put(key, webJar);
            }
            webJar.entries.put(path, entry);
            if (!Objects.equals(path, moduleNameFile)) continue;
            webJar.moduleId = this.getJsonModuleId(WebJarExtractor.copyAndClose(zipFile.getInputStream(entry)));
        }
        return webJars.values();
    }

    private void copyZipEntry(ZipFile zipFile, ZipArchiveEntry entry, File copyTo, String key) throws IOException {
        Cacheable forCache = new Cacheable(entry.getName(), entry.getTime());
        log.debug("Checking whether {} is up to date at {}", (Object)entry.getName(), (Object)copyTo);
        if (!copyTo.exists() || !this.cache.isUpToDate(key, forCache)) {
            int mode;
            log.debug("Up to date check failed, copying {} to {}", (Object)entry.getName(), (Object)copyTo);
            this.ensureIsDirectory(copyTo.getParentFile());
            WebJarExtractor.copyAndClose(zipFile.getInputStream(entry), copyTo);
            if (SystemUtils.IS_OS_UNIX && (mode = entry.getUnixMode()) > 0) {
                Files.setPosixFilePermissions(copyTo.toPath(), WebJarExtractor.toPerms(mode));
            }
            this.cache.put(key, forCache);
        }
    }

    private String getFileNodeModuleIdEntry(File packageJsonFile) throws IOException {
        String moduleId = null;
        if (packageJsonFile.exists()) {
            String packageJson = WebJarExtractor.copyAndClose(new FileInputStream(packageJsonFile));
            moduleId = this.getJsonModuleId(packageJson);
        }
        return moduleId;
    }

    String getJsonModuleId(String packageJson) throws IOException {
        JsonFactory factory = new JsonFactory();
        JsonParser parser = factory.createParser(packageJson);
        if (parser.nextToken() != JsonToken.START_OBJECT) {
            throw new IOException("package.json is not a valid JSON object");
        }
        String moduleId = null;
        while (moduleId == null) {
            parser.nextToken();
            String fieldName = parser.getCurrentName();
            if (!"name".equals(fieldName) || !parser.getParsingContext().getParent().inRoot()) continue;
            parser.nextToken();
            moduleId = parser.getText();
        }
        parser.close();
        return moduleId;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private static void copyAndClose(InputStream source, File to) throws IOException {
        try (FileOutputStream dest = new FileOutputStream(to);){
            byte[] buffer = new byte[8192];
            int read = source.read(buffer);
            while (read > 0) {
                ((OutputStream)dest).write(buffer, 0, read);
                read = source.read(buffer);
            }
            dest.flush();
        }
        finally {
            WebJarExtractor.closeQuietly(source);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private static String copyAndClose(InputStream source) throws IOException {
        StringBuilder sb = new StringBuilder();
        try (InputStreamReader is = new InputStreamReader(source, StandardCharsets.UTF_8);){
            char[] buffer = new char[8192];
            int read = ((Reader)is).read(buffer, 0, buffer.length);
            sb.append(buffer, 0, read);
        }
        finally {
            WebJarExtractor.closeQuietly(source);
        }
        return sb.toString();
    }

    private static void closeQuietly(Closeable closeable) {
        if (closeable != null) {
            try {
                closeable.close();
            }
            catch (IOException e) {
                log.debug("Exception while closing resource", (Throwable)e);
            }
        }
    }

    private static Set<PosixFilePermission> toPerms(int mode) {
        HashSet<PosixFilePermission> perms = new HashSet<PosixFilePermission>();
        if ((mode & 0x100) > 0) {
            perms.add(PosixFilePermission.OWNER_READ);
        }
        if ((mode & 0x80) > 0) {
            perms.add(PosixFilePermission.OWNER_WRITE);
        }
        if ((mode & 0x40) > 0) {
            perms.add(PosixFilePermission.OWNER_EXECUTE);
        }
        if ((mode & 0x20) > 0) {
            perms.add(PosixFilePermission.GROUP_READ);
        }
        if ((mode & 0x10) > 0) {
            perms.add(PosixFilePermission.GROUP_WRITE);
        }
        if ((mode & 8) > 0) {
            perms.add(PosixFilePermission.GROUP_EXECUTE);
        }
        if ((mode & 4) > 0) {
            perms.add(PosixFilePermission.OTHERS_READ);
        }
        if ((mode & 2) > 0) {
            perms.add(PosixFilePermission.OTHERS_WRITE);
        }
        if ((mode & 1) > 0) {
            perms.add(PosixFilePermission.OTHERS_EXECUTE);
        }
        return perms;
    }

    public static final class Cacheable {
        private final String path;
        private final long lastModified;

        public Cacheable(String path, long lastModified) {
            this.path = path;
            this.lastModified = lastModified;
        }

        public String getPath() {
            return this.path;
        }

        public long getLastModified() {
            return this.lastModified;
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            Cacheable cacheable = (Cacheable)o;
            return this.lastModified == cacheable.lastModified && this.path.equals(cacheable.path);
        }

        public int hashCode() {
            int result = this.path.hashCode();
            result = 31 * result + (int)(this.lastModified ^ this.lastModified >>> 32);
            return result;
        }
    }

    public static class MemoryCache
    implements Cache {
        private final Map<String, Cacheable> cache = new HashMap<String, Cacheable>();

        @Override
        public boolean isUpToDate(String key, Cacheable cacheable) {
            return cacheable.equals(this.cache.get(key));
        }

        @Override
        public void put(String key, Cacheable cacheable) {
            this.cache.put(key, cacheable);
        }
    }

    private static class NoCache
    implements Cache {
        private NoCache() {
        }

        @Override
        public boolean isUpToDate(String key, Cacheable cacheable) {
            return false;
        }

        @Override
        public void put(String key, Cacheable cacheable) {
        }
    }

    public static interface Cache {
        public boolean isUpToDate(String var1, Cacheable var2);

        public void put(String var1, Cacheable var2);
    }

    private class JarFileWebJar {
        final String name;
        final String version;
        final Map<String, ZipArchiveEntry> entries = new HashMap<String, ZipArchiveEntry>();
        String moduleId;

        private JarFileWebJar(String name, String version) {
            this.name = name;
            this.version = version;
        }

        private String getWebModuleId() {
            return this.moduleId == null ? this.name : this.moduleId;
        }
    }
}

