package com.rapid.j2ee.framework.batchjob;

import java.io.File;
import java.util.List;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;

import com.rapid.j2ee.framework.batchjob.constants.RapidSystemBatchJobTaskConstants;
import com.rapid.j2ee.framework.batchjob.constants.RapidSystemBatchJobTaskStatus;
import com.rapid.j2ee.framework.batchjob.domain.SystemBatchJobTask;
import com.rapid.j2ee.framework.batchjob.persistence.RapidSystemBatchJobTaskMapper;
import com.rapid.j2ee.framework.core.exception.ExceptionUtils;
import com.rapid.j2ee.framework.core.exception.NetworkDisconnectSystemException;
import com.rapid.j2ee.framework.core.io.file.FileWriter;

import com.rapid.j2ee.framework.core.spring.SpringPropertiesHolder;
import com.rapid.j2ee.framework.core.utils.DateTimeUtils;
import com.rapid.j2ee.framework.core.utils.FileUtils;
import com.rapid.j2ee.framework.core.utils.IPAddressUtils;
import com.rapid.j2ee.framework.core.utils.StringUtils;
import com.rapid.j2ee.framework.core.utils.TypeChecker;
import com.rapid.j2ee.framework.core.utils.support.DateTimeFormat;
import com.rapid.j2ee.framework.orm.medium.MediumSessionMapperOperations;

/**
 * 
 * 提供后台任务分配处理的Service，开发人员只要提供业务的doExecuteJob实现即可。
 * 
 * 
 * @author John Hao
 * 
 */
public abstract class AbstractRapidSystemBatchJobTaskFire implements
		RapidSystemBatchJobTaskConstants {

	private int connectionErrorCount;

	public AbstractRapidSystemBatchJobTaskFire(String funcType,
			String subFuncType) {
		this.funcType = funcType;
		this.subFuncType = subFuncType;
	}

	/**
	 * 可以通过重载该方法，定制自己的查询未处理的任务的逻辑
	 * 
	 * @return List<SystemBatchJobTask> 系统JOB任务集合
	 * 
	 */
	protected List<SystemBatchJobTask> getSystemJobLogsUnprocessed(
			String funcType, String subFuncType) {

		return systemJobLogBatchMapper.getSystemJobLogsUnprocessed(funcType,
				subFuncType);

	}

	/**
	 * 处理后台Job
	 * 
	 */
	public void execute() {

		this.startConnectionErrorCount();

		List<SystemBatchJobTask> systemJobLogs = this
				.getSystemJobLogsUnprocessed(this.funcType, this.subFuncType);

		if (TypeChecker.isEmpty(systemJobLogs)) {
			return;
		}

		for (SystemBatchJobTask systemJobLog : systemJobLogs) {

			if (this.isSystemJobTerminatedByConnectionError()) {
				return;
			}

			this.executeJobWithCheckRunable(systemJobLog);
		}

	}

	private void startConnectionErrorCount() {
		connectionErrorCount = 0;
	}

	private boolean isSystemJobTerminatedByConnectionError() {
		return this.connectionErrorCount > Enforce_Terminate_System_Job_As_Connection_Error_Max_Permit_Value;
	}

	private void executeJobWithCheckRunable(SystemBatchJobTask systemJobLog) {
		try {

			log.info("executeJobWithCheckRunable SystemJobLog:" + systemJobLog
					+ " in thread " + Thread.currentThread().getId());

			this.prepareSystemJobLogForStart(systemJobLog);

			if (!isSystemJobCanBeRunable(systemJobLog)) {

				log.info("executeJob has been executed SystemJobLog:"
						+ systemJobLog + " in thread "
						+ Thread.currentThread().getId());

				return;
			}

		} catch (Throwable ep) {
			log.error("isSystemJobCanBeRunable error!!!! SystemJobLog:"
					+ systemJobLog, ep);
			return;
		}

		try {

			log.info("executeJob is being processed SystemJobLog:"
					+ systemJobLog + " in thread "
					+ Thread.currentThread().getId());

			this.doExecuteJob(systemJobLog);

		} catch (NetworkDisconnectSystemException networkDisconnectSystemException) {

			log.error("executeJob error in process!  SystemJobLog:"
					+ systemJobLog + " in thread "
					+ Thread.currentThread().getId(),
					networkDisconnectSystemException);

			connectionErrorCount++;

			// Roll back Processing Job....
			systemJobLog.setJobStatus(RapidSystemBatchJobTaskStatus.ERROR
					.getStatus());
			systemJobLog
					.setJobExpErrorReason(StringUtils
							.getStringContextWithinLength(
									ExceptionUtils
											.printThrowableStack(networkDisconnectSystemException),
									2.5, 2500, "..."));
		} catch (Throwable ep) {

			log.error("executeJob error in process!  SystemJobLog:"
					+ systemJobLog + " in thread "
					+ Thread.currentThread().getId(), ep);

			// Roll back Processing Job....
			systemJobLog.setJobStatus(RapidSystemBatchJobTaskStatus.ERROR
					.getStatus());
			systemJobLog.setJobResponseCode("");
			systemJobLog.setJobResponseMessage("");
			systemJobLog.setJobExpErrorReason(StringUtils
					.getStringContextWithinLength(ExceptionUtils
							.printThrowableStack(ep), 2.5, 2500, "..."));

		} finally {
			this.updateSystemJobLogForEnd(systemJobLog);

		}
	}

	/**
	 * 抽象方法，用于完成自己的业务
	 * 
	 * @param SystemBatchJobTask
	 *            系统JOB日志
	 */
	protected abstract void doExecuteJob(SystemBatchJobTask systemJobLog)
			throws Throwable;

	private void prepareSystemJobLogForStart(SystemBatchJobTask systemJobLog) {

		systemJobLog.setJobProcessServer(IPAddressUtils.getLocalHostIP());
		systemJobLog.setJobStatus(RapidSystemBatchJobTaskStatus.Processing
				.getStatus());
		systemJobLog.setJobStartDate(DateTimeUtils.getCurrentDate());
		systemJobLog.setJobExecCount(systemJobLog.getJobExecCount() + 1);
		systemJobLog.setJobExpErrorReason("");
	}

	private void updateSystemJobLogForEnd(SystemBatchJobTask systemJobLog) {

		systemJobLog.setJobEndDate(DateTimeUtils.getCurrentDate());

		mediumSessionMapperOperations.update(systemJobLog);

		log.info("executeJob is processed SystemJobLog:" + systemJobLog
				+ " in thread " + Thread.currentThread().getId());
	}

	private boolean isSystemJobCanBeRunable(SystemBatchJobTask systemJobLog) {

		return systemJobLogBatchMapper
				.updateSystemJobLogForStartJob(systemJobLog) == 1;

	}

	public final String saveSystemJobResourceAsLog(String fileIndicator,
			SystemBatchJobTask systemJobLog, String resourceContent,
			String charset) {

		String mainFolder = getSystemBatchJobLogPath(SystemBatchJobTask.class
				.getName());

		if (TypeChecker.isEmpty(mainFolder)) {
			log.info("No log SystemBatchJobTasksFireService!");
			return "";
		}

		String targetFolder = FileUtils.getFullFilePathName(mainFolder,
				DateTimeUtils.convertDateToWeb(systemJobLog.getCreateDate(),
						DateTimeFormat.YYYY_MM_DD));

		targetFolder = FileUtils.getFullFilePathName(FileUtils
				.getFullFilePathName(targetFolder, funcType), this.subFuncType);

		FileUtils.makeDir(targetFolder);

		File file = new File(targetFolder, StringUtils.getStringBunch(
				systemJobLog.getJobReferNos(), "-")
				+ "-" + fileIndicator + ".data");

		log.info("SaveSystemJobResourceAsLog Charset:" + charset + " / "
				+ systemJobLog + " in thread " + this.getClass().getName());

		FileWriter fileWriter = new FileWriter(file, charset);

		fileWriter.write(resourceContent);

		fileWriter.flush();

		fileWriter.close();

		log.info("System Job Resource Log:" + file.getAbsolutePath());

		return StringUtils.substringAfter(fileWriter.getFile()
				.getAbsolutePath(), new File(mainFolder).getAbsolutePath());

	}

	protected String getSystemBatchJobLogPath(String categoryName) {
		return System.getProperty("RapidBatchJob.ResourceLog.Path");
	}

	// ORM 数据库操作
	@Autowired
	@Qualifier("RapidBatchJob_SessionMapper")
	private MediumSessionMapperOperations mediumSessionMapperOperations;

	@Autowired
	private RapidSystemBatchJobTaskMapper systemJobLogBatchMapper;

	private String funcType;

	private String subFuncType;

	private Log log = LogFactory
			.getLog(AbstractRapidSystemBatchJobTaskFire.class);

}
