package com.rapid.j2ee.framework.bean.dictionary;

import java.lang.reflect.Field;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.util.Assert;

import com.rapid.j2ee.framework.bean.dictionary.aop.ComplexBeanFieldTranslation;
import com.rapid.j2ee.framework.bean.dictionary.aop.ComplexBeanTranslation;
import com.rapid.j2ee.framework.core.collections.SortHashMap;
import com.rapid.j2ee.framework.core.memorycache.strategy.CacheStrategy.Strategy;
import com.rapid.j2ee.framework.core.pagination.PageOutput;
import com.rapid.j2ee.framework.core.reflect.BeanUtils;
import com.rapid.j2ee.framework.core.reflect.InvokeUtils;
import com.rapid.j2ee.framework.core.utils.ClassUtils;
import com.rapid.j2ee.framework.core.utils.ObjectUtils;
import com.rapid.j2ee.framework.core.utils.StringUtils;
import com.rapid.j2ee.framework.core.utils.TypeChecker;

public class BeanDictionaryFieldsResolvableTranslator {

	@Autowired(required = false)
	private List<DictionaryTranslationResolver> resolvers = new ArrayList<DictionaryTranslationResolver>();

	private String translatorFieldAlias = "Text";

	private static ThreadLocal<Object> translatedTempResult = new ThreadLocal<Object>();

	public static final DictionaryTranslationResolver Default_DictionaryTranslationResolver = new DictionaryTranslationResolver() {

		public String getResolverName() {

			return "DEFAULT";
		}

		public String translate(Map<String, String> paramters, Object bean,
				String fieldName) {

			return BeanUtils.getProperty(bean, fieldName);
		}

		public Strategy getCacheStrategy() {

			return Strategy.NoCache;
		}

	};

	public DictionaryTranslationResolver findByResovlerName(String name) {

		if (TypeChecker.isEmpty(resolvers)) {
			return Default_DictionaryTranslationResolver;
		}

		for (DictionaryTranslationResolver resolver : resolvers) {
			if (resolver.getResolverName().equalsIgnoreCase(name)) {
				return resolver;
			}
		}

		Assert.isTrue(false,
				"Please provide a DictionaryTranslationResolver by the resovler name  "
						+ name);

		return Default_DictionaryTranslationResolver;

	}

	public <T> Collection<T> resolveCollection(Collection<T> beans) {

		if (TypeChecker.isEmpty(beans)) {
			return beans;
		}

		for (Object bean : beans) {

			resolve(bean);
		}

		return beans;
	}

	public Object resolve(Object bean) {

		if (TypeChecker.isNull(bean)) {
			return bean;
		}

		if (bean instanceof PageOutput) {
			return this.resolvePageOutput((PageOutput) bean);
		}

		if (bean instanceof Collection) {
			return this.resolveCollection((Collection) bean);
		}

		if (bean.getClass().isAnnotationPresent(ComplexBeanTranslation.class)) {
			return resolveComplexBean(bean);
		}

		return this.resolveSimpleObject(bean);

	}

	private Object resolveComplexBean(Object bean) {

		resolveSimpleObject(bean);

		Field[] fs = ClassUtils.getAllFieldsAsClassByAnnotation(
				bean.getClass(), ComplexBeanFieldTranslation.class);

		for (Field f : fs) {

			resolve(InvokeUtils.getFieldValue(bean, f));
		}

		return bean;

	}

	private Object resolvePageOutput(PageOutput pageOutput) {

		resolveCollection(pageOutput.getResult());

		resolveCollection(pageOutput.getAdditionalResult());
		resolveCollection(pageOutput.getAdditionalResult2());
		resolveCollection(pageOutput.getAdditionalResult3());

		resolveCollection(pageOutput.getAdditionalResult4());

		return pageOutput;

	}

	private Object resolveSimpleObject(Object bean) {

		Field[] fields = ClassUtils.getAllFieldsAsClassByAnnotation(bean
				.getClass(), FieldTranslator.class);

		if (TypeChecker.isEmpty(fields)) {
			return bean;
		}

		for (Field f : fields) {

			clearTempValueInThreadLocale();

			doTranslateFieldValue(f, f.getAnnotation(FieldTranslator.class),
					bean);

		}

		return bean;
	}

	private void doTranslateFieldValue(Field f,
			FieldTranslator fieldTranslator, Object bean) {

		if (isFieldValueTranslated(fieldTranslator, bean, f)) {
			return;
		}

		String[] resolverNames = StringUtils.splitBySeparators(fieldTranslator
				.resolverName(), ",", ";");

		List<SortHashMap> parameters = getParametersWithIndexes(fieldTranslator);

		int index = 0;

		for (String resolverName : resolverNames) {

			doTranslateFieldValue(resolverName, f, fieldTranslator, bean,
					getParametersByIndex(parameters, index));

			index++;
		}

	}

	private boolean isFieldValueTranslated(FieldTranslator anno, Object bean,
			Field f) {

		Object valueTranslated = BeanUtils.getPropertyObject(bean,
				getRelatedFieldName(anno, f));

		if (TypeChecker.isNull(valueTranslated)
				|| TypeChecker.isEmpty(String.valueOf(valueTranslated))) {

			return false;
		}

		return true;

	}

	private SortHashMap getParametersByIndex(List<SortHashMap> maps, int index) {

		if (TypeChecker.isEmpty(maps)) {
			return ObjectUtils.EMPTY_SORTHASHMAP;
		}

		if (maps.size() <= index) {
			return ObjectUtils.EMPTY_SORTHASHMAP;
		}

		return maps.get(index);
	}

	private List<SortHashMap> getParametersWithIndexes(
			FieldTranslator fieldTranslator) {

		String parameter = fieldTranslator.parameters();

		if (TypeChecker.isEmpty(parameter)) {
			return ObjectUtils.EMPTY_LIST;
		}

		parameter = StringUtils.replaceAll(parameter, ",", ";");

		String[] parameters = StringUtils.splitClosure(parameter, StringUtils
				.trimToEmpty(fieldTranslator.closureStart(), "{"), StringUtils
				.trimToEmpty(fieldTranslator.closureEnd(), "}"));

		List<SortHashMap> maps = new ArrayList<SortHashMap>(parameters.length);

		for (String param : parameters) {

			maps.add(StringUtils.splitSortHashMapByGroupKey(param, ";", ":"));
		}

		return maps;
	}

	private void doTranslateFieldValue(String resolverName, Field f,
			FieldTranslator anno, Object bean, Map<String, String> parameters) {

		DictionaryTranslationResolver resolver = findByResovlerName(resolverName);

		if (Default_DictionaryTranslationResolver == resolver) {
			return;
		}

		String fieldNameText = getRelatedFieldName(anno, f);

		Object fieldValue = resolver.translate(parameters, bean, f.getName());

		setValueToObjectTempValueThreadLocale(fieldValue);

		InvokeUtils.setField(bean, fieldNameText, fieldValue);

	}

	private String getRelatedFieldName(FieldTranslator anno, Field f) {
		return StringUtils.trimToEmpty(anno.relatedFieldName(), f.getName()
				+ translatorFieldAlias);
	}

	public static Object getObjectValueWithTempValueThreadLocaleChecking(
			Object fieldValue) {

		if (!TypeChecker.isNull(translatedTempResult.get())) {
			return translatedTempResult.get();
		}

		return fieldValue;
	}

	public static void clearTempValueInThreadLocale() {
		translatedTempResult.remove();
	}

	private static void setValueToObjectTempValueThreadLocale(Object value) {
		translatedTempResult.set(value);
	}

	public void setTranslatorFieldAlias(String translatorFieldAlias) {
		this.translatorFieldAlias = translatorFieldAlias;
	}
}
