package com.rapid.j2ee.framework.core.charset;

import java.io.UnsupportedEncodingException;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Properties;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.core.io.ClassPathResource;

import com.rapid.j2ee.framework.core.exception.SystemException;
import com.rapid.j2ee.framework.core.utils.ObjectUtils;
import com.rapid.j2ee.framework.core.utils.StringUtils;
import com.rapid.j2ee.framework.core.utils.TypeChecker;

public class Charsets {

	private Map<String, Charset> charsetsContainer = new HashMap<String, Charset>();

	private static Log Logger = LogFactory.getLog(Charsets.class);

	public final static Charset UTF_8 = new CharsetImp(CharsetType.UTF_8, "UTF-8");

	public final static Charset Chinese = new CharsetImp(CharsetType.CHINESE, "GBK");

	public final static Charset English = new CharsetImp(CharsetType.ENGLISH,
			"ISO8859-1");

	private static final Charsets INSTANCE = new Charsets();

	private Charsets() {

		charsetsContainer.put(CharsetType.UTF_8, UTF_8);

		charsetsContainer.put(CharsetType.CHINESE, Chinese);

		charsetsContainer.put(CharsetType.ENGLISH, English);

		prepareCharsets(loadConfigure());

	}

	public static Charset getCharsetInstance(String key) {
		return INSTANCE.getCharset(key);
	}

	public static CharsetEncode getCharsetEncodeInstance(String key) {
		return (CharsetEncode) getCharsetInstance(key);
	}

	private Properties loadConfigure() {

		Properties charsets = new Properties();

		try {

			charsets.load(new ClassPathResource("charset.properties")
					.getInputStream());

		} catch (Exception e) {
			throw new SystemException(
					"The charset resouce [charsetSupport.properties] cannot be loaded in the current class loader.",
					e);
		}

		try {

			Properties addProps = new Properties();

			addProps.load(new ClassPathResource("charsetSupport.properties")
					.getInputStream());

			charsets.putAll(addProps);

		} catch (Exception e) {

		}

		return charsets;

	}

	public Charset getCharset(String symbolKey) {

		symbolKey = StringUtils.trimToEmpty(symbolKey).toLowerCase();

		if (!charsetsContainer.containsKey(symbolKey)) {
			throw new IllegalArgumentException("The symbol key [" + symbolKey
					+ "] is invalid.");
		}

		return charsetsContainer.get(symbolKey);
	}

	public CharsetEncode getCharsetConverter(String symbolKey) {
		return (CharsetEncode) getCharset(symbolKey);
	}

	private void prepareCharsets(Properties charsets) {

		for (Iterator it = charsets.keySet().iterator(); it.hasNext();) {
			String key = StringUtils.trimToEmpty((String) it.next())
					.toLowerCase();

			if (charsetsContainer.containsKey(key)) {
				continue;
			}

			Charset charset = new CharsetImp(key, charsets.getProperty(key));

			charsetsContainer.put(key, charset);

		}
		Logger.info("\nCharsets:");
		Logger.info(charsetsContainer);

	}

	public static final String encode(String target, String from, String to) {

		if (TypeChecker.isEmpty(target) || from.equalsIgnoreCase(to)) {
			return target;
		}

		try {
			return new String(target.getBytes(from), to);
		} catch (UnsupportedEncodingException e) {
			return target;
		}

	}

	private static class CharsetImp implements Charset, CharsetEncode {

		private String key;

		private String charset;

		public CharsetImp(String key, String charset) {
			this.key = key;
			this.charset = charset;
		}

		public String getCharset() {

			return charset;
		}

		public String getSymbolKey() {

			return key;
		}

		@Override
		public String toString() {
			return "Charset:" + this.charset;
		}

		@Override
		public int hashCode() {
			return 13 + this.key.hashCode() * 7 + this.charset.hashCode() * 19;
		}

		@Override
		public boolean equals(Object item) {

			if (this == item) {
				return true;
			}

			Charset charset = (Charset) item;

			return StringUtils.equalsIgnoreCase(charset.getCharset(),
					this.charset);

		}

		public String toIso8859(String target) {

			return encode(target, this.charset, Charsets.getCharsetInstance(
					CharsetType.ENGLISH).getCharset());
		}

		public String toNative(String target) {

			return encode(target, Charsets.getCharsetInstance(
					CharsetType.ENGLISH).getCharset(), this.charset);
		}

	}

}
