package com.rapid.j2ee.framework.core.io.barcode;

import java.awt.AlphaComposite;
import java.awt.Color;
import java.awt.Graphics2D;
import java.awt.Image;
import java.awt.geom.AffineTransform;
import java.awt.image.AffineTransformOp;
import java.awt.image.BufferedImage;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Hashtable;
import javax.imageio.ImageIO;

import org.springframework.util.Assert;

import com.google.zxing.BarcodeFormat;
import com.google.zxing.BinaryBitmap;
import com.google.zxing.DecodeHintType;
import com.google.zxing.EncodeHintType;
import com.google.zxing.LuminanceSource;
import com.google.zxing.MultiFormatReader;
import com.google.zxing.MultiFormatWriter;
import com.google.zxing.ReaderException;
import com.google.zxing.Result;
import com.google.zxing.WriterException;
import com.google.zxing.client.j2se.BufferedImageLuminanceSource;
import com.google.zxing.client.j2se.MatrixToImageWriter;
import com.google.zxing.common.BitMatrix;
import com.google.zxing.common.HybridBinarizer;
import com.google.zxing.qrcode.decoder.ErrorCorrectionLevel;
import com.rapid.j2ee.framework.core.charset.Charset;
import com.rapid.j2ee.framework.core.charset.CharsetType;
import com.rapid.j2ee.framework.core.charset.Charsets;
import com.rapid.j2ee.framework.core.exception.ExceptionUtils;
import com.rapid.j2ee.framework.core.utils.DateTimeUtils;
import com.rapid.j2ee.framework.core.utils.FileUtils;
import com.rapid.j2ee.framework.core.utils.ImageCompressUtils;
import com.rapid.j2ee.framework.core.utils.TypeChecker;
import com.rapid.j2ee.framework.core.utils.ImageCompressUtils.DimensionScale;
import com.rapid.j2ee.framework.core.utils.support.DateTimeFormat;

/**
 * 二维码Encode & Decode
 * 
 * @author John Hao
 * 
 */
public class ImageQRCodeUtil {

	/**
	 * 编码（将文本生成二维码）
	 * 
	 * @param content
	 *            二维码中的内容
	 * @param width
	 *            二维码图片宽度
	 * @param height
	 *            二维码图片高度
	 * @param OutputStream
	 *            二维码图片存放os
	 */
	public static void encode(String content, int width, int height,
			OutputStream os) {
		encode(content, width, height, os, "png", Charsets.UTF_8);
	}

	/**
	 * 编码（将文本生成二维码）
	 * 
	 * @param content
	 *            二维码中的内容
	 * @param width
	 *            二维码图片宽度
	 * @param height
	 *            二维码图片高度
	 * @param OutputStream
	 *            二维码图片存放os
	 * @param imageType --
	 *            二维码图片类型 png， gif jpg etc
	 */
	public static void encode(String content, int width, int height,
			OutputStream os, String imageType, Charset charset) {

		Assert.notNull(os);

		Hashtable<EncodeHintType, Object> hints = new Hashtable<EncodeHintType, Object>();

		// 设置编码类型为utf-8
		hints.put(EncodeHintType.CHARACTER_SET, charset.getCharset());

		try {
			// 生成二维码
			BitMatrix byteMatrix = new MultiFormatWriter().encode(content,
					BarcodeFormat.QR_CODE, width, height, hints);

			MatrixToImageWriter.writeToStream(
					getBitMatrixWithoutWhiteBorder(byteMatrix), imageType, os);

		} catch (Exception e) {
			throw ExceptionUtils.convertThrowableToBaseException(e);
		} finally {
			FileUtils.close(os);
		}

	}

	private static BitMatrix getBitMatrixWithoutWhiteBorder(BitMatrix byteMatrix) {

		int[] rec = byteMatrix.getEnclosingRectangle();

		int resWidth = rec[2] + 1;
		int resHeight = rec[3] + 1;
		BitMatrix resMatrix = new BitMatrix(resWidth, resHeight);
		resMatrix.clear();
		for (int i = 0; i < resWidth; i++) {
			for (int j = 0; j < resHeight; j++) {
				if (byteMatrix.get(i + rec[0], j + rec[1])) {
					resMatrix.set(i, j);
				}
			}
		}

		return resMatrix;
	}

	/**
	 * 编码（将文本生成二维码）
	 * 
	 * @param content
	 *            二维码中的内容
	 * @param width
	 *            二维码图片宽度
	 * @param height
	 *            二维码图片高度
	 * @param imagePath
	 *            二维码图片存放位置
	 * @return 图片地址
	 */
	public static String encode(String content, int width, int height,
			File qrcodeImageFile, Charset charset) {

		Assert.notNull(qrcodeImageFile);

		if (!qrcodeImageFile.exists()) {
			FileUtils.makeDir(qrcodeImageFile);
		}

		Hashtable<EncodeHintType, Object> hints = new Hashtable<EncodeHintType, Object>();

		// 设置编码类型为utf-8
		hints.put(EncodeHintType.CHARACTER_SET, charset.getCharset());

		try {
			// 生成二维码
			BitMatrix byteMatrix = new MultiFormatWriter().encode(content,
					BarcodeFormat.QR_CODE, width, height, hints);

			MatrixToImageWriter
					.writeToFile(getBitMatrixWithoutWhiteBorder(byteMatrix),
							FileUtils.getFileNameAndExtension(qrcodeImageFile
									.getName())[1], qrcodeImageFile);

			return qrcodeImageFile.getAbsolutePath();

		} catch (Exception e) {
			throw ExceptionUtils.convertThrowableToBaseException(e);
		}

	}

	public static String encode(String content, int width, int height,
			File qrcodeImageFile) {
		return encode(content, width, height, qrcodeImageFile, Charsets.UTF_8);
	}

	public static String encode(String content, int width, int height,
			String qrcodeImagePath) {
		return encode(content, width, height, new File(qrcodeImagePath));
	}

	/**
	 * 解码（读取二维码图片中的文本信息）
	 * 
	 * @param imagePath
	 *            二维码图片路径
	 * @return 文本信息
	 */
	public static String decode(InputStream is, Charset charset) {

		try {

			if (TypeChecker.isNull(is)) {
				return "";
			}

			BufferedImage image = null;
			image = ImageIO.read(is);
			if (null == image) {
				return "";
			}
			// 解码
			LuminanceSource source = new BufferedImageLuminanceSource(image);
			BinaryBitmap bitmap = new BinaryBitmap(new HybridBinarizer(source));
			Hashtable hints = new Hashtable();
			hints.put(DecodeHintType.CHARACTER_SET, charset.getCharset());

			Result rs = new MultiFormatReader().decode(bitmap, hints);
			return rs.getText();
		} catch (Exception e) {

			throw ExceptionUtils.convertThrowableToBaseException(e);
		} finally {
			FileUtils.close(is);
		}

	}

	public static String decode(File rqcodeImageFile, Charset charset) {
		try {
			return decode(new FileInputStream(rqcodeImageFile), charset);
		} catch (FileNotFoundException e) {
			throw ExceptionUtils.convertThrowableToBaseException(e);
		}
	}

	public static String decode(String rqcodeImagePath, Charset charset) {
		try {
			return decode(new FileInputStream(rqcodeImagePath), charset);
		} catch (FileNotFoundException e) {
			throw ExceptionUtils.convertThrowableToBaseException(e);
		}
	}

	/**
	 * 图片打水印
	 * 
	 * @param bgImage
	 *            背景图
	 * @param waterImg
	 *            水印图
	 * 
	 * @return 新图片路径
	 */
	public static File addWaterImage(File cqCodeImageFile, File bgImageFile) {

		try {

			if (!bgImageFile.exists()) {
				FileUtils.makeDir(bgImageFile);
			}

			DimensionScale dimensionScale = ImageCompressUtils
					.getImageDimension(cqCodeImageFile);

			// 缩放水印图片,为保证二维码的读取正确，图片不超过二维码图片的五分之一，这里设为六分之一
			bgImageFile = resize(bgImageFile, dimensionScale.width / 6,
					dimensionScale.height / 6, true);

			BufferedImage bufferedImage = new BufferedImage(
					dimensionScale.width, dimensionScale.height,
					BufferedImage.TYPE_INT_RGB);

			Graphics2D g = bufferedImage.createGraphics();

			g.drawImage(ImageIO.read(cqCodeImageFile), 0, 0,
					dimensionScale.width, dimensionScale.height, null);

			Image waterImage = ImageIO.read(bgImageFile); // 水印文件

			int width_water = waterImage.getWidth(null);
			int height_water = waterImage.getHeight(null);
			g.setComposite(AlphaComposite.getInstance(AlphaComposite.SRC_ATOP,
					1));

			int widthDiff = dimensionScale.width - width_water;
			int heightDiff = dimensionScale.height - height_water;

			int x = widthDiff / 2;
			int y = heightDiff / 2;

			// 水印文件结束
			g.drawImage(waterImage, x, y, width_water, height_water, null);
			g.dispose();

			File rqCodeNewImg = ImageCompressUtils.getFileByAlias(
					cqCodeImageFile, "_water");

			ImageIO.write(bufferedImage, FileUtils
					.getFileNameAndExtension(cqCodeImageFile.getName())[1],
					rqCodeNewImg);

			FileUtils.delete(bgImageFile);

			return rqCodeNewImg;

		} catch (Exception e) {
			throw ExceptionUtils.convertThrowableToBaseException(e);
		}

	}

	/**
	 * 图片缩放
	 * 
	 * @param filePath
	 *            图片路径
	 * @param height
	 *            缩放到高度
	 * @param width
	 *            缩放宽度
	 * @param fill
	 *            比例足时是否填白 true为填白，二维码是黑白色，这里调用时建议设为true
	 * 
	 * @return 新图片路径
	 */
	private static File resize(File f, int width, int height, boolean fill) {

		try {

			BufferedImage bi = ImageIO.read(f);
			Image itemp = bi.getScaledInstance(width, height,
					BufferedImage.SCALE_SMOOTH);

			if (height != 0 && width != 0) {
				double ratio = 1;
				// 计算比例
				if ((bi.getHeight() > height) || (bi.getWidth() > width)) {
					if (bi.getHeight() > bi.getWidth()) {
						ratio = (new Integer(height)).doubleValue()
								/ bi.getHeight();
					} else {
						ratio = (new Integer(width)).doubleValue()
								/ bi.getWidth();
					}
					AffineTransformOp op = new AffineTransformOp(
							AffineTransform.getScaleInstance(ratio, ratio),
							null);
					itemp = op.filter(bi, null);
				}
			}

			if (fill) {
				BufferedImage image = new BufferedImage(width, height,
						BufferedImage.TYPE_INT_RGB);
				Graphics2D g = image.createGraphics();
				g.setColor(Color.white);
				g.fillRect(0, 0, width, height);
				if (width == itemp.getWidth(null)) {
					g.drawImage(itemp, 0, (height - itemp.getHeight(null)) / 2,
							itemp.getWidth(null), itemp.getHeight(null),
							Color.white, null);
				} else {
					g.drawImage(itemp, (width - itemp.getWidth(null)) / 2, 0,
							itemp.getWidth(null), itemp.getHeight(null),
							Color.white, null);
				}
				g.dispose();
				itemp = image;
			}

			File newImg = new File(f.getParentFile(), System
					.currentTimeMillis()
					+ "_" + f.getName());

			ImageIO.write((BufferedImage) itemp, FileUtils
					.getFileNameAndExtension(newImg.getName())[1], newImg);

			return newImg;

		} catch (Exception e) {
			throw ExceptionUtils.convertThrowableToBaseException(e);
		}

	}

	/**
	 * 图片添加边框
	 * 
	 * @param mainImgPath
	 *            要加边框的图片
	 * @param bgImgPath
	 *            背景图（实际上是将图片放在背景图上，只利用背景图的边框效果）
	 * @return 制作完成的图片路径
	 */
	private static File createImageBorder(File source, int border) {

		try {

			DimensionScale dimensionScale = ImageCompressUtils
					.getImageDimension(source);

			// 背景图长宽都比主图多4像素，这是因为我画的背景图的边框效果的大小正好是4像素，
			// 主图周边比背景图少4像素正好能把背景图的边框效果完美显示出来

			return resize(source, dimensionScale.width + border,
					dimensionScale.height + border, true);

		} catch (Exception e) {
			throw ExceptionUtils.convertThrowableToBaseException(e);
		}

	}

	public static void main(String[] args) {

		ImageQRCodeUtil.encode("1ewewew", 150, 150, new File("test.png"));

		// String qrImgPath = encode(
		// "[Spring学习笔记 3 ] spring 注解详解，完全注解，[Spring学习笔记 3 ] spring
		// 注解详解，完全注解[Spring学习笔记 3 ] spring 注解详解，完全注解[Spring学习笔记 3 ] spring
		// 注解详解，完全注解[Spring学习笔记 3 ] spring 注解详解，完全注解常用注解
		// http://www.sina.com.cn",
		// 300, 300, new File("test.png"));
		//
		// System.out.println(decode(new File("test.png")));
		//
		// createImageBorder(new File("Penguins.jpg"), 4);
		//
		// System.out.println(addWaterImage(new File(qrImgPath), new File(
		// "Penguins.jpg")));

	}
}
