package com.rapid.j2ee.framework.core.io.file;

import java.io.BufferedWriter;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.OutputStream;
import java.io.OutputStreamWriter;

import com.rapid.j2ee.framework.core.charset.Charset;
import com.rapid.j2ee.framework.core.exception.ExceptionUtils;
import com.rapid.j2ee.framework.core.utils.FileUtils;
import com.rapid.j2ee.framework.core.utils.StringUtils;

public class FileWriter {

	private File file;

	private OutputStreamWriter osw;

	private BufferedWriter bw;

	private static final FileWriterFilter DEFAULT_FILTER = new FileWriterFilter() {

		public boolean accept(String content) {

			return true;
		}

		public String convert(String content) {

			return content;
		}

	};

	private FileWriterFilter filter = DEFAULT_FILTER;

	private static int DEFAULT_CHARSET_BUFFER_SIZE = 1024 * 2;

	public FileWriter(File file, Charset charset) {
		this(file, charset, false);
	}

	public FileWriter(File file, String charset) {
		this(file, charset, false);
	}

	public FileWriter(File file, Charset charset, boolean append) {
		try {
			this.file = file;
			FileUtils.makeDir(file.getParentFile());
			init(new FileOutputStream(file, append), charset.getCharset());
		} catch (FileNotFoundException ex) {
			this.close();
			throw ExceptionUtils.convertThrowableToBaseException(ex);
		}
	}

	public FileWriter(File file, String charset, boolean append) {
		try {
			this.file = file;
			FileUtils.makeDir(file.getParentFile());
			init(new FileOutputStream(file, append), charset);
		} catch (FileNotFoundException ex) {
			this.close();
			throw ExceptionUtils.convertThrowableToBaseException(ex);
		}
	}

	private void init(OutputStream os, String charset) {
		try {
			osw = new OutputStreamWriter(os, charset);
			bw = new BufferedWriter(osw, DEFAULT_CHARSET_BUFFER_SIZE);
		} catch (Exception ex) {
			this.close();
			throw ExceptionUtils.convertThrowableToBaseException(ex);
		}

	}

	public void write(String value) {
		try {
			if (!filter.accept(value)) {
				return;
			}
			bw.write(filter.convert(value));
		} catch (Exception ex) {
			this.close();
			throw ExceptionUtils.convertThrowableToBaseException(ex);
		}
	}

	public void writeln(String value) {
		if (!filter.accept(value)) {
			return;
		}
		write(filter.convert(value));
		write("\r\n");
	}

	public void flush() {
		try {
			bw.flush();
		} catch (Exception ex) {
		}
		try {
			osw.flush();
		} catch (Exception ex) {
		}

	}

	public void newLine() {
		this.writeln("");
	}

	public void writeTrim(String value) {
		if (!filter.accept(value)) {
			return;
		}
		this.write(filter.convert(StringUtils.trimToEmpty(value)));
	}

	public void writeTrimln(String value) {
		if (!filter.accept(value)) {
			return;
		}
		this.writeln(filter.convert(StringUtils.trimToEmpty(value)));
	}

	public void addFilter(FileWriterFilter filter) {
		this.filter = filter;
	}

	public void close() {

		flush();

		try {
			osw.close();
		} catch (Exception ex2) {
		}
		try {
			bw.close();
		} catch (Exception ex3) {
		}

	}

	public File getFile() {
		return file;
	}

}