package com.rapid.j2ee.framework.core.io.net.disk;

import java.io.File;
import java.io.FileOutputStream;
import java.io.InputStream;
import java.io.OutputStream;

import javax.servlet.http.HttpServletResponse;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.util.Assert;

import com.rapid.j2ee.framework.core.exception.ExceptionUtils;
import com.rapid.j2ee.framework.core.exception.SystemException;
import com.rapid.j2ee.framework.core.io.ResponseOutputStream;
import com.rapid.j2ee.framework.core.io.net.ResourceStorageClient;
import com.rapid.j2ee.framework.core.io.net.disk.category.ResourceCategoryDirectoryRegistry;
import com.rapid.j2ee.framework.core.io.net.disk.interceptor.ResourceDiskStoreageGetInterceptor;
import com.rapid.j2ee.framework.core.io.net.disk.interceptor.ResourceDiskStoreagePutInterceptor;
import com.rapid.j2ee.framework.core.utils.FileCopyUtils;
import com.rapid.j2ee.framework.core.utils.FileUtils;
import com.rapid.j2ee.framework.core.utils.NumberUtils;
import com.rapid.j2ee.framework.core.utils.StringUtils;
import com.rapid.j2ee.framework.core.utils.TypeChecker;
import com.rapid.j2ee.framework.mvc.utils.ActionContextUtils;

public class ResourceDiskStorageClient implements ResourceStorageClient {

	public String getResourceAsNotFound(String category) {
		return resourceCategoryDirectoryRegistry
				.getResourceAsNotFound(category);
	}

	public boolean get(String category, String supplyFolder, String fileName,
			OutputStream os) {

		if (TypeChecker.isEmpty(fileName)) {
			fileName = getResourceAsNotFound(category);
		}

		Assert.hasLength(fileName, "Please provide a file name!");

		File targetFileName = new File(this.getMakeDirectory(category,
				supplyFolder), fileName);

		if (!targetFileName.exists()) {

			logger.info("Parent Path:"
					+ targetFileName.getParent()
					+ " Lost Default Name:"
					+ resourceCategoryDirectoryRegistry
							.getResourceAsNotFound(category));

			targetFileName = FileUtils.findFileWithParents(targetFileName
					.getParent(), resourceCategoryDirectoryRegistry
					.getResourceAsNotFound(category));

			logger.info("Default File Name :" + targetFileName + " Under "
					+ category);
		}

		if (TypeChecker.isNull(targetFileName) || !targetFileName.exists()
				|| targetFileName.isDirectory()) {
			return false;
		}

		if (!resourceDiskStoreageGetInterceptor.before(category, supplyFolder,
				fileName, targetFileName, os)) {

			return false;
		}

		try {

			long beginPoint = this.getResumeTransferFileRangeStartPoint(os);

			this.setResponseInformation(beginPoint, os, category,
					targetFileName);

			FileCopyUtils.copy(resourceDiskStoreageGetInterceptor
					.getRandomAccessFile(category, targetFileName), beginPoint,
					os);

		} catch (Exception e) {
			throw ExceptionUtils.convertThrowableToBaseException(e);
		}

		return true;

	}

	private void setResponseInformation(long beginPoint, OutputStream os,
			String category, File file) {

		if (!(os instanceof ResponseOutputStream)) {
			return;
		}

		HttpServletResponse response = ActionContextUtils.getHttpResponse();

		long fileSize = resourceDiskStoreageGetInterceptor.getSize(category,
				file);

		response.setHeader("Content-Length", String.valueOf(fileSize
				- beginPoint));

		logger.info("Request Content-Length : " + (fileSize - beginPoint));

		logger.info("Request Range Start Point : " + beginPoint);

		if (beginPoint == 0) {
			response.setStatus(200);
			return;
		}

		response.setStatus(206);

		response.setHeader("Content-Range", "bytes " + beginPoint + "-"
				+ (fileSize - 1) + "/" + fileSize);

	}

	private long getResumeTransferFileRangeStartPoint(OutputStream os) {

		if (!(os instanceof ResponseOutputStream)) {
			return 0;
		}

		ResponseOutputStream responseOutputStream = (ResponseOutputStream) os;

		String range = ActionContextUtils.getRequestHeader(responseOutputStream
				.getRequest(), "RANGE");

		logger.info("Request Header Range == " + range);

		if (TypeChecker.isEmpty(range)) {
			return 0;
		}

		range = StringUtils.substringAfter(range.toLowerCase(), "bytes=");

		return NumberUtils.parseLong(StringUtils.substringBefore(range, "-"));
	}

	public boolean get(String category, String fileName, OutputStream os) {
		return this.get(category, null, fileName, os);
	}

	public boolean put(String category, String targetFileName, File sourceFile) {
		return this.put(category, null, targetFileName, sourceFile);
	}

	public boolean put(String category, String targetFileName,
			InputStream inputStream) {
		return this.put(category, null, targetFileName, inputStream);
	}

	public boolean put(String category, String supplyFolder,
			String targetFileName, InputStream inputStream) {
		OutputStream os = null;

		try {

			File destinationFile = new File(this.getMakeDirectory(category,
					supplyFolder), targetFileName);

			FileUtils
					.makeDir(destinationFile.getParentFile().getAbsolutePath());

			os = new FileOutputStream(destinationFile);

			FileUtils.copy(inputStream, os, true);

			return true;

		} catch (Exception e) {
			throw new SystemException("Upload File Path:"
					+ this.getMakeDirectory(category, supplyFolder) + " Name:"
					+ targetFileName + " Failed.", e);
		} finally {
			this.close(os);

		}
	}

	public boolean put(String category, String supplyFolder,
			String targetFileName, File sourceFile) {

		OutputStream os = null;

		try {

			File destinationFile = new File(this.getMakeDirectory(category,
					supplyFolder), targetFileName);

			if (!resourceDiskStoreagePutInterceptor.before(category,
					destinationFile, sourceFile)) {
				return false;
			}

			FileUtils
					.makeDir(destinationFile.getParentFile().getAbsolutePath());

			os = new FileOutputStream(destinationFile);

			FileUtils.copy(resourceDiskStoreagePutInterceptor
					.getFileInputStream(category, sourceFile), os, true);

			return true;

		} catch (Exception e) {
			throw new SystemException("Upload File Path:"
					+ this.getMakeDirectory(category, supplyFolder) + " Name:"
					+ targetFileName + " Failed.", e);
		} finally {
			this.close(os);

		}

	}

	public boolean delete(String category, String supplyFolder, String fileName) {

		File file = new File(this.getMakeDirectory(category, supplyFolder),
				fileName);

		return file.delete();
	}

	public boolean delete(String category, String fileName) {
		return this.delete(category, null, fileName);
	}

	public String getDirectory(String category) {
		return this.getDirectory(category, null);
	}

	public String getDirectory(String category, String supplyFolder) {

		Assert.notNull(resourceCategoryDirectoryRegistry,
				"Please provide a ResourceCategoryDirectoryRegistry!");

		String path = resourceCategoryDirectoryRegistry.getDirectory(category);

		if (TypeChecker.isEmpty(path)) {
			Assert.notNull(path,
					"Please provide a resource disk path by category  "
							+ category);
		}

		if (!TypeChecker.isEmpty(supplyFolder)) {
			path = FileUtils.getFullFilePathName(path, supplyFolder);
		}

		return path;

	}

	public String getMakeDirectory(String category, String supplyFolder) {

		String path = getDirectory(category, supplyFolder);

		FileUtils.makeDir(path);

		return path;

	}

	public final void close(InputStream is) {
		try {
			is.close();
		} catch (Exception e) {

		}
	}

	public final void close(OutputStream os) {
		try {
			os.flush();
		} catch (Exception e) {

		}
		try {
			os.close();
		} catch (Exception e) {

		}
	}

	public void setResourceCategoryDirectoryRegistry(
			ResourceCategoryDirectoryRegistry resourceCategoryDirectoryRegistry) {
		this.resourceCategoryDirectoryRegistry = resourceCategoryDirectoryRegistry;
	}

	public void setResourceDiskStoreageGetInterceptor(
			ResourceDiskStoreageGetInterceptor resourceDiskStoreageGetInterceptor) {
		this.resourceDiskStoreageGetInterceptor = resourceDiskStoreageGetInterceptor;
	}

	public void setResourceDiskStoreagePutInterceptor(
			ResourceDiskStoreagePutInterceptor resourceDiskStoreagePutInterceptor) {
		this.resourceDiskStoreagePutInterceptor = resourceDiskStoreagePutInterceptor;
	}

	private ResourceCategoryDirectoryRegistry resourceCategoryDirectoryRegistry;

	private ResourceDiskStoreageGetInterceptor resourceDiskStoreageGetInterceptor = ResourceDiskStoreageGetInterceptor.Get_Default_Interceptor;

	private ResourceDiskStoreagePutInterceptor resourceDiskStoreagePutInterceptor = ResourceDiskStoreagePutInterceptor.Put_Default_Interceptor;

	private static final Log logger = LogFactory
			.getLog(ResourceDiskStorageClient.class);

}
