package com.rapid.j2ee.framework.core.io.net.ftp;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.SocketException;
import java.util.Arrays;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import org.apache.commons.net.ftp.FTPClient;
import org.apache.commons.net.ftp.FTPReply;

import com.rapid.j2ee.framework.core.utils.FileUtils;
import com.rapid.j2ee.framework.core.utils.ObjectAnalyzer;
import com.rapid.j2ee.framework.core.utils.StringUtils;
import com.rapid.j2ee.framework.core.utils.TypeChecker;

/**
 * FTP客户端
 * 
 */
public class FTPClientTemplate {

	private static final Log Logger = LogFactory
			.getLog(FTPClientTemplate.class);

	private static ThreadLocal<FTPClient> ftpClientThreadLocal = new ThreadLocal<FTPClient>();

	private FTPClientTemplateConfigure ftpClientTemplateConfigure;

	private static ThreadLocal<Boolean> autoCommitThreadLocal = new ThreadLocal<Boolean>() {
		@Override
		protected Boolean initialValue() {

			return true;
		}
	};

	public FTPClientTemplate(
			FTPClientTemplateConfigure ftpClientTemplateConfigure) {
		this.ftpClientTemplateConfigure = ftpClientTemplateConfigure;
	}

	public static void setAutoCommit(boolean commit) {
		autoCommitThreadLocal.set(commit);
	}

	public static boolean isAutoCommit() {
		return TypeChecker.isNull(autoCommitThreadLocal.get())
				|| autoCommitThreadLocal.get();
	}

	// ---------------------------------------------------------------------
	// private method
	// ---------------------------------------------------------------------
	/**
	 * 返回一个FTPClient实例
	 * 
	 * @throws FTPClientException
	 */
	private FTPClient getFTPClient() throws FTPClientException {

		if (!TypeChecker.isNull(ftpClientThreadLocal.get())
				&& ftpClientThreadLocal.get().isConnected()) {
			return ftpClientThreadLocal.get();
		}

		long time = System.currentTimeMillis();

		FTPClient ftpClient = connect(); // 连接到ftp服务器

		// 设置为passive模式
		if (ftpClientTemplateConfigure.isPassiveMode()) {
			ftpClient.enterLocalPassiveMode();
		}

		setFileType(ftpClient); // 设置文件传输类型

		try {
			ftpClient.setSoTimeout(ftpClientTemplateConfigure
					.getClientTimeout());

		} catch (SocketException e) {
			throw new FTPClientException("Set timeout error.", e);
		}

		ftpClient
				.setBufferSize(this.ftpClientTemplateConfigure.getBufferSize());

		ftpClientThreadLocal.set(ftpClient);

		Logger.info("Open SFTP Session [" + (System.currentTimeMillis() - time)
				/ 1000 + " Sec]！ IP:" + ftpClientTemplateConfigure.getHost()
				+ " Port:" + ftpClientTemplateConfigure.getPort()
				+ " User Name:" + ftpClientTemplateConfigure.getUserName());

		return ftpClient;

	}

	/**
	 * 设置文件传输类型
	 * 
	 * @throws FTPClientException
	 * @throws IOException
	 */
	private void setFileType(FTPClient ftpClient) throws FTPClientException {
		try {
			if (ftpClientTemplateConfigure.isBinaryTransfer()) {
				ftpClient.setFileType(FTPClient.BINARY_FILE_TYPE);
			} else {
				ftpClient.setFileType(FTPClient.ASCII_FILE_TYPE);
			}
		} catch (Exception e) {
			throw new FTPClientException("Could not to set file type.", e);
		}
	}

	/**
	 * 连接到ftp服务器
	 * 
	 * @param ftpClient
	 * @return 连接成功返回true，否则返回false
	 * @throws FTPClientException
	 */
	private FTPClient connect() throws FTPClientException {

		FTPClient ftpClient = new FTPClient(); // 构造一个FtpClient实例

		ftpClient.setControlEncoding(ftpClientTemplateConfigure.getEncoding()); // 设置字符集

		try {

			ftpClient.connect(ftpClientTemplateConfigure.getHost(),
					ftpClientTemplateConfigure.getPort());

			// 连接后检测返回码来校验连接是否成功
			int reply = ftpClient.getReplyCode();

			if (FTPReply.isPositiveCompletion(reply)) {

				// 登陆到ftp服务器
				if (ftpClient.login(ftpClientTemplateConfigure.getUserName(),
						ftpClientTemplateConfigure.getPassword())) {

					setFileType(ftpClient);

					return ftpClient;
				}

			} else {

				ftpClient.disconnect();

				throw new FTPClientException("FTP server refused connection.");
			}

		} catch (Exception e) {

			this.shutdown();

			throw new FTPClientException("Could not connect to server.", e);
		}

		return ftpClient;
	}

	public void shutdown() {

		_shutdown(true);
	}

	public static void shutdownThreadLocal() {

		try {
			FTPClient ftpClient = ftpClientThreadLocal.get();
			ftpClient.logout();

			if (ftpClient.isConnected()) {
				ftpClient.disconnect();
				ftpClient = null;
			}

		} catch (Exception e) {

		} finally {
			ftpClientThreadLocal.remove();
			autoCommitThreadLocal.remove();

		}

		Logger.info("FTP Session Closed!");
	}

	// ---------------------------------------------------------------------
	// public method
	// ---------------------------------------------------------------------
	/**
	 * 断开ftp连接
	 * 
	 * @throws FTPClientException
	 */
	private void _shutdown(boolean mustoff) throws FTPClientException {

		if (mustoff) {
			FTPClientTemplate.setAutoCommit(true);
		}

		if (!isAutoCommit()) {
			return;
		}

		shutdownThreadLocal();

		Logger.info("FTP Session Closed!");
	}

	public boolean mkdir(String pathname) throws FTPClientException {
		return mkdir(pathname, null);
	}

	public boolean mkdirs(String path) throws FTPClientException {

		boolean flag = isAutoCommit();
		try {
			if (path.contains(".")) {
				path = StringUtils.substringBeforeLast(path, "/");
			}

			String[] folders = StringUtils.splitBySeparator(path, "/");

			if (TypeChecker.isEmpty(folders)) {
				return false;
			}

			String currentFolder = null;

			setAutoCommit(false);

			for (String temp : folders) {

				mkdir(temp, currentFolder);

				currentFolder = temp;
			}

			return true;
		} finally {

			setAutoCommit(flag);

			this._shutdown(false);
		}
	}

	/**
	 * 在ftp服务器端创建目录（不支持一次创建多级目录）
	 * 
	 * @param pathname
	 * @param autoClose
	 *            是否自动关闭当前连接
	 * @return
	 * @throws FTPClientException
	 */
	public boolean mkdir(String pathname, String workingDirectory)
			throws FTPClientException {
		try {
			getFTPClient().changeWorkingDirectory(workingDirectory);
			return getFTPClient().makeDirectory(pathname);
		} catch (Exception e) {
			this.shutdown();
			throw new FTPClientException("Could not mkdir.", e);
		} finally {
			this._shutdown(false);
		}
	}

	/**
	 * 上传一个本地文件到远程指定文件
	 * 
	 * @param remoteAbsoluteFile
	 *            远程文件名(包括完整路径)
	 * @param localAbsoluteFile
	 *            本地文件名(包括完整路径)
	 * @param autoClose
	 *            是否自动关闭当前连接
	 * @return 成功时，返回true，失败返回false
	 * @throws FTPClientException
	 */
	public boolean put(String remoteAbsoluteFile, String localAbsoluteFile)
			throws FTPClientException {

		InputStream input = null;
		try {

			// 处理传输
			input = new FileInputStream(localAbsoluteFile);

			return getFTPClient().storeFile(remoteAbsoluteFile, input);

		} catch (FileNotFoundException e) {
			this.shutdown();
			throw new FTPClientException("local file not found.", e);
		} catch (Exception e) {
			this.shutdown();
			throw new FTPClientException("Could not put file to server.", e);
		} finally {

			FileUtils.close(input);

			this._shutdown(false);
		}
	}

	/**
	 * 下载一个远程文件到本地的指定文件
	 * 
	 * @param remoteAbsoluteFile
	 *            远程文件名(包括完整路径)
	 * @param localAbsoluteFile
	 *            本地文件名(包括完整路径)
	 * @param autoClose
	 *            是否自动关闭当前连接
	 * 
	 * @return 成功时，返回true，失败返回false
	 * @throws FTPClientException
	 */
	public boolean get(String remoteAbsoluteFile, String localAbsoluteFile)
			throws FTPClientException {
		OutputStream output = null;
		try {
			File saveFile = new File(localAbsoluteFile);

			FileUtils.makeDir(saveFile.getParentFile());

			output = new FileOutputStream(localAbsoluteFile);

			return get(remoteAbsoluteFile, output);

		} catch (Exception e) {

			throw new FTPClientException("local file not found.", e);
		} finally {
			FileUtils.close(output);

		}
	}

	/**
	 * 下载一个远程文件到指定的流 处理完后记得关闭流
	 * 
	 * @param remoteAbsoluteFile
	 * @param output
	 * @param delFile
	 * @return
	 * @throws FTPClientException
	 */
	public boolean get(String remoteAbsoluteFile, OutputStream output)
			throws FTPClientException {
		try {
			FTPClient ftpClient = getFTPClient();
			// 处理传输
			return ftpClient.retrieveFile(remoteAbsoluteFile, output);
		} catch (Exception e) {
			this.shutdown();
			throw new FTPClientException("Couldn't get file from server.", e);
		} finally {

			this._shutdown(false);
		}
	}

	public boolean exists(String path, String fileName) {

		String[] files = this.listNames(path);

		if (TypeChecker.isEmpty(files)) {
			return false;
		}

		for (String file : files) {

			if (file.endsWith(fileName)) {
				return true;
			}
		}

		return false;

	}

	/**
	 * 从ftp服务器上删除一个文件
	 * 
	 * @param delFile
	 * @param autoClose
	 *            是否自动关闭当前连接
	 * 
	 * @return
	 * @throws FTPClientException
	 */
	public boolean delete(String delFile) throws FTPClientException {
		try {
			return getFTPClient().deleteFile(delFile);
		} catch (Exception e) {
			this.shutdown();
			throw new FTPClientException("Couldn't delete file from server.", e);
		} finally {
			this._shutdown(false);
		}
	}

	/**
	 * 批量删除
	 * 
	 * @param delFiles
	 * @param autoClose
	 *            是否自动关闭当前连接
	 * 
	 * @return
	 * @throws FTPClientException
	 */
	public boolean delete(String[] delFiles) throws FTPClientException {
		try {
			FTPClient ftpClient = getFTPClient();
			for (String s : delFiles) {
				ftpClient.deleteFile(s);
			}
			return true;
		} catch (Exception e) {
			this.shutdown();
			throw new FTPClientException("Couldn't delete file from server.", e);
		} finally {
			this._shutdown(false);
		}
	}

	/**
	 * 列出远程目录下所有的文件
	 * 
	 * @param remotePath
	 *            远程目录名
	 * @param autoClose
	 *            是否自动关闭当前连接
	 * 
	 * @return 远程目录下所有文件名的列表，目录不存在或者目录下没有文件时返回0长度的数组
	 * @throws FTPClientException
	 */
	public String[] listNames(String remotePath) throws FTPClientException {
		try {

			String[] listNames = getFTPClient().listNames(remotePath);

			if (TypeChecker.isEmpty(listNames)) {
				return listNames;
			}

			return listNames;
		} catch (Exception e) {
			this.shutdown();
			throw new FTPClientException("List file Error!!! Path:"
					+ remotePath, e);
		} finally {
			this._shutdown(false);
		}
	}

	public static void main(String[] args) {

		FTPClientTemplateConfigure ftpConfigure = new FTPClientTemplateConfigure();
		ftpConfigure.setHost("10.10.134.141");
		ftpConfigure.setPort(21);
		ftpConfigure.setUserName("oa");
		ftpConfigure.setPassword("oa");

		FTPClientTemplate.setAutoCommit(false);

		FTPClientTemplate ftpClientTemplate = new FTPClientTemplate(
				ftpConfigure);

		System.out.println(ftpClientTemplate.get("A/B/C/D/index.jsp",
				"c:/index.jsp"));

		System.out.println(ftpClientTemplate.get("A/B/C/D/index.jsp",
				"c:/index.jsp"));

		FTPClientTemplate.shutdownThreadLocal();

		//
		// String folder = "A/B/C/D";
		// ftpClientTemplate.setAutoCommit(false);
		// String workFolder = null;
		// for (String temp : StringUtils.splitBySeparator(folder, "/")) {
		//
		// ftpClientTemplate.mkdir(temp, workFolder);
		// workFolder = temp;
		// }
		//
		// ftpClientTemplate.disconnect();

	}
}
