package com.rapid.j2ee.framework.core.io.net.sftp;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.List;
import java.util.Properties;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.commons.net.ftp.FTPClient;

import com.jcraft.jsch.Channel;
import com.jcraft.jsch.ChannelSftp;
import com.jcraft.jsch.JSch;
import com.jcraft.jsch.Session;
import com.jcraft.jsch.SftpException;
import com.jcraft.jsch.ChannelSftp.LsEntry;

import com.rapid.j2ee.framework.core.exception.ExceptionUtils;
import com.rapid.j2ee.framework.core.utils.*;

public class SFTPClientTemplate {

	private static ThreadLocal<ChannelSftp> channelSftpThreadLocal = new ThreadLocal<ChannelSftp>();

	private SFTPClientTemplateConfigure sftpClientTemplateConfigure;

	private static ThreadLocal<Boolean> autoCommitThreadLocal = new ThreadLocal<Boolean>() {
		@Override
		protected Boolean initialValue() {

			return true;
		}
	};

	public SFTPClientTemplate(
			SFTPClientTemplateConfigure sftpClientTemplateConfigure) {
		this.sftpClientTemplateConfigure = sftpClientTemplateConfigure;
	}

	public ChannelSftp getChannelSftp() {

		if (!TypeChecker.isNull(channelSftpThreadLocal.get())
				&& channelSftpThreadLocal.get().isConnected()
				&& !channelSftpThreadLocal.get().isClosed()) {
			return channelSftpThreadLocal.get();
		}

		channelSftpThreadLocal.set(this.connect());

		return channelSftpThreadLocal.get();

	}

	public static void setAutoCommit(boolean commit) {
		autoCommitThreadLocal.set(commit);
	}

	public static boolean isAutoCommit() {
		return TypeChecker.isNull(autoCommitThreadLocal.get())
				|| autoCommitThreadLocal.get();
	}

	/**
	 * 連接sftp服務器 *
	 * 
	 * @param host主機 *
	 * @param port端口 *
	 * @param username用戶名 *
	 * @param password密碼 *
	 * @return
	 */
	private ChannelSftp connect() {

		ChannelSftp sftp = null;
		try {
			long time = System.currentTimeMillis();
			JSch jsch = new JSch();
			jsch.getSession(this.sftpClientTemplateConfigure.getUserName(),
					sftpClientTemplateConfigure.getHost(),
					sftpClientTemplateConfigure.getPort());
			Session sshSession = jsch.getSession(
					this.sftpClientTemplateConfigure.getUserName(),
					sftpClientTemplateConfigure.getHost(),
					sftpClientTemplateConfigure.getPort());
			sshSession.setPassword(sftpClientTemplateConfigure.getPassword());
			Properties sshConfig = new Properties();
			sshConfig.put("StrictHostKeyChecking", "no");
			sshSession.setConfig(sshConfig);
			sshSession.connect();

			Channel channel = sshSession.openChannel("sftp");
			channel.connect();
			sftp = (ChannelSftp) channel;

			Logger.info("Open SFTP Session ["
					+ (System.currentTimeMillis() - time) / 1000 + " Sec]！ IP:"
					+ sftpClientTemplateConfigure.getHost() + " Port:"
					+ sftpClientTemplateConfigure.getPort() + " User Name:"
					+ sftpClientTemplateConfigure.getUserName());

			System.out.println("Open SFTP Session ["
					+ (System.currentTimeMillis() - time) / 1000 + " Sec]！ IP:"
					+ sftpClientTemplateConfigure.getHost() + " Port:"
					+ sftpClientTemplateConfigure.getPort() + " User Name:"
					+ sftpClientTemplateConfigure.getUserName());

		} catch (Exception e) {
			this.shutdown();
			throw ExceptionUtils.convertThrowableToBaseException(e);
		}
		return sftp;
	}

	public void put(String directory, String resourceFile) {
		put(directory, resourceFile, new File(resourceFile).getName());
	}

	/**
	 * 上傳文件 *
	 * 
	 */
	public void put(String directory, String resourceFile, String ftpFileName) {

		FileInputStream fileInputStream = null;

		try {

			ChannelSftp sftp = this.getChannelSftp();

			sftp.cd(directory);

			FileUtils.makeDir(new File(ftpFileName).getParentFile());

			fileInputStream = new FileInputStream(resourceFile);

			sftp.put(fileInputStream, ftpFileName);

			fileInputStream.close();

		} catch (Exception e) {

			this.shutdown();

			throw ExceptionUtils.convertThrowableToBaseException(e);

		} finally {

			FileUtils.close(fileInputStream);

			this._shutdown(false);
		}

	}

	/**
	 * 下载文件
	 * 
	 * @param directory
	 *            下载目录
	 * @param downloadFile
	 *            下载的文件
	 * @param saveFile
	 *            存在本地的路径
	 * @param sftp
	 */
	public void get(String directory, String ftpFileName, String saveFile) {

		FileOutputStream fileOutputStream = null;

		try {

			this.getChannelSftp().cd(directory);

			File saveFileFile = new File(saveFile);

			FileUtils.makeDir(saveFileFile.getParentFile());

			fileOutputStream = new FileOutputStream(saveFileFile);

			this.getChannelSftp().get(ftpFileName, fileOutputStream);

		} catch (Exception e) {

			this.shutdown();

			throw ExceptionUtils.convertThrowableToBaseException(e);

		} finally {

			FileUtils.close(fileOutputStream);

			this._shutdown(false);
		}
	}

	public void get(String directory, String ftpFileName, OutputStream os) {

		try {

			this.getChannelSftp().cd(directory);

			this.getChannelSftp().get(ftpFileName, os);

		} catch (Exception e) {
			this.shutdown();
			throw ExceptionUtils.convertThrowableToBaseException(e);
		} finally {

			this._shutdown(false);
		}
	}

	public boolean delete(String directory, String fileName) {
		try {

			if (!exists(directory, fileName)) {
				return false;
			}

			this.getChannelSftp().cd(directory);
			this.getChannelSftp().rm(fileName);

			return true;

		} catch (Exception e) {
			this.shutdown();
			throw ExceptionUtils.convertThrowableToBaseException(e);
		} finally {

			this._shutdown(false);
		}
	}

	/**
	 * 列出目錄下的文件 *
	 * 
	 * @param directory要列出的目錄 *
	 * @param sftp *
	 * @return *
	 * @throws SftpException
	 */
	public List<LsEntry> listFiles(String directory) {
		try {

			return new ArrayList<LsEntry>(this.getChannelSftp().ls(directory));
		} catch (SftpException e) {
			this.shutdown();
			throw ExceptionUtils.convertThrowableToBaseException(e);
		} finally {

			this._shutdown(false);
		}

	}

	public boolean exists(String directory, String target) {

		List<LsEntry> lsEntries = listFiles(directory);

		for (LsEntry lsEntry : lsEntries) {
			if (target.equalsIgnoreCase(lsEntry.getFilename())) {
				return true;
			}
		}

		return false;
	}

	public void shutdown() {
		_shutdown(true);
	}

	public static void shutdownThreadLocal() {

		ChannelSftp channelSftp = channelSftpThreadLocal.get();

		try {
			channelSftp.getSession().disconnect();
		} catch (Exception e) {

		}

		try {

			channelSftp.quit();
		} catch (Exception e) {

		}

		try {

			channelSftp.disconnect();
		} catch (Exception e) {

		}
		try {
			channelSftp.exit();
		} catch (Exception e) {

		}

		channelSftpThreadLocal.remove();
		autoCommitThreadLocal.remove();

		Logger.info("SFTP Session Closed!");

		System.out.println("SFTP Session Closed!");
	}

	private void _shutdown(boolean mustShutdown) {

		if (mustShutdown) {
			SFTPClientTemplate.setAutoCommit(true);
		}

		if (!SFTPClientTemplate.isAutoCommit()) {
			return;
		}

		shutdownThreadLocal();

	}

	private static final Log Logger = LogFactory
			.getLog(SFTPClientTemplate.class);

	public static void main(String[] args) {

		// SFTPClientTemplate sClientTemplate = new SFTPClientTemplate();
		// sClientTemplate.setHost("10.2.51.51");
		// sClientTemplate.setUsername("bea1");
		// sClientTemplate.setPassword("bea1test5151");

		// sClientTemplate.put(directory, resourceFile)

		// /home/bea1/Galaxy_Earth/apache-tomcat-7.0.29/webapps/GalaxyEarth

		// sClientTemplate.shutdown();

		// SFTPClientTemplate sClientTemplate = new SFTPClientTemplate();

		//
		// String targetFTPDirectory =
		// "/home/bea1/Galaxy_Earth/fileupload/galaxyearth/csmanager";
		//
		// int imageCount =
		// sClientTemplate.listFiles(targetFTPDirectory).size();
		// int proceedCount = 0;
		//
		// FileUtils.delete("C:\\image_temp");
		//
		// for (LsEntry lsEntry : sClientTemplate.listFiles(targetFTPDirectory))
		// {
		//			
		// if(lsEntry.getFilename().length()<5){
		// continue;
		// }
		//
		// if (lsEntry.getAttrs().getSize() <= 15 * 1024) {
		// continue;
		// }
		//			
		// System.out.println(lsEntry.getFilename());
		//
		// File file = new File("C:\\image_temp\\" + lsEntry.getFilename());
		//			
		// file.getParentFile().mkdir();
		//
		// File smallerFile = new File("C:\\image_temp\\smaller\\"
		// + lsEntry.getFilename());
		//
		// smallerFile.getParentFile().mkdir();
		//
		// sClientTemplate.get(targetFTPDirectory, lsEntry.getFilename(), file
		// .getAbsolutePath());
		//
		// try {
		// ImageCompressUtils.compress(new FileInputStream(file),
		// new FileOutputStream(smallerFile), 80 * 2, 90 * 2);
		//
		// sClientTemplate.put(targetFTPDirectory, smallerFile
		// .getAbsolutePath(), lsEntry.getFilename());
		//
		// } catch (Exception e) {
		//
		// e.printStackTrace();
		// }
		//
		// proceedCount++;
		//
		// System.out.println(proceedCount * 100.0 / imageCount + "%");
		// }
		//
		// sClientTemplate.shutdown();

	}
}
