package com.rapid.j2ee.framework.core.io.xls;

import java.io.File;
import java.io.IOException;
import java.io.OutputStream;
import java.util.Date;
import java.util.List;
import java.util.Locale;
import java.util.Map;

import javax.print.attribute.standard.NumberUp;

import org.springframework.util.Assert;

import com.rapid.j2ee.framework.core.exception.ExceptionUtils;
import com.rapid.j2ee.framework.core.io.xls.ExcelDefinitionWriter.Type;
import com.rapid.j2ee.framework.core.io.xls.config.CellImage;
import com.rapid.j2ee.framework.core.io.xls.config.CellType;
import com.rapid.j2ee.framework.core.io.xls.config.ExcelSheetCellTotalRecordIndicator;
import com.rapid.j2ee.framework.core.io.xls.config.ExcelSheetConfigure;
import com.rapid.j2ee.framework.core.io.xls.config.ExcelSheetRowConfigure;
import com.rapid.j2ee.framework.core.io.xls.config.ExcelSheetSummaryCellConfigure;
import com.rapid.j2ee.framework.core.io.xls.config.ExcelSheetTitleConfigure;
import com.rapid.j2ee.framework.core.io.xls.config.SheetCellChildRecords;
import com.rapid.j2ee.framework.core.io.xls.config.SheetExcelCellColumn;
import com.rapid.j2ee.framework.core.utils.NumberUtils;
import com.rapid.j2ee.framework.core.utils.TypeChecker;

import jxl.Workbook;
import jxl.WorkbookSettings;
import jxl.format.CellFormat;
import jxl.write.DateTime;
import jxl.write.Label;
import jxl.write.WritableCell;
import jxl.write.WritableImage;
import jxl.write.WritableSheet;
import jxl.write.WritableWorkbook;
import jxl.write.biff.RowsExceededException;

public class ExcelComplexWriter {

	protected WritableWorkbook workbook;

	protected WritableSheet currentSheet;

	private OutputStream os;

	private int sheetIndex = 0;

	private int sheetCurrentRowNumber = 0;

	private ExcelSheetConfigure currentExcelSheetConfigure;

	protected static final int ROW_VIEW_HEIGHT = 450;

	private ExcelDefinitionWriter excelDefinitionWriter = ExcelDefinitionWriter.ExcelDefinitionWriter_Default;

	public ExcelComplexWriter(String file, Locale l) {
		try {
			WorkbookSettings ws = new WorkbookSettings();

			ws.setLocale(l);

			workbook = Workbook.createWorkbook(new File(file), ws);

		} catch (IOException e) {
			throw ExceptionUtils.convertThrowableToBaseException(e);
		}
	}

	public ExcelComplexWriter(OutputStream os, Locale l) {
		try {
			this.os = os;
			WorkbookSettings ws = new WorkbookSettings();

			ws.setLocale(l);
			workbook = Workbook.createWorkbook(this.os, ws);

		} catch (Exception e) {
			throw ExceptionUtils.convertThrowableToBaseException(e);
		}
	}

	public void writeSheetData(ExcelSheetCustomWritable excelSheetCustomWritable) {

		try {
			excelSheetCustomWritable.write(this, this.workbook);
		} catch (Throwable e) {
			throw ExceptionUtils.convertThrowableToBaseException(e);
		}
	}

	public void writeSheetData(ExcelSheetTitleConfigure titleConfig,
			ExcelSheetConfigure sheet, List data) {

		try {
			currentExcelSheetConfigure = sheet;

			sheet.prepare();

			resetVariableValues();

			createWorksheet(titleConfig);

			createWorksheetTitle(titleConfig, currentExcelSheetConfigure);

			writeSheetAfterTitle(this.currentSheet,
					this.currentExcelSheetConfigure, this.sheetCurrentRowNumber);

			sheetCurrentRowNumber = excelDefinitionWriter
					.write(Type.BeforeWriteSheetData, this.currentSheet,
							this.sheetCurrentRowNumber,
							this.currentExcelSheetConfigure);

			createWorksheetTableColumns(currentExcelSheetConfigure);

			createWorksheetTableContents(currentExcelSheetConfigure, data);

			sheetCurrentRowNumber = excelDefinitionWriter
					.write(Type.AppendWriteSheetData, this.currentSheet,
							this.sheetCurrentRowNumber,
							this.currentExcelSheetConfigure);

		} catch (Throwable e) {
			throw ExceptionUtils.convertThrowableToBaseException(e);
		}
	}

	protected void writeSheetAfterTitle(WritableSheet currentSheet,
			ExcelSheetConfigure currentExcelSheetConfigure,
			int sheetCurrentRowNumber) throws Throwable {

	}

	public void appendWorksheetTableContents(List data) {
		try {
			createWorksheetTableContents(currentExcelSheetConfigure, data);
		} catch (Exception e) {
			throw ExceptionUtils.convertThrowableToBaseException(e);
		}
	}

	protected void resetVariableValues() {
		sheetCurrentRowNumber = 0;

	}

	public WritableSheet createWorksheet(ExcelSheetTitleConfigure titleConfig)
			throws Exception {

		return createWorksheet(titleConfig.getSheetName());

	}

	public WritableSheet createWorksheet(String sheetName) throws Exception {

		currentSheet = workbook.createSheet(sheetName, sheetIndex);

		sheetIndex++;

		return currentSheet;

	}

	private void createWorksheetTitle(ExcelSheetTitleConfigure titleConfig,
			ExcelSheetConfigure sheet) throws Exception {

		setCurrentRowIndex(titleConfig.getTitleStartRowNumber());

		this.setCurrentRowHeight(Math.max(titleConfig.getTitleCellHeight(),
				ROW_VIEW_HEIGHT * 3));

		Label titleLabel = new Label(0, titleConfig.getTitleStartRowNumber(),
				titleConfig.getTitleName(), titleConfig.getTitleCellFormat());

		currentSheet.mergeCells(0, titleConfig.getTitleStartRowNumber(), sheet
				.getSheetCellColumns().size() - 1, titleConfig
				.getTitleStartRowNumber());

		currentSheet.addCell(titleLabel);

		increaseRowByOne();

	}

	private void createWorksheetTableColumns(ExcelSheetConfigure sheet)
			throws Exception {

		List<SheetExcelCellColumn> cellColumns = sheet.getSheetCellColumns();

		Assert
				.notEmpty(cellColumns,
						"The getSheetCellColumns() in ExcelSheetConfigure must not be NULL.");

		for (int i = 0, j = cellColumns.size(); i < j; i++) {

			SheetExcelCellColumn cellColumn = cellColumns.get(i);

			currentSheet.setColumnView(i, cellColumn.getColumnWidth());

			Label column = new Label(i, this.sheetCurrentRowNumber, cellColumn
					.getColumnName(), sheet.getTableColumnCellFormat(i,
					cellColumn));

			this.currentSheet.addCell(column);
		}

	}

	private void createWorksheetTableContents(ExcelSheetConfigure sheet,
			List data) throws Exception {

		if (TypeChecker.isEmpty(data)) {
			return;
		}

		for (Object record : data) {
			createWorksheetTableContent(sheet, record);
		}

	}

	private void createWorksheetTableContent(ExcelSheetConfigure sheet,
			Object record) throws Exception {

		createWorksheetTableParentContent(sheet, record);

		appendWorksheetTableNecessaryChildrenContent(sheet, record);

		mergeTotalLabelCells(sheet, record);
	}

	private void createWorksheetTableParentContent(ExcelSheetConfigure sheet,
			Object record) throws Exception {

		List<SheetExcelCellColumn> cellColumns = sheet.getSheetCellColumns();

		increaseRowByOne();

		int currentRowTmp = this.sheetCurrentRowNumber;

		int childRowSize = 0;

		for (int column = 0, size = cellColumns.size(); column < size; column++) {

			SheetExcelCellColumn cellColumn = cellColumns.get(column);

			CellFormat contentCellFormat = sheet.findCellFormatAtTableContent(
					column, cellColumn, record);

			if (sheet instanceof ExcelSheetRowConfigure) {
				contentCellFormat = ((ExcelSheetRowConfigure) sheet)
						.findCellFormatAtTableContent(
								this.sheetCurrentRowNumber, column, cellColumn,
								record);
			}

			// merge cells
			childRowSize = mergeWritableNecessaryParentCell(column, cellColumn,
					record);

			this.currentSheet.addCell(getWritableCell(column, cellColumn, sheet
					.getCellValue(column, getRecord(cellColumn, record),
							cellColumn, record), contentCellFormat));

			if (cellColumn.getCellType() == CellType.IMAGE) {

				this.writeImage(column, (CellImage) sheet.getCellValue(column,
						getRecord(cellColumn, record), cellColumn, record));

				continue;
			}

		}

	}

	private int mergeWritableNecessaryParentCell(int column,
			SheetExcelCellColumn cellColumn, Object value) throws Exception {

		if (!(value instanceof SheetCellChildRecords)
				|| cellColumn.isChildColumnField()) {
			return 0;
		}

		List childs = ((SheetCellChildRecords) value).getChildRecords();

		if (TypeChecker.isEmpty(childs) || childs.size() == 1) {
			return 0;
		}

		// merge cell
		currentSheet.mergeCells(column, this.sheetCurrentRowNumber, column,
				this.sheetCurrentRowNumber + childs.size() - 1);

		return childs.size();

	}

	private void appendWorksheetTableNecessaryChildrenContent(
			ExcelSheetConfigure sheet, Object record) throws Exception {

		if (!(record instanceof SheetCellChildRecords)) {
			return;
		}

		// Get next record Child....

		List childs = ((SheetCellChildRecords) record).getChildRecords();

		if (TypeChecker.isEmpty(childs)) {
			return;
		}

		List<SheetExcelCellColumn> cellColumns = sheet.getSheetCellColumns();

		for (int i = 1, j = childs.size(); i < j; i++) {

			increaseRowByOne();

			Object child = childs.get(i);

			for (int l = 0, m = cellColumns.size(); l < m; l++) {
				SheetExcelCellColumn cellColumn = cellColumns.get(l);

				if (!cellColumn.isChildColumnField()) {
					continue;
				}

				CellFormat contentCellFormat = sheet
						.findCellFormatAtTableContent(l, cellColumn, record);

				if (sheet instanceof ExcelSheetRowConfigure) {
					contentCellFormat = ((ExcelSheetRowConfigure) sheet)
							.findCellFormatAtTableContent(
									this.sheetCurrentRowNumber, l, cellColumn,
									record);
				}

				this.currentSheet.addCell(getWritableCell(l, cellColumn, sheet
						.getCellValue(l, child, cellColumn, record),
						contentCellFormat));

				if (cellColumn.getCellType() == CellType.IMAGE) {

					this.writeImage(l, (CellImage) sheet.getCellValue(l, child,
							cellColumn, record));

					continue;
				}

			}

		}

	}

	private void writeImage(int column, CellImage image) {

		WritableImage writableImage = new WritableImage(column,
				this.sheetCurrentRowNumber, image.getWidth(),
				image.getHeight(), image.getImages());

		currentSheet.setColumnView(column, image.getColumnWidth());

		try {
			currentSheet
					.setRowView(sheetCurrentRowNumber, image.getRowHeight());
		} catch (RowsExceededException e) {

		}

		currentSheet.addImage(writableImage);

	}

	private Object getRecord(SheetExcelCellColumn cellColumn, Object record) {

		if (!cellColumn.isChildColumnField()) {
			return record;
		}

		List childs = ((SheetCellChildRecords) record).getChildRecords();

		if (TypeChecker.isEmpty(childs)) {
			return null;
		}

		return childs.get(0);

	}

	private WritableCell getWritableCell(int column,
			SheetExcelCellColumn cellColumn, Object value, CellFormat cellFormat)
			throws Exception {

		if (TypeChecker.isNull(value)) {
			return new Label(column, this.sheetCurrentRowNumber, "", cellFormat);
		}

		// 必须判断是否是数字
		if (TypeChecker.isBlueNumber(value)) {

			if (cellColumn.getCellType() == CellType.INTEGER) {

				return new jxl.write.Number(column, this.sheetCurrentRowNumber,
						NumberUtils.toNumber(value).intValue(), cellFormat);
			}

			if (cellColumn.getCellType() == CellType.NUMBER
					|| cellColumn.getCellType() == CellType.CURRENCY) {

				return new jxl.write.Number(column, this.sheetCurrentRowNumber,
						NumberUtils.toNumber(value).doubleValue(), cellFormat);
			}

			if (cellColumn.getCellType() == CellType.PERCENT) {
				return new jxl.write.Number(column, this.sheetCurrentRowNumber,
						NumberUtils.toNumber(value).doubleValue() / 100.0,
						cellFormat);
			}

		}

		if (cellColumn.getCellType() == CellType.DATE
				|| cellColumn.getCellType() == CellType.DATETIME_YYYYMMDDHHMM
				|| cellColumn.getCellType() == CellType.DATETIME_YYYYMMDDHHMMSS) {
			return new DateTime(column, this.sheetCurrentRowNumber,
					(Date) value, cellFormat);
		}

		if (cellColumn.getCellType() == CellType.IMAGE) {
			return new Label(column, this.sheetCurrentRowNumber, "", cellFormat);
		}

		return new Label(column, this.sheetCurrentRowNumber, String
				.valueOf(value), cellFormat);
	}

	private void mergeTotalLabelCells(ExcelSheetConfigure sheet, Object record)
			throws Exception {

		if (!(record instanceof ExcelSheetCellTotalRecordIndicator)
				|| !(sheet instanceof ExcelSheetSummaryCellConfigure)) {
			return;
		}

		ExcelSheetCellTotalRecordIndicator cellTotal = (ExcelSheetCellTotalRecordIndicator) record;

		if (!cellTotal.isTotalRecord()) {
			return;
		}

		ExcelSheetSummaryCellConfigure summarySheetConfig = (ExcelSheetSummaryCellConfigure) sheet;

		currentSheet.mergeCells(0, sheetCurrentRowNumber, summarySheetConfig
				.getLabelMergeCellSize(cellTotal.getTotalLevel()) - 1,
				sheetCurrentRowNumber);

	}

	public void close() {
		try {
			this.workbook.write();
		} catch (Exception e) {

		}
		try {
			this.workbook.close();
		} catch (Exception e) {

		}
		try {
			this.os.flush();
		} catch (Exception e) {

		}
		try {
			this.os.close();
		} catch (Exception e) {

		}
	}

	protected void setCurrentRowIndex(int rowNumber) {
		sheetCurrentRowNumber = rowNumber;
	}

	protected void increaseRowByOne() {
		setCurrentRowIndex(sheetCurrentRowNumber + 1);
	}

	protected void setCurrentRowHeight(int height) throws Exception {

		currentSheet.setRowView(this.sheetCurrentRowNumber, height);
	}

	public void addExcelDefinitionWriter(
			ExcelDefinitionWriter excelDefinitionWriter) {
		this.excelDefinitionWriter = excelDefinitionWriter;
	}

}
