package com.rapid.j2ee.framework.core.io.xml;

import java.lang.annotation.Annotation;
import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

import org.springframework.util.Assert;

import com.rapid.j2ee.framework.core.reflect.ConstructorUtils;
import com.rapid.j2ee.framework.core.reflect.InvokeUtils;
import com.rapid.j2ee.framework.core.utils.ClassUtils;
import com.rapid.j2ee.framework.core.utils.DateTimeUtils;
import com.rapid.j2ee.framework.core.utils.ObjectUtils;
import com.rapid.j2ee.framework.core.utils.StringUtils;
import com.rapid.j2ee.framework.core.utils.TypeChecker;
import com.rapid.j2ee.framework.core.utils.support.DateTimeFormat;

public class XmlWriteBuilder {

	private XmlWriter xmlWriter;

	private boolean humpTagType;

	private Map<Class, XmlElementAttributeDecoration> xmlElementAttributeDecorations;

	public XmlWriteBuilder(XmlWriter xmlWriter, boolean humpTagType) {
		this.xmlWriter = xmlWriter;
		this.humpTagType = humpTagType;
		xmlElementAttributeDecorations = new HashMap<Class, XmlElementAttributeDecoration>();
	}

	public XmlWriteBuilder() {
		this(new XmlWriter(), true);
	}

	public XmlWriteBuilder(XmlWriter xmlWriter) {
		this(xmlWriter, false);
	}

	public void createRootElement(String rootTagName) {
		this.xmlWriter.addRootElement(getTagName(rootTagName));
	}

	public void createRootEmptyElement() {
		this.xmlWriter.addRootEmptyElement();
	}

	public String toXmlText() {
		return this.xmlWriter.toXmlOriginalText();
	}

	public void addXmlTagSection(String tagName) {
		this.xmlWriter.addElement(getTagName(tagName));
	}

	public void addXmlTagSection(String tagName, Object... itemResult) {

		this.xmlWriter.addElement(getTagName(tagName));

		addXmlItems(itemResult);

	}

	public void addXmlItems(Object... itemResult) {

		if (TypeChecker.isEmpty(itemResult)) {
			return;
		}

		for (Object item : itemResult) {

			if (TypeChecker.isNull(item)) {
				continue;
			}

			if (this.isSingleObject(item)) {

				addElementForSingleObject(item, true);

				continue;
			}

			if (item instanceof Map) {

				addElementForMap("map", (Map) item, true);
				continue;
			}

			if (TypeChecker.isEmpty((Collection) item)) {
				continue;
			}

			Collection values = (Collection) item;

			Iterator value = values.iterator();

			value.hasNext();

			addElementForCollection(getTagName(value.next().getClass()),
					(Collection) item, true);
		}

	}

	private void addElementForMap(String tagName, Map results,
			boolean backToCurrentSection) {

		if (TypeChecker.isEmpty(results)) {
			return;
		}

		try {
			this.xmlWriter.addElement(this.getTagName(tagName, true));

			for (Iterator it = results.keySet().iterator(); it.hasNext();) {
				Object key = (Object) it.next();
				Assert.isTrue(key instanceof String,
						"Please provide a string as map key!");

				this.xmlWriter.addElement("key");
				xmlWriter.addElementAttribute("name", String.valueOf(key));

				addXmlItems(results.get(key));

				xmlWriter.previousElement();

			}

		} finally {
			if (backToCurrentSection) {
				xmlWriter.previousElement();
			}
		}

	}

	private String getTagElementName(Annotation fieldTagAnnotation, String name) {

		if (fieldTagAnnotation instanceof XmlFieldAttribute) {

			name = StringUtils
					.trimToEmpty(((XmlFieldAttribute) fieldTagAnnotation)
							.tagAttributeName(), name);
		} else if (fieldTagAnnotation instanceof XmlMethodAttribute) {
			name = StringUtils.trimToEmpty(
					((XmlMethodAttribute) fieldTagAnnotation)
							.tagAttributeName(), name);
		}

		return this.getTagName(name);
	}

	private boolean isSingleObject(Object item) {
		return !(item instanceof Collection) && !(item instanceof Map);
	}

	public void addElementForSingleObject(Object tagItem,
			boolean backToCurrentSection) {

		if (TypeChecker.isNull(tagItem)) {
			return;
		}

		try {

			String tagName = getTagName(tagItem.getClass());

			if (!TypeChecker.isNull(tagName)) {
				this.xmlWriter.addElement(tagName);

			} else {
				backToCurrentSection = false;
			}

			this.createTagElementsByFields(tagItem);

			this.createTagElementsByMethods(tagItem);

		} finally {
			if (backToCurrentSection) {
				xmlWriter.previousElement();
			}
		}

	}

	private void createTagElementsByFields(Object tagItem) {

		for (Field field : ClassUtils.getAllFieldsAsClassByAnnotation(tagItem
				.getClass(), XmlFieldAttribute.class)) {

			Object value = InvokeUtils.getFieldValue(tagItem, field);

			if (TypeChecker.isNull(value)) {
				continue;
			}

			if (ClassUtils.isAssignable(value.getClass(), Collection.class)) {

				addElementForCollection(getTagElementName(field
						.getAnnotation(XmlFieldAttribute.class), field
						.getName()), (Collection) value, true);
				continue;
			}

			if (ClassUtils.isAssignable(value.getClass(), Map.class)) {
				addElementForMap(getTagElementName(field
						.getAnnotation(XmlFieldAttribute.class), field
						.getName()), (Map) value, true);
				continue;
			}

			if (ObjectUtils.isWidePrimitive(value.getClass())) {

				this.addElementAttribute(tagItem, field
						.getAnnotation(XmlFieldAttribute.class), field
						.getName(), field.getType(), value);

				continue;
			}

			addElementForSingleObject(value, true);

		}

	}

	private void createTagElementsByMethods(Object tagItem) {

		for (Method method : ClassUtils.getAllMethodsAsClassByAnnotation(
				tagItem.getClass(), XmlMethodAttribute.class)) {

			Object value = InvokeUtils.invoke(tagItem, method, null);

			if (ClassUtils.isAssignable(method.getReturnType(),
					Collection.class)) {

				addElementForCollection(this.getTagElementName(method
						.getAnnotation(XmlMethodAttribute.class), method
						.getName()), (Collection) value, true);

				continue;
			}

			if (TypeChecker.isNull(value)) {
				continue;
			}

			if (ObjectUtils.isWidePrimitive(value.getClass())) {

				this.addElementAttribute(tagItem, method
						.getAnnotation(XmlMethodAttribute.class), method
						.getName(), method.getReturnType(), value);

				continue;
			}

			addElementForSingleObject(value, true);

		}
	}

	private String getCDataElementName(Object tagItem, String elementName) {

		if (tagItem.getClass().getAnnotation(XmlTag.class)
				.cdataTagNameUpperFirstCase()) {
			return StringUtils.upperStartChar(elementName);
		}

		return elementName;
	}

	private void addElementAttribute(Object tagItem, Annotation annotation,
			String targetName, Class targetType, Object value) {

		String attributeName = getTagElementName(annotation, targetName);

		String valueElement = decorateXmlTagValue(tagItem, targetName,
				convertAnyToString(value));

		if (isCDataElement(annotation)) {

			xmlWriter.addElement(getCDataElementName(tagItem, attributeName));
			xmlWriter.addElementAttribute("type", "CDATA");
			xmlWriter.addElementCDATA(valueElement);
			xmlWriter.previousElement();

			return;
		}

		this.xmlWriter.addElementAttribute(attributeName, valueElement);

		if (!TypeChecker.isNull(targetType) && targetType != String.class) {
			this.xmlWriter.addElementAttribute(getTagName(attributeName
					+ "Type"), targetType.getSimpleName());
		}

	}

	private boolean isCDataElement(Annotation fieldTagAnnotation) {

		if (fieldTagAnnotation instanceof XmlFieldAttribute) {
			return ((XmlFieldAttribute) fieldTagAnnotation).cdata();
		} else if (fieldTagAnnotation instanceof XmlMethodAttribute) {
			return ((XmlMethodAttribute) fieldTagAnnotation).cdata();
		}

		return false;
	}

	private String decorateXmlTagValue(Object tagItem, String tagName,
			String value) {

		Class<?> tagItemClass = tagItem.getClass();

		if (!tagItemClass.isAnnotationPresent(XmlTag.class)) {
			return value;
		}

		Class xmlElementAttrDecorationClazz = tagItem.getClass().getAnnotation(
				XmlTag.class).xmlElementAttributeDecoration();

		if (XmlElementAttributeDecoration.class == xmlElementAttrDecorationClazz) {
			return value;
		}

		if (!this.xmlElementAttributeDecorations.containsKey(tagItemClass)) {

			xmlElementAttributeDecorations.put(tagItemClass,
					(XmlElementAttributeDecoration) ConstructorUtils
							.newInstance(tagItem.getClass().getAnnotation(
									XmlTag.class)
									.xmlElementAttributeDecoration()));
		}

		return xmlElementAttributeDecorations.get(tagItemClass).decorate(
				tagItem, tagName, value);
	}

	public void addElementForCollection(String tagName, Collection tagItems,
			boolean backToCurrentSection) {

		if (TypeChecker.isEmpty(tagItems)) {
			return;
		}

		this.xmlWriter.addElement(this.getTagName(tagName, true));

		for (Object item : tagItems) {
			addElementForSingleObject(item, true);
		}

		if (backToCurrentSection) {
			xmlWriter.previousElement();
		}

	}

	private String convertAnyToString(Object value) {

		if (TypeChecker.isNull(value)) {
			return "";
		}

		if (value instanceof Date) {
			return DateTimeUtils.convertDateToWeb((Date) value,
					DateTimeFormat.YYYY_MM_DD_HH_MM_SS);
		}

		return StringUtils.trimToEmpty(String.valueOf(value));
	}

	private String getTagName(Class clz) {

		String tagName = clz.getSimpleName();

		if (clz.isAnnotationPresent(XmlTag.class)) {

			XmlTag xmlTag = ((XmlTag) clz.getAnnotation(XmlTag.class));

			if (xmlTag.espcaseTag()) {
				return null;
			}

			tagName = StringUtils.trimToEmpty(xmlTag.tagName(), tagName);
		}

		return getTagName(tagName);
	}

	private String getTagName(String tagName) {
		return getTagName(tagName, false);
	}

	protected String getTagName(String tagName, boolean plural) {

		tagName = StringUtils
				.convertFieldNameToXmlTagName(tagName, humpTagType);

		if (plural && !tagName.endsWith("s")) {
			tagName = tagName + "s";
		}

		return tagName;
	}

}
