package com.rapid.j2ee.framework.core.io.xml;

import java.io.StringWriter;
import java.util.HashMap;
import java.util.Map;

import org.dom4j.Document;
import org.dom4j.DocumentHelper;
import org.dom4j.Element;
import org.dom4j.io.OutputFormat;
import org.dom4j.io.XMLWriter;
import org.springframework.util.Assert;

import com.rapid.j2ee.framework.core.exception.SystemException;
import com.rapid.j2ee.framework.core.utils.StringUtils;
import com.rapid.j2ee.framework.core.utils.TypeChecker;

/**
 * 生成XML
 * 
 * @author JohnHao
 * 
 */
public class XmlWriter {

	private Document document;

	private String charset;

	private Map<String, Element> xmlElementMappers;

	private Element currentElement;

	private String currentElementFullPath;

	public static final String Empty_Tag_Name = "@@@@@@";

	public XmlWriter(String charset) {
		this.charset = StringUtils.trimToEmpty(charset, "utf-8");
		document = DocumentHelper.createDocument();
		xmlElementMappers = new HashMap<String, Element>();
	}

	public XmlWriter() {
		this("utf-8");
	}

	public Element addRootElement(String rootName) {

		Element element = document.addElement(rootName);

		this.currentElement = element;
		this.currentElementFullPath = getRevisedNodePath(rootName);
		xmlElementMappers.put(this.currentElementFullPath, element);

		return element;

	}

	public Element addRootEmptyElement() {
		return this.addRootElement(Empty_Tag_Name);
	}

	public void addElement(String elementName) {

		checkCurrentElement();

		this.currentElement = this.currentElement.addElement(elementName);

		this.currentElementFullPath = getRevisedNodePath(
				this.currentElementFullPath, elementName);

		putElementMappers(currentElement, currentElementFullPath);

	}

	public void addElementText(String text) {
		checkCurrentElement();
		this.currentElement.setText(StringUtils.trimToEmpty(text));

	}

	public void addElementCDATA(String text) {
		checkCurrentElement();
		this.currentElement.addCDATA(StringUtils.trimToEmpty(text));

	}

	public void addElementComment(String comment) {
		checkCurrentElement();
		this.currentElement.addComment(StringUtils.trimToEmpty(comment));
	}

	public void addElementAttribute(String attrName, String attrText) {
		checkCurrentElement();
		this.currentElement.addAttribute(attrName, StringUtils
				.trimToEmpty(attrText));
	}

	public void addElementAttribute(String attrName, int attrText) {
		addElementAttribute(attrName, String.valueOf(attrText));
	}

	public Element previousElement() {

		String path = this.currentElementFullPath.substring(0,
				this.currentElementFullPath.lastIndexOf("/",
						this.currentElementFullPath.length() - 2));

		return moveElementByFullNodePath(getRevisedNodePath(path));

	}

	public void moveByElementName(String elementName) {

		elementName = "/" + elementName + "/";

		String lastNode = this.currentElementFullPath
				.substring(this.currentElementFullPath.lastIndexOf(elementName)
						+ elementName.length());

		String path = this.currentElementFullPath.substring(0,
				currentElementFullPath.lastIndexOf(lastNode));

		moveElementByFullNodePath(getRevisedNodePath(path));
	}

	private void putElementMappers(Element element, String... nodePaths) {

		String nodePath = getRevisedNodePath(nodePaths);

		Object ele = xmlElementMappers.remove(nodePath);

		ele = null;

		xmlElementMappers.put(nodePath, element);

		this.currentElement = element;

	}

	private void checkCurrentElement() {
		Assert.notNull(this.currentElement, "Not current element");
	}

	public Element moveElementByFullNodePath(String... nodePaths) {

		String nodePath = getRevisedNodePath(nodePaths);

		if (!xmlElementMappers.containsKey(nodePath)) {

			throw new SystemException(
					"XmlWriter Cannot found element by path [" + nodePath + "]");

		}

		this.currentElement = xmlElementMappers.get(nodePath);
		this.currentElementFullPath = nodePath;

		return this.currentElement;
	}

	private static String getRevisedNodePath(String... nodePaths) {

		if (TypeChecker.isEmpty(nodePaths)) {
			return "/";
		}

		StringBuffer nodePathSb = new StringBuffer();

		for (String nodePath : nodePaths) {
			nodePath = nodePath.endsWith("/") ? nodePath : nodePath + "/";
			nodePathSb.append(nodePath);
		}

		return nodePathSb.toString();
	}

	public String toXmlText() {

		return this.toXmlText(OutputFormat.createCompactFormat());
	}

	public String toXmlOriginalText() {

		return this.toXmlText(OutputFormat.createPrettyPrint());
	}

	public String toXmlText(OutputFormat format) {

		XMLWriter xmlWriter = null;

		try {

			format.setEncoding(this.charset);

			StringWriter sw = new StringWriter(500);
			xmlWriter = new XMLWriter(sw, format);

			xmlWriter.write(document);
			
			return StringUtils.remove(sw.getBuffer().toString(), "<"
					+ Empty_Tag_Name + ">", "</" + Empty_Tag_Name + ">");

		} catch (Exception e) {
			return "";
		} finally {
			try {
				xmlWriter.close();
			} catch (Exception e) {

			}
		}
	}

	public static void main(String[] args) {

		/**
		 * <books> <!--This is a test for dom4j, holen, 2004.9.11--> <book
		 * show="yes"> <title>Dom4j Tutorials</title> </book> <book show="yes">
		 * <title>Lucene Studing</title> </book> <book show="no"> <title>Lucene
		 * in Action</title> </book> <owner>O'Reilly</owner> </books>
		 */

		XmlWriter xmlWriter = new XmlWriter();

		xmlWriter.addRootEmptyElement();

		xmlWriter.addElement("body");

		xmlWriter.addElementText("[Download Resource:" + "]");
		System.out.println(xmlWriter.toXmlText());

	}

}
