package com.rapid.j2ee.framework.core.reflect;

import java.lang.annotation.Annotation;
import java.lang.reflect.Field;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Modifier;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import org.apache.commons.beanutils.PropertyUtils;
import org.apache.ibatis.reflection.MetaClass;
import org.springframework.util.Assert;
import org.springframework.util.ReflectionUtils;

import com.rapid.j2ee.framework.core.exception.ExceptionUtils;
import com.rapid.j2ee.framework.core.exception.SystemException;
import com.rapid.j2ee.framework.core.utils.ClassUtils;
import com.rapid.j2ee.framework.core.utils.ObjectUtils;
import com.rapid.j2ee.framework.core.utils.TypeChecker;

public final class BeanUtils {

	public static final BeanMergePropertyAcceptor BEAN_MERGEPROPERTY_NULLVOID_ACCEPTOR = new BeanMergePropertyNullVoidAcceptor();

	private BeanUtils() {

	}

	public static Map getBeanPopulatedMap(Object bean) {

		try {

			if (TypeChecker.isNull(bean)) {
				return ObjectUtils.EMPTY_MAP;
			}

			Map beanFieldMaps = new HashMap();

			for (Field f : ClassUtils.getAllFieldsAsClass(bean.getClass())) {

				if (!BEAN_MAP_ACCEPTOR_CLASSES.contains(f.getGenericType())) {
					continue;
				}

				ReflectionUtils.makeAccessible(f);

				beanFieldMaps.put(f.getName(), ReflectionUtils
						.getField(f, bean));

			}

			return beanFieldMaps;
		} catch (Exception e) {

			throw ExceptionUtils.convertThrowableToBaseException(e);
		}
	}

	public static Object merge(Object from, Object to) {
		return merge(from, to, BEAN_MERGEPROPERTY_NULLVOID_ACCEPTOR);
	}

	public static Object merge(Object from, Object to,
			BeanMergePropertyAcceptor acceptor) {

		if (TypeChecker.isNull(from) || TypeChecker.isNull(to)) {
			return to;
		}

		MetaClass metaFromClass = MetaClass.forClass(from.getClass());

		MetaClass metaToClass = MetaClass.forClass(to.getClass());

		String[] getters = metaFromClass.getGetterNames();

		if (TypeChecker.isNull(acceptor)) {
			acceptor = BEAN_MERGEPROPERTY_NULLVOID_ACCEPTOR;
		}

		for (String getter : getters) {

			try {

				if (TypeChecker.isNull(metaFromClass.getGetInvoker(getter))
						|| TypeChecker
								.isNull(metaToClass.getSetInvoker(getter))) {
					continue;
				}

				Object value = metaFromClass.getGetInvoker(getter).invoke(from,
						null);

				if (!acceptor.accept(getter, value)) {
					continue;
				}

				metaToClass.getSetInvoker(getter).invoke(to,
						new Object[] { acceptor.getValue(value) });

			} catch (Exception e) {

				System.out.println("Error Getter :" + getter);
			}

		}

		return to;

	}

	public static Object getPropertyObject(Object bean, String fieldName) {

		try {
			return PropertyUtils.getProperty(bean, fieldName);
		} catch (Exception e) {

			throw new SystemException("Sorry, the method [" + fieldName
					+ "] cannot be invoked " + bean.getClass().getName(), e);
		}
	}

	public static String getProperty(Object bean, String fieldName) {
		try {
			if (TypeChecker.isNull(bean)) {
				return "";
			}

			return org.apache.commons.beanutils.BeanUtils.getProperty(bean,
					fieldName);
		} catch (Exception e) {

			throw new SystemException("Sorry, the method [" + fieldName
					+ "] cannot be invoked " + bean.getClass().getName(), e);
		}
	}

	public static void setProperty(Object targetBean, String fieldName,
			Object fieldValue) {
		try {

			if (TypeChecker.isNull(targetBean)) {
				return;
			}

			org.apache.commons.beanutils.BeanUtils.setProperty(targetBean,
					fieldName, fieldValue);

		} catch (Exception e) {

			throw new SystemException("Sorry, the method [" + fieldName
					+ "] cannot be invoked " + targetBean.getClass().getName(),
					e);
		}
	}

	public static void copy(Object from, Object to,
			Class<? extends Annotation>... annoationClasses) {

		for (Class annotationClass : annoationClasses) {
			for (Field field : ClassUtils.getAllFieldsAsClassByAnnotation(from
					.getClass(), annotationClass)) {
				InvokeUtils.setField(to, field, InvokeUtils.getFieldValue(from,
						field));
			}
		}

	}

	public static void copy(Object from, Object to, boolean ignoreCaseField) {

		Assert.notNull(from);
		Assert.notNull(to);

		for (Field field : ClassUtils.getAllFieldsAsClass(from.getClass())) {

			Field toField = ClassUtils.getFieldByIngoreCaseFieldName(to
					.getClass(), field.getName());

			if (TypeChecker.isNull(toField)) {
				continue;
			}

			if (Modifier.isFinal(toField.getModifiers())) {
				continue;
			}

			InvokeUtils.setSetterMethodOrField(to, toField, InvokeUtils
					.getFieldValue(from, field));

		}
	}

	public static Object populate(Object bean, Map values) {
		try {
			org.apache.commons.beanutils.BeanUtils.populate(bean, values);

			return bean;
		} catch (IllegalAccessException e) {
			throw ExceptionUtils.convertThrowableToBaseException(e);
		} catch (InvocationTargetException e) {
			throw ExceptionUtils.convertThrowableToBaseException(e);
		}
	}

	private static final Set<Class> BEAN_MAP_ACCEPTOR_CLASSES = new HashSet<Class>();

	static {
		BEAN_MAP_ACCEPTOR_CLASSES.add(String.class);
		BEAN_MAP_ACCEPTOR_CLASSES.add(Integer.class);
		BEAN_MAP_ACCEPTOR_CLASSES.add(Double.class);
		BEAN_MAP_ACCEPTOR_CLASSES.add(Float.class);
	}
}
