package com.rapid.j2ee.framework.core.reflect;

import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.beanutils.PropertyUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.BeanWrapper;
import org.springframework.util.Assert;
import org.springframework.util.ReflectionUtils;

import com.rapid.j2ee.framework.core.exception.BaseException;
import com.rapid.j2ee.framework.core.exception.ExceptionUtils;
import com.rapid.j2ee.framework.core.exception.SystemException;
import com.rapid.j2ee.framework.core.exception.WriteFileSystemException;
import com.rapid.j2ee.framework.core.utils.ClassUtils;
import com.rapid.j2ee.framework.core.utils.ObjectUtils;
import com.rapid.j2ee.framework.core.utils.StringUtils;
import com.rapid.j2ee.framework.core.utils.TypeChecker;

public final class InvokeUtils {

	private final static String[] PREFIXS_METHODS = new String[] { "" };

	private InvokeUtils() {

	}

	public static Object invoke(Object target, Method method, Object[] args) {
		try {

			if (!method.isAccessible()) {
				method.setAccessible(true);
			}

			return method.invoke(target, args);

		} catch (Exception e) {

			BaseException ep = ExceptionUtils
					.convertThrowableToBaseException(e);

			if (ep instanceof WriteFileSystemException) {
				throw ep;
			}

			throw new SystemException("Sorry, the method [" + method.getName()
					+ "] cannot be invoked in " + target.getClass().getName(),
					e);
		}
	}

	public static Object invoke(Object target, String methodName,
			Class[] argClasses, Object[] args) {
		return invoke(target, methodName, argClasses, args, PREFIXS_METHODS);
	}

	public static Object invoke(Object target, String methodName,
			Class[] argClasses, Object[] args, String[] prefixs) {
		return invoke(target, methodName, argClasses, args, prefixs, null);

	}

	public static Object invoke(Object target, String methodName,
			Class[] argClasses, Object[] args, String[] prefixs,
			MethodInvokeCallback callback) {

		if (target == null) {
			return null;
		}

		Method method = getMethod(target.getClass(), methodName, argClasses,
				prefixs, true);

		if (TypeChecker.isNull(method)) {
			throw new SystemException("Sorry, the method [" + methodName
					+ "] cannot be invoked in " + target.getClass().getName());
		}

		if (!method.isAccessible()) {
			method.setAccessible(true);
		}

		if (!TypeChecker.isNull(callback)) {
			callback.callback(method);
		}

		try {
			return method.invoke(target, args);

		} catch (Throwable e) {

			BaseException ep = ExceptionUtils
					.convertThrowableToBaseException(e);

			if (ep instanceof WriteFileSystemException) {
				throw ep;
			}

			throw new SystemException("Sorry, the method [" + methodName
					+ "] cannot be invoked in " + target.getClass().getName(),
					e);
		}

	}

	public static Method getMethod(Class clz, String methodName,
			Class[] paramClass, String[] prefixs, boolean ignoreExp) {
		return MethodCache.getInstance().getMethod(clz, methodName, paramClass,
				prefixs, ignoreExp);
	}

	public static Method getMethod(Class clz, String methodName,
			Class[] paramClass, boolean ignoreExp) {
		return getMethod(clz, methodName, paramClass, PREFIXS_METHODS,
				ignoreExp);
	}

	public static Method findMethod(Class clz, String methodName) {

		Method[] methods = ClassUtils.getAllMethodsAsClass(clz);

		for (Method method : methods) {

			if (method.getName().equals(methodName)) {
				return method;
			}
		}

		return null;
	}

	public static Method findMethod(Class clz, String methodName,
			Class[] paramClass, boolean ignoreExp) {
		try {

			if (TypeChecker.isNull(paramClass)) {

				return findMethod(clz, methodName);

			}

			return clz.getDeclaredMethod(methodName, paramClass);

		} catch (Exception e) {
			if (ignoreExp) {
				return null;
			}
			throw new SystemException("Sorry, the method [" + methodName
					+ "] cannot be found in " + clz.getName(), e);
		}
	}

	public static boolean isMethodAvailable(Class clz, String methodName,
			Class[] paramClass) {
		try {
			Method method = findMethod(clz, methodName, paramClass, true);

			return !TypeChecker.isNull(method);

		} catch (Throwable e) {

			return false;
		}
	}

	public static String getMethodName(String prefix, String methodName) {

		if (TypeChecker.isEmpty(prefix)) {
			return methodName;
		}

		if (methodName.length() <= 1) {
			return prefix + methodName.toUpperCase();
		}

		return prefix + methodName.substring(0, 1).toUpperCase()
				+ methodName.substring(1);

	}

	private static class MethodCache {

		private Map<String, Method> methodContainers = new HashMap<String, Method>();

		private static final MethodCache INSTANCE = new MethodCache();

		private MethodCache() {

		}

		public static MethodCache getInstance() {
			return INSTANCE;
		}

		public synchronized Method getMethod(Class clz, String methodName,
				Class[] paramClass, String[] prefixs, boolean ignoreExp) {

			String key = getMethodCacheKey(clz, methodName, paramClass);

			if (methodContainers.containsKey(key)) {

				Method method = methodContainers.get(key);

				if (TypeChecker.isNull(method)) {
					if (!ignoreExp) {
						throw new SystemException("Sorry, any methods "
								+ methodName + " cannot be discoveried in "
								+ clz.getName());
					}

				}

				return method;

			}

			StringBuffer allMethods = new StringBuffer(50);

			for (String prefix : getPrefixs(prefixs)) {

				String realMethodName = getMethodName(prefix, methodName);

				allMethods.append(realMethodName + "(...) ");

				if (InvokeUtils.isMethodAvailable(clz, realMethodName,
						paramClass)) {
					Method method = InvokeUtils.findMethod(clz, getMethodName(
							prefix, methodName), paramClass, true);

					methodContainers.put(key, method);

					return methodContainers.get(key);
				}
			}

			methodContainers.put(key, null);

			if (!ignoreExp) {
				throw new SystemException("Sorry, any methods " + allMethods
						+ " cannot be discoveried in " + clz.getName());
			}

			return null;

		}

		private List<String> getPrefixs(String[] prefixs) {

			if (prefixs == null) {
				prefixs = new String[0];
			}

			List<String> prefixLists = new ArrayList<String>(prefixs.length + 1);

			prefixLists.add("");

			if (prefixs.length > 0) {
				prefixLists.addAll(Arrays.asList(prefixs));
			}

			return prefixLists;
		}

		private static String getMethodCacheKey(Class clz, String methodName,
				Class[] paramClass) {

			if (paramClass == null) {
				return clz.getName() + "." + methodName;
			}

			String paramKey = "";

			for (Class paramClz : paramClass) {
				paramKey += ("." + paramClz.getName());
			}

			return clz.getName() + "." + methodName + paramKey;
		}

	}

	public static Field findField(Class targetClz, String fieldName) {
		return ReflectionUtils.findField(targetClz, fieldName);
	}

	public static Object getFieldValue(Object target, Field field) {
		Assert.notNull(target);

		Assert.notNull(field);

		ReflectionUtils.makeAccessible(field);

		return ReflectionUtils.getField(field, target);
	}

	public static Object getFieldValue(Object target, String fieldName) {
		Assert.notNull(target);

		return getFieldValue(target, findField(target.getClass(), fieldName));
	}

	public static void setField(Object target, String fieldName, Object value) {

		Assert.notNull(target);

		Field field = InvokeUtils.findField(target.getClass(), fieldName);

		setField(target, field, value);
	}

	public static void setSetterMethodOrField(Object target, Field fieldName,
			Object value) {
		try {

			if (TypeChecker.isNull(value)) {
				value = ObjectUtils.getPrimitiveNullValue(fieldName.getType());
			}

			if (PropertyUtils.isWriteable(target, fieldName.getName())) {
				PropertyUtils.setProperty(target, fieldName.getName(), value);
			} else {
				InvokeUtils.setField(target, fieldName, value);
			}

		} catch (Exception e) {

			InvokeUtils.setField(target, fieldName, value);
		}
	}

	public static void setSetterMethodOrField(Object target, String fieldName,
			Object value) {
		try {

			if (PropertyUtils.isWriteable(target, fieldName)) {

				BeanUtils.setProperty(target, fieldName, value);

			} else {

				InvokeUtils.setField(target, fieldName, value);
			}

		} catch (Exception e) {

			InvokeUtils.setField(target, fieldName, value);
		}
	}

	public static void setField(Object target, Field field, Object value) {
		try {

			Assert.notNull(target);
			Assert.notNull(field);

			ReflectionUtils.makeAccessible(field);

			if (field.getType().isArray() && (value instanceof String)) {

				value = StringUtils.splitBySeparators(String.valueOf(value),
						",", ";");

			}

			if (TypeChecker.isNull(value)) {
				value = ObjectUtils.getPrimitiveNullValue(field.getType());
			} else {

				if ((ObjectUtils.isBasicPrimitiveType(field.getType()) || Number.class
						.isAssignableFrom(field.getType()))
						&& (value instanceof String)) {
					value = ObjectUtils.getPrimitiveValue(field.getType(),
							value);
				}

			}

			ReflectionUtils.setField(field, target, value);

		} catch (Exception e) {
			Logger
					.error(
							"\r\n--------------------------------------------------------------------\r\n Field:"
									+ field
									+ " Value :"
									+ value
									+ " Target:"
									+ target, e);
			throw ExceptionUtils.convertThrowableToBaseException(e);
		}
	}

	public static Class getCollectionActualGenericType(Field f) {

		if (Collection.class.isAssignableFrom(f.getType())) {

			// 关键的地方，如果是List类型，得到其Generic的类型

			Type fc = f.getGenericType();

			// 【3】如果是泛型参数的类型
			if (fc instanceof ParameterizedType) {

				ParameterizedType pt = (ParameterizedType) fc;

				return (Class) pt.getActualTypeArguments()[0];
			}
		}

		return null;

	}

	private final static Log Logger = LogFactory.getLog(InvokeUtils.class);

}
