package com.rapid.j2ee.framework.core.reflect;

import java.lang.reflect.Constructor;
import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.lang.reflect.ReflectPermission;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;

import org.apache.commons.collections.MultiHashMap;

import com.rapid.j2ee.framework.core.exception.SystemException;
import com.rapid.j2ee.framework.core.reflect.invoker.GetFieldInvoker;
import com.rapid.j2ee.framework.core.reflect.invoker.Invoker;
import com.rapid.j2ee.framework.core.reflect.invoker.MethodInvoker;
import com.rapid.j2ee.framework.core.reflect.invoker.SetFieldInvoker;
import com.rapid.j2ee.framework.core.reflect.property.PropertyNamer;
import com.rapid.j2ee.framework.core.utils.ObjectUtils;

public class Reflector {

	private Class<?> type;

	private String[] readablePropertyNames = ObjectUtils.EMPTY_STRING_ARRAYS;

	private String[] writeablePropertyNames = ObjectUtils.EMPTY_STRING_ARRAYS;

	private Map<String, Invoker> setMethods = new HashMap<String, Invoker>();

	private Map<String, Invoker> getMethods = new HashMap<String, Invoker>();

	private Map<String, Class<?>> setTypes = new HashMap<String, Class<?>>();

	private Map<String, Class<?>> getTypes = new HashMap<String, Class<?>>();

	private Constructor<?> defaultConstructor;

	private Map<String, String> caseInsensitivePropertyMap = new HashMap<String, String>();

	/*
	 * Gets an instance of ClassInfo for the specified class.
	 * 
	 * @param clazz The class for which to lookup the method cache. @return The
	 * method cache for the class
	 */
	public static Reflector forClass(Class<?> clazz) {
		return new Reflector(clazz);
	}

	private Reflector(Class<?> clazz) {
		type = clazz;
		addDefaultConstructor(clazz);
		addGetMethods(clazz);
		addSetMethods(clazz);
		addFields(clazz);
		readablePropertyNames = getMethods.keySet().toArray(
				new String[getMethods.keySet().size()]);
		writeablePropertyNames = setMethods.keySet().toArray(
				new String[setMethods.keySet().size()]);
		for (String propName : readablePropertyNames) {
			caseInsensitivePropertyMap.put(
					propName.toUpperCase(Locale.ENGLISH), propName);
		}
		for (String propName : writeablePropertyNames) {
			caseInsensitivePropertyMap.put(
					propName.toUpperCase(Locale.ENGLISH), propName);
		}
	}

	/*
	 * Gets the name of the class the instance provides information for
	 * 
	 * @return The class name
	 */
	public Class<?> getType() {
		return type;
	}

	public Constructor<?> getDefaultConstructor() {
		if (defaultConstructor != null) {
			return defaultConstructor;
		} else {
			throw new SystemException("There is no default constructor for "
					+ type);
		}
	}

	public Invoker getSetInvoker(String propertyName) {
		Invoker method = setMethods.get(propertyName);
		if (method == null) {
			throw new SystemException("There is no setter for property named '"
					+ propertyName + "' in '" + type + "'");
		}
		return method;
	}

	public Invoker getGetInvoker(String propertyName) {
		Invoker method = getMethods.get(propertyName);
		if (method == null) {
			throw new SystemException("There is no getter for property named '"
					+ propertyName + "' in '" + type + "'");
		}
		return method;
	}

	/*
	 * Gets the type for a property setter
	 * 
	 * @param propertyName - the name of the property @return The Class of the
	 * propery setter
	 */
	public Class<?> getSetterType(String propertyName) {
		Class<?> clazz = setTypes.get(propertyName);
		if (clazz == null) {
			throw new SystemException("There is no setter for property named '"
					+ propertyName + "' in '" + type + "'");
		}
		return clazz;
	}

	/*
	 * Gets the type for a property getter
	 * 
	 * @param propertyName - the name of the property @return The Class of the
	 * propery getter
	 */
	public Class<?> getGetterType(String propertyName) {
		Class<?> clazz = getTypes.get(propertyName);
		if (clazz == null) {
			throw new SystemException("There is no getter for property named '"
					+ propertyName + "' in '" + type + "'");
		}
		return clazz;
	}

	/*
	 * Gets an array of the readable properties for an object
	 * 
	 * @return The array
	 */
	public String[] getGetablePropertyNames() {
		return readablePropertyNames;
	}

	/*
	 * Gets an array of the writeable properties for an object
	 * 
	 * @return The array
	 */
	public String[] getSetablePropertyNames() {
		return writeablePropertyNames;
	}

	/*
	 * Check to see if a class has a writeable property by name
	 * 
	 * @param propertyName - the name of the property to check @return True if
	 * the object has a writeable property by the name
	 */
	public boolean hasSetter(String propertyName) {
		return setMethods.keySet().contains(propertyName);
	}

	/*
	 * Check to see if a class has a readable property by name
	 * 
	 * @param propertyName - the name of the property to check @return True if
	 * the object has a readable property by the name
	 */
	public boolean hasGetter(String propertyName) {
		return getMethods.keySet().contains(propertyName);
	}

	public String findPropertyName(String name) {
		return caseInsensitivePropertyMap.get(name.toUpperCase(Locale.ENGLISH));
	}

	private void addDefaultConstructor(Class<?> clazz) {
		Constructor<?>[] consts = clazz.getDeclaredConstructors();
		for (Constructor<?> constructor : consts) {
			if (constructor.getParameterTypes().length == 0) {
				if (canAccessPrivateMethods()) {
					try {
						constructor.setAccessible(true);
					} catch (Exception e) {
						// Ignored. This is only a final precaution, nothing we
						// can do.
					}
				}
				if (constructor.isAccessible()) {
					this.defaultConstructor = constructor;
				}
			}
		}
	}

	private void addGetMethods(Class<?> cls) {
		MultiHashMap methodMaps = getClassMethods(cls);

		for (Iterator<Class> it = methodMaps.keySet().iterator(); it.hasNext();) {

			Class clz = it.next();

			Collection<Method> methods = methodMaps.getCollection(clz);

			for (Method method : methods) {

				String name = method.getName();

				if (name.startsWith("get") && name.length() > 3) {
					if (method.getParameterTypes().length == 0) {
						name = PropertyNamer.methodToProperty(name);
						addGetMethod(clz, name, method);
					}
				} else if (name.startsWith("is") && name.length() > 2) {
					if (method.getParameterTypes().length == 0) {
						name = PropertyNamer.methodToProperty(name);
						addGetMethod(clz, name, method);
					}
				}
			}

		}
	}

	private void addGetMethod(Class target, String name, Method method) {
		if (isValidPropertyName(name)) {
			getMethods.put(name, new MethodInvoker(target, method));
			getTypes.put(name, method.getReturnType());
		}
	}

	private void addSetMethods(Class<?> cls) {
		Map<String, List<Method>> conflictingSetters = new HashMap<String, List<Method>>();

		MultiHashMap methodMaps = getClassMethods(cls);

		for (Iterator<Class> it = methodMaps.keySet().iterator(); it.hasNext();) {

			Class clz = it.next();

			Collection<Method> methods = methodMaps.getCollection(clz);

			for (Method method : methods) {
				String name = method.getName();
				if (name.startsWith("set") && name.length() > 3) {
					if (method.getParameterTypes().length == 1) {
						name = PropertyNamer.methodToProperty(name);
						addSetterConflict(conflictingSetters, name, method);
					}
				}
			}

			resolveSetterConflicts(clz, conflictingSetters);

		}
	}

	private void addSetterConflict(
			Map<String, List<Method>> conflictingSetters, String name,
			Method method) {
		List<Method> list = conflictingSetters.get(name);
		if (list == null) {
			list = new ArrayList<Method>();
			conflictingSetters.put(name, list);
		}
		list.add(method);
	}

	private void resolveSetterConflicts(Class clz,
			Map<String, List<Method>> conflictingSetters) {
		for (String propName : conflictingSetters.keySet()) {
			List<Method> setters = conflictingSetters.get(propName);
			Method firstMethod = setters.get(0);
			if (setters.size() == 1) {
				addSetMethod(clz, propName, firstMethod);
			} else {
				Class<?> expectedType = getTypes.get(propName);
				if (expectedType == null) {

					continue;
					/*
					 * throw new ReflectionSystemException( "Illegal overloaded
					 * setter method with ambiguous type for property " +
					 * propName + " in class " + firstMethod.getDeclaringClass() + ".
					 * This breaks the JavaBeans " + "specification and can
					 * cause unpredicatble results.");
					 */
				} else {
					Iterator<Method> methods = setters.iterator();
					Method setter = null;
					while (methods.hasNext()) {
						Method method = methods.next();
						if (method.getParameterTypes().length == 1
								&& expectedType.equals(method
										.getParameterTypes()[0])) {
							setter = method;
							break;
						}
					}
					if (setter == null) {
						continue;
						// throw new ReflectionSystemException(
						// "Illegal overloaded setter method with ambiguous type
						// for property "
						// + propName
						// + " in class "
						// + firstMethod.getDeclaringClass()
						// + ". This breaks the JavaBeans "
						// + "specification and can cause unpredicatble
						// results.");
					}
					addSetMethod(clz, propName, setter);
				}
			}
		}
	}

	private void addSetMethod(Class clz, String name, Method method) {
		if (isValidPropertyName(name)) {
			setMethods.put(name, new MethodInvoker(clz, method));
			setTypes.put(name, method.getParameterTypes()[0]);
		}
	}

	private void addFields(Class<?> clazz) {
		Field[] fields = clazz.getDeclaredFields();
		for (Field field : fields) {
			if (canAccessPrivateMethods()) {
				try {
					field.setAccessible(true);
				} catch (Exception e) {
					// Ignored. This is only a final precaution, nothing we can
					// do.
				}
			}
			if (field.isAccessible()) {
				if (!setMethods.containsKey(field.getName())) {
					addSetField(field);
				}
				if (!getMethods.containsKey(field.getName())) {
					addGetField(field);
				}
			}
		}
		if (clazz.getSuperclass() != null) {
			addFields(clazz.getSuperclass());
		}
	}

	private void addSetField(Field field) {
		if (isValidPropertyName(field.getName())) {
			setMethods.put(field.getName(), new SetFieldInvoker(field));
			setTypes.put(field.getName(), field.getType());
		}
	}

	private void addGetField(Field field) {
		if (isValidPropertyName(field.getName())) {
			getMethods.put(field.getName(), new GetFieldInvoker(field));
			getTypes.put(field.getName(), field.getType());
		}
	}

	private boolean isValidPropertyName(String name) {
		return !(name.startsWith("$") || "serialVersionUID".equals(name) || "class"
				.equals(name));
	}

	/*
	 * This method returns an array containing all methods declared in this
	 * class and any superclass. We use this method, instead of the simpler
	 * Class.getMethods(), because we want to look for private methods as well.
	 * 
	 * @param cls The class @return An array containing all methods in this
	 * class
	 */

	private MultiHashMap getClassMethods(Class<?> cls) {

		MultiHashMap uniqueMethods = new MultiHashMap();
		Set<String> methodCache = new HashSet<String>();

		Class<?> currentClass = cls;

		while (currentClass != null) {

			addUniqueMethods(methodCache, uniqueMethods, currentClass,
					currentClass.getDeclaredMethods());

			// we also need to look for interface methods -
			// because the class may be abstract
			Class<?>[] interfaces = currentClass.getInterfaces();

			for (Class<?> anInterface : interfaces) {
				addUniqueMethods(methodCache, uniqueMethods, anInterface,
						anInterface.getMethods());
			}

			currentClass = currentClass.getSuperclass();
		}

		return uniqueMethods;
	}

	private void addUniqueMethods(Set<String> methodCache,
			MultiHashMap uniqueMethods, Class type, Method[] methods) {

		for (Method currentMethod : methods) {
			if (!currentMethod.isBridge()) {

				String signature = getSignature(currentMethod);

				if (methodCache.contains(signature)) {
					continue;
				}
				// check to see if the method is already known
				// if it is known, then an extended class must have
				// overridden a method
				if (!uniqueMethods.containsKey(signature)) {
					if (canAccessPrivateMethods()) {
						try {
							currentMethod.setAccessible(true);
						} catch (Exception e) {

						}
					}

					uniqueMethods.put(type, currentMethod);
				}
			}
		}
	}

	private String getSignature(Method method) {

		Class<?>[] parameters = method.getParameterTypes();

		StringBuffer sb = new StringBuffer(method.getName().length()
				+ parameters.length * 10);

		sb.append(method.getName() + ":");

		for (int i = 0, j = parameters.length; i < j; i++) {
			sb.append(parameters[i].getName());
			sb.append(',');
		}
		return sb.toString();
	}

	private static boolean canAccessPrivateMethods() {
		try {
			SecurityManager securityManager = System.getSecurityManager();
			if (null != securityManager) {
				securityManager.checkPermission(new ReflectPermission(
						"suppressAccessChecks"));
			}
		} catch (SecurityException e) {
			return false;
		}
		return true;
	}

}
