package com.rapid.j2ee.framework.core.spring;

import java.lang.reflect.Field;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Locale;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.DisposableBean;
import org.springframework.beans.factory.annotation.Autowire;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.config.ListFactoryBean;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.util.Assert;

import com.rapid.j2ee.framework.core.reflect.InvokeUtils;
import com.rapid.j2ee.framework.core.utils.ClassUtils;
import com.rapid.j2ee.framework.core.utils.CollectionsUtil;
import com.rapid.j2ee.framework.core.utils.StringUtils;
import com.rapid.j2ee.framework.core.utils.TypeChecker;

/**
 * 用于持有spring的applicationContext,一个系统只能有一个ApplicationContextHolder
 * 
 * @author John Hao
 * 
 */
public class SpringApplicationContextHolder implements ApplicationContextAware,
		DisposableBean {

	private static Log log = LogFactory
			.getLog(SpringApplicationContextHolder.class);

	private static ApplicationContext applicationContext;

	private static boolean isDestroyed = false;

	public void setApplicationContext(ApplicationContext context)
			throws BeansException {

		this.isDestroyed = false;
		this.applicationContext = context;
		log.info("holded applicationContext,displayName:"
				+ applicationContext.getDisplayName());
	}

	public static ApplicationContext getApplicationContext() {
		if (isDestroyed) {
			throw new IllegalStateException(
					"'ApplicationContextHolder already destroyed.");
		}
		if (applicationContext == null) {
			throw new IllegalStateException(
					"'applicationContext' property is null,ApplicationContextHolder not yet init.");
		}
		return applicationContext;
	}

	public static boolean isApplicationContextHolden() {
		return applicationContext != null;
	}

	public static Object getBean(String beanName) {
		return getApplicationContext().getBean(beanName);
	}

	public static <T> T getBean(Class<T> beanClass) {
		return (T) getApplicationContext().getBean(beanClass);
	}

	public void destroy() throws Exception {
		if (applicationContext != null) {
			log.info("destroy() ,displayName:"
					+ applicationContext.getDisplayName());
		}
		this.isDestroyed = true;
		this.applicationContext = null;
	}

	public static void inject(Object object) {

		if (!isApplicationContextHolden()) {
			return;
		}

		for (Field field : ClassUtils.getAllFieldsAsClassByAnnotation(object
				.getClass(), Autowired.class)) {

			if (ApplicationContext.class.isAssignableFrom(field.getType())) {
				InvokeUtils.setField(object, field, applicationContext);
				continue;
			}

			if (List.class.isAssignableFrom(field.getType())) {

				Class type = ClassUtils.forName(ClassUtils
						.getFieldCollectionItemClassName(field));

				autowireObjectFieldValue(object, field, type);

				continue;
			}

			autowireObjectFieldValue(object, field, field.getType());

		}
	}

	public static void autowireObjectFieldValue(Object object, String field,
			Class type) {

		autowireObjectFieldValue(object, ClassUtils
				.getFieldByIngoreCaseFieldName(object.getClass(), field), type);

	}

	public static void autowireObjectFieldValue(Object object, Field field,
			Class type) {

		Autowired autowired = (Autowired) field.getAnnotation(Autowired.class);

		if (Collection.class.isAssignableFrom(field.getType())) {

			Object value = new ArrayList(applicationContext
					.getBeansOfType(type).values());

			if (autowired != null && autowired.required()) {
				Assert.notEmpty((ArrayList) value);
			}

			InvokeUtils.setField(object, field, value);

			return;

		}

		if (field.getType().isArray()) {

			Object value = CollectionsUtil.toArray(applicationContext
					.getBeansOfType(type).values(), type);

			if (autowired != null && autowired.required()) {
				Assert.notEmpty((Object[]) value);
			}

			InvokeUtils.setField(object, field, value);

			return;
		}

		if (field.isAnnotationPresent(Qualifier.class)) {

			String beanName = field.getAnnotation(Qualifier.class).value();

			if (autowired != null && autowired.required() == false) {

				if (!applicationContext.containsBean(beanName)) {
					return;
				}

			}

			InvokeUtils.setField(object, field, applicationContext
					.getBean(beanName));

			return;

		}

		if (autowired != null && autowired.required() == false) {

			if (TypeChecker.isEmpty(applicationContext
					.getBeanNamesForType(type))) {
				return;
			}

		}

		InvokeUtils.setField(object, field, applicationContext.getBean(type));

	}

	public static String getMessage(String code, Object[] args, Locale locale) {
		return applicationContext.getMessage(code, args, locale);
	}

	public static String getMessage(String code, Object[] args) {
		return applicationContext.getMessage(code, args, Locale.CHINA);
	}

	public static String getMessage(String code, Locale locale) {
		return applicationContext.getMessage(code, null, locale);
	}

	public static String getMessage(String code) {
		return applicationContext.getMessage(code, null, Locale.CHINA);
	}

	public static void main(String[] args) {
		List<String> tests = new ArrayList<String>(5);
		tests.add("1");
		tests.add("2");

		System.out.println(StringUtils.printArrays(tests.toArray()));
	}
}
