package com.rapid.j2ee.framework.core.utils;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.List;
import java.util.zip.CRC32;
import java.util.zip.CheckedOutputStream;
import java.util.zip.Deflater;

import org.apache.tools.zip.ZipEntry;
import org.apache.tools.zip.ZipFile;
import org.apache.tools.zip.ZipOutputStream;

import com.rapid.j2ee.framework.core.exception.ExceptionUtils;
import com.rapid.j2ee.framework.core.utils.support.Callable;

/**
 * 功能:使用Apache Ant里提供的org.apache.tools.zip实现zip压缩和解压 (支持中文文件名)
 * 解决了由于java.util.zip包不支持汉字的问题。 使用java.util.zip包时,当zip文件中有名字为中文的文件时,
 * 就会出现异常:"Exception in thread "main " java.lang.IllegalArgumentException at
 * java.util.zip.ZipInputStream.getUTF8String(ZipInputStream.java:285)
 * 
 * 
 * 使用 Ant.jar
 */
@SuppressWarnings("unchecked")
public class AntZipUtils {

	public static boolean zip(final String zipFilePath) {
		return zip(zipFilePath, false);
	}

	/**
	 * Zip File & Path
	 * 
	 * @param zipFilePath
	 * 
	 * @return success / false
	 */
	public static boolean zip(final String zipFilePath,
			boolean includedZipPackageName) {

		File zipFilePathFile = new File(zipFilePath);

		if (zipFilePathFile.isFile()) {

			return zip(zipFilePath, FileUtils
					.getFileNameAndExtension(zipFilePath)[0]
					+ ".zip", includedZipPackageName);

		}

		if (TypeChecker.isNull(zipFilePathFile.getParentFile())) {
			throw new IllegalArgumentException("The disk is not supported! "
					+ zipFilePath);
		}

		return zip(zipFilePath, new File(zipFilePathFile.getParent(),
				zipFilePathFile.getName() + ".zip"), includedZipPackageName);

	}

	public static boolean zip(final String zipFilePath,
			final String zipStoreFullPathFile) {
		return zip(zipFilePath, zipStoreFullPathFile, false);
	}

	/**
	 * Zip File
	 * 
	 * @param zipFilePath
	 * @param zipStoreFullPathFile
	 * @return
	 */
	public static boolean zip(final String zipFilePath,
			final String zipStoreFullPathFile, boolean includedZipPackageName) {
		return zip(zipFilePath, new File(zipStoreFullPathFile),
				includedZipPackageName);
	}

	/**
	 * Zip File
	 * 
	 * @param zipFilePath
	 * @param zipStoreFullPathFile
	 * @return
	 */
	public static boolean zip(final String zipFilePath,
			final File zipStoreFullPathFile, boolean includedZipPackageName) {
		try {

			File targetFile = zipStoreFullPathFile;

			FileUtils.makeDir(zipStoreFullPathFile);

			if (targetFile.isFile() && targetFile.exists()) {
				FileUtils.delete(targetFile);
			}

			File zipFilePathFile = new File(zipFilePath);

			if (!zipFilePathFile.exists()) {
				System.err.println("指定的要压缩的文件或目录不存在.");
				return false;
			}

			zip(zipFilePath, newZipOutputStream(zipStoreFullPathFile), true,
					includedZipPackageName);

		} catch (Exception e) {
			throw ExceptionUtils.convertThrowableToBaseException(e);
		}

		return true;
	}

	private static ZipOutputStream newZipOutputStream(File targetFile)
			throws FileNotFoundException {

		FileOutputStream fileOutputStream = new FileOutputStream(targetFile);

		CheckedOutputStream cs = new CheckedOutputStream(fileOutputStream,
				new CRC32());

		return new ZipOutputStream(new BufferedOutputStream(cs));

	}

	public static boolean zip(final String[] zipFilePaths,
			final String zipStoreFullPathFile) {
		return zip(zipFilePaths, new File(zipStoreFullPathFile));
	}

	public static boolean zip(final String[] zipFilePaths,
			final File zipStoreFullPathFile) {

		File targetFile = zipStoreFullPathFile;

		FileUtils.makeDir(zipStoreFullPathFile);

		if (targetFile.isFile() && targetFile.exists()) {
			FileUtils.delete(targetFile);
		}

		if (TypeChecker.isEmpty(zipFilePaths)) {
			return false;
		}

		ZipOutputStream out = null;

		try {

			out = newZipOutputStream(zipStoreFullPathFile);

			for (String zipFilePath : zipFilePaths) {
				System.out.println(zipFilePath);
				if (!zip(zipFilePath, out, false, true)) {
					return false;
				}

			}

			return true;

		} catch (Exception e) {
			throw ExceptionUtils.convertThrowableToBaseException(e);
		} finally {
			try {
				out.close();
			} catch (Exception e) {

			}
		}
	}

	public static boolean zip(final String zipFilePath,
			final ZipOutputStream out, boolean closeStream,
			final boolean includedZipPackageName) {
		try {

			final File zipFilePathFile = new File(zipFilePath);

			if (!zipFilePathFile.exists()) {
				System.err.println("指定的要压缩的文件或目录不存在.");
				return false;
			}

			FileUtils.executeEachFile(zipFilePathFile, new Callable<File>() {

				public File call(File zipFile, Object... arguments) {
					InputStream in = null;
					try {

						String fileZipFile = StringUtils.replaceAll(zipFile
								.getAbsolutePath(), String
								.valueOf(File.separatorChar), "/");

						fileZipFile = fileZipFile.substring(zipFilePath
								.length());

						String filePath = "";

						if (zipFilePathFile.isDirectory()
								&& includedZipPackageName) {
							filePath = zipFilePathFile.getName();
						}

						ZipEntry e = new ZipEntry(FileUtils
								.getFullFilePathName(filePath, fileZipFile));

						out.putNextEntry(e);

						in = new FileInputStream(zipFile);

						FileUtils.copy(in, out, false);

						out.closeEntry();

					} catch (Exception e) {
						throw ExceptionUtils.convertThrowableToBaseException(e);
					} finally {
						FileUtils.close(in);
					}

					return zipFile;
				}

			}, zipFilePath);

			if (closeStream) {
				out.close();
			}

		} catch (Exception e) {
			throw ExceptionUtils.convertThrowableToBaseException(e);
		}

		return true;
	}

	/**
	 * 调用org.apache.tools.zip实现解压缩，支持目录嵌套和中文名
	 * 也可以使用java.util.zip不过如果是中文的话，解压缩的时候文件名字会是乱码。原因是解压缩软件的编码格式跟java.util.zip.ZipInputStream的编码字符集(固定是UTF-8)不同
	 * 
	 * @param zipFileName
	 *            要解压缩的文件
	 * @param outputDirectory
	 *            要解压到的目录
	 * @throws Exception
	 */

	public static boolean unZip(String zipFileName, String outputDirectory) {

		ZipFile zipFile = null;

		try {

			zipFile = new ZipFile(zipFileName);

			FileUtils.makeDir(outputDirectory);

			Enumeration<ZipEntry> zipEntryEnums = (Enumeration<ZipEntry>) zipFile
					.getEntries();

			for (ZipEntry zipEntry = null; zipEntryEnums.hasMoreElements();) {

				zipEntry = zipEntryEnums.nextElement();

				if (zipEntry.isDirectory()) {
					new File(outputDirectory, zipEntry.getName()).mkdirs();
					continue;
				}

				File zipFileEntry = new File(outputDirectory, zipEntry
						.getName());

				FileUtils.makeDir(zipFileEntry);

				InputStream in = zipFile.getInputStream(zipEntry);

				FileOutputStream out = new FileOutputStream(zipFileEntry);

				FileUtils.copy(in, out, true);

			}

			return true;

		} catch (Exception e) {
			throw ExceptionUtils.convertThrowableToBaseException(e);
		} finally {
			try {
				zipFile.close();
			} catch (Exception e) {

			}
		}

	}

	private AntZipUtils() {

	}

	public static void main(String[] args) {
		// System.out
		// .println(AntZipUtils
		// .zip(
		// new String[] {
		// "G:/Workspace(JohnHao)/ABIT/eMap_Workspace/Recruitment/webapp",
		// "G:/Workspace(JohnHao)/ABIT/eMap_Workspace/Recruitment/tomcat" },
		// "G:/Workspace(JohnHao)/ABIT/eMap_Workspace/Recruitment/test.zip"));
		//		
		AntZipUtils
				.unZip(
						"G:/Workspace(JohnHao)/ABIT/eMap_Workspace/Recruitment/test.zip",
						"G:/Workspace(JohnHao)/ABIT/eMap_Workspace/Recruitment/testZip");

	}
}
