package com.rapid.j2ee.framework.core.utils;

import java.lang.annotation.Annotation;
import java.lang.reflect.Array;
import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.springframework.util.AntPathMatcher;
import org.springframework.util.ReflectionUtils;

import com.rapid.j2ee.framework.core.exception.ExceptionUtils;
import com.rapid.j2ee.framework.core.reflect.MetaClass;
import com.rapid.j2ee.framework.core.reflect.invoker.Invoker;
import com.rapid.j2ee.framework.core.reflect.invoker.MethodInvoker;

public final class ClassUtils extends org.apache.commons.lang.ClassUtils {

	public static String getClassModifierName(Class clz) {

		String result = "";
		switch (clz.getModifiers()) {
		case Modifier.PRIVATE:
			result = "private";
			break;
		case Modifier.PUBLIC:
			result = "public";
			break;
		case Modifier.PROTECTED:
			result = "protected";
			break;
		case Modifier.ABSTRACT:
			result = "abstract";
			break;
		case Modifier.FINAL:
			result = "final";
			break;
		case Modifier.NATIVE:
			result = "native";
			break;
		case Modifier.STATIC:
			result = "static";
			break;
		case Modifier.SYNCHRONIZED:
			result = "synchronized";
			break;
		case Modifier.STRICT:
			result = "strict";
			break;
		case Modifier.TRANSIENT:
			result = "transient";
			break;
		case Modifier.VOLATILE:
			result = "volatile";
			break;
		case Modifier.INTERFACE:
			result = "interface";
			break;
		default:
			break;
		}
		return result;
	}

	private static final ThreadLocal<Map<Class, Field[]>> THREADLOCALE_FIELDARRAYS_MAPPER = new ThreadLocal<Map<Class, Field[]>>() {
		@Override
		protected Map<Class, Field[]> initialValue() {

			return new HashMap<Class, Field[]>();
		}
	};

	public static boolean hasAnnotationAtClass(Class target,
			Class annotationClass) {
		return target.isAnnotationPresent(annotationClass);
	}

	public static boolean hasAnnotationAtMethod(Method method,
			Class annotationClass) {
		return !TypeChecker.isNull(method.getAnnotation(annotationClass));
	}

	public static Field getFieldByIngoreCaseFieldName(Class clz, String name) {
		return ClassUtils.getFieldByIngoreCaseFieldName(clz, name, false);
	}

	public static Field getFieldByIngoreCaseFieldName(Class clz, String name,
			boolean ignoreUnderline) {

		return getFieldByIngoreCaseFieldName(
				getAllThreadLocalFieldsAsClass(clz), getFieldNameResolved(name,
						ignoreUnderline), ignoreUnderline);
	}

	public static String getFieldNameResolved(String name,
			boolean ignoreUnderline) {

		name = StringUtils.remove(name, "-");

		if (ignoreUnderline) {
			name = StringUtils.remove(name, "_");

		}

		return name;
	}

	public static Field getFieldByName(Class clz, String name) {

		try {
			return clz.getDeclaredField(name);
		} catch (Exception e) {
			return getFieldByIngoreCaseFieldName(clz, name);
		}

	}

	public static Field getFieldByIngoreCaseFieldName(Field[] fields,
			String name, boolean ignoreUnderline) {

		for (Field field : fields) {

			if (getFieldNameResolved(field.getName(), ignoreUnderline)
					.equalsIgnoreCase(name)) {
				return field;
			}
		}

		return null;
	}

	public static Field[] getAllThreadLocalFieldsAsClass(Class clz) {

		if (!THREADLOCALE_FIELDARRAYS_MAPPER.get().containsKey(clz)) {
			THREADLOCALE_FIELDARRAYS_MAPPER.get().put(clz,
					ClassUtils.getAllFieldsAsClass(clz));
		}

		return THREADLOCALE_FIELDARRAYS_MAPPER.get().get(clz);
	}

	public static Field[] getFieldsByAnnotation(Class clz, Class annotationClass) {

		Field[] fs = getAllFieldsAsClass(clz);

		List<Field> rs = new ArrayList<Field>(fs.length);

		for (Field f : fs) {
			if (TypeChecker.isNull(f.getAnnotation(annotationClass))) {
				continue;
			}

			rs.add(f);
		}

		return (Field[]) rs.toArray(new Field[rs.size()]);

	}

	public static Field[] getFieldsByAntPattern(Class clz, String pattern) {

		AntPathMatcher matcher = new AntPathMatcher();

		Field[] fs = getAllFieldsAsClass(clz);

		List<Field> rs = new ArrayList<Field>(fs.length);

		for (Field f : fs) {

			if (matcher.match(pattern, f.getName())) {
				rs.add(f);
			}

		}

		return (Field[]) rs.toArray(new Field[rs.size()]);

	}

	public static Field getFieldByAnnotation(Class clz, Class annotationClass) {

		Field[] fs = getAllFieldsAsClass(clz);

		for (Field f : fs) {
			if (TypeChecker.isNull(f.getAnnotation(annotationClass))) {
				continue;
			}

			return f;
		}

		return null;

	}

	public static Annotation[] getMethodParameterAnnotationsByAnnotation(
			Method method, Class annotationClass) {

		Annotation[] annos = new Annotation[method.getParameterTypes().length];

		Annotation[][] paramAnnos = method.getParameterAnnotations();

		for (int i = 0, j = paramAnnos.length; i < j; i++) {

			for (Annotation anno : paramAnnos[i]) {

				if (anno.annotationType().equals(annotationClass)) {
					annos[i] = anno;
				}
			}
		}

		return annos;
	}

	public static Field findFieldByType(Class clz, Class fieldType) {

		Field[] fs = getAllFieldsAsClass(clz);

		for (Field f : fs) {
			if (f.getType() == fieldType) {
				return f;
			}
		}

		return null;
	}

	public static List<Field> getFieldsByType(Class clz, Class fieldType) {

		Field[] fs = getAllFieldsAsClass(clz);

		List<Field> fields = new ArrayList<Field>(fs.length / 2);

		for (Field f : fs) {
			if (f.getType() == fieldType) {
				fields.add(f);
			}
		}

		return fields;
	}

	public static String getAllFieldsBunchStringAsClass(Class claz,
			String prefix, String suffix) {

		Field[] fs = getAllFieldsAsClass(claz);

		StringBuffer sb = new StringBuffer(fs.length * 25);

		for (Field f : fs) {
			sb.append(prefix + f.getName() + suffix);
			sb.append(",");
		}

		if (TypeChecker.isEmpty(sb)) {
			return "";
		}

		return sb.substring(0, sb.length() - 1);
	}

	@SuppressWarnings("unchecked")
	public static Field[] getAllFieldsAsClass(Class claz) {
		List fields = new ArrayList();
		do {
			fields.addAll(Arrays.asList(claz.getDeclaredFields()));
			claz = claz.getSuperclass();

		} while (claz != null && claz != Object.class);
		return (Field[]) fields.toArray(new Field[fields.size()]);
	}

	@SuppressWarnings("unchecked")
	public static Field[] getAllFieldsAsClassByAnnotation(Class claz,
			Class annotation) {
		Field[] fields = getAllFieldsAsClass(claz);

		List<Field> fieldArrays = new ArrayList<Field>(fields.length / 3 + 1);
		for (Field f : fields) {
			if (TypeChecker.isNull(f.getAnnotation(annotation))) {
				continue;
			}

			fieldArrays.add(f);
		}

		return fieldArrays.toArray(new Field[fieldArrays.size()]);

	}

	public static Method getMethodAsClassByAnnotation(Class claz,
			Class annotation) {
		Method[] methods = getAllMethodsAsClass(claz);

		for (Method method : methods) {
			if (!ClassUtils.hasAnnotationAtMethod(method, annotation)) {
				continue;
			}

			return method;

		}

		return null;

	}

	public static Method[] getAllMethodsAsClassByAnnotation(Class claz,
			Class annotation) {
		Method[] methods = getAllMethodsAsClass(claz);

		List<Method> methodArrays = new ArrayList<Method>(
				methods.length / 3 + 1);

		for (Method method : methods) {
			if (!ClassUtils.hasAnnotationAtMethod(method, annotation)) {
				continue;
			}

			methodArrays.add(method);

		}

		return methodArrays.toArray(new Method[methodArrays.size()]);

	}

	@SuppressWarnings("unchecked")
	public static Method[] getAllMethodsAsClass(Class claz) {
		List methods = new ArrayList();
		do {
			methods.addAll(Arrays.asList(claz.getDeclaredMethods()));
			claz = claz.getSuperclass();

		} while (claz != null && claz != Object.class);
		return (Method[]) methods.toArray(new Method[methods.size()]);
	}

	public static Method findSetterMethodAsClassByFieldName(Class claz,
			String fieldName) {

		return ReflectionUtils.findMethod(claz, "set"
				+ StringUtils.upperStartChar(fieldName));

	}

	public static boolean hasSetterMethodAsClassByFieldName(Class claz,
			String fieldName) {
		return !TypeChecker.isNull(findSetterMethodAsClassByFieldName(claz,
				fieldName));
	}

	public static boolean isImplementedClass(Class clz) {
		return !clz.isInterface() && !Modifier.isAbstract(clz.getModifiers());
	}

	public static String getPackageName(Class clz, int dotCounts) {

		String[] parts = StringUtils.split(clz.getName(), ".");

		int maxEnd = Math.min(parts.length, dotCounts);
		StringBuffer rePackageName = new StringBuffer(clz.getName().length());

		for (int i = 0; i < maxEnd; i++) {
			rePackageName.append(parts[i]);

			if (i != maxEnd - 1) {
				rePackageName.append(".");
			}
		}

		return rePackageName.toString();
	}

	@SuppressWarnings("unchecked")
	public static List<Class> getAllClasses(Class clz, Set<Class> excludedClz) {

		if (TypeChecker.isNull(excludedClz)) {
			excludedClz = Collections.EMPTY_SET;
		}

		List<Class> allClasses = new ArrayList<Class>(5);
		allClasses.add(clz);

		List<Class> superClasses = (List<Class>) ClassUtils
				.getAllSuperclasses(clz);

		for (Class superClz : superClasses) {

			if (superClz == Object.class || excludedClz.contains(superClz)) {
				continue;
			}

			allClasses.add(superClz);

		}

		return allClasses;

	}

	public static List<MethodInvoker> getGetInvokers(Class clz,
			Set<Class> excludedClasses) {

		List<Class> classes = getAllClasses(clz, excludedClasses);

		List<MethodInvoker> methodInvokers = new ArrayList<MethodInvoker>();

		for (Class clzItem : classes) {

			MetaClass meta = MetaClass.forClass(clzItem);

			for (String name : meta.getGetterNames()) {

				Invoker invoker = (Invoker) meta.getGetInvoker(name);

				if (invoker instanceof MethodInvoker) {

					MethodInvoker method = (MethodInvoker) invoker;

					if (!isTargetMethod(method, classes)) {
						continue;
					}
					// method
					methodInvokers.add(method);
				}

			}
		}

		return methodInvokers;

	}

	private static boolean isTargetMethod(MethodInvoker method,
			List<Class> classes) {

		for (Class clzItem : classes) {
			if (method.target(clzItem)) {
				return true;
			}
		}

		return false;
	}

	/**
	 * Replacement for <code>Class.forName()</code> that also returns Class
	 * instances for primitives (like "int") and array class names (like
	 * "String[]").
	 * <p>
	 * Always uses the thread context class loader.
	 * 
	 * @param name
	 *            the name of the Class
	 * @return Class instance for the supplied name
	 * @see java.lang.Class#forName(String, boolean, ClassLoader)
	 * @see java.lang.Thread#getContextClassLoader()
	 */
	public static Class forName(String name) {
		return forName(name, Thread.currentThread().getContextClassLoader());
	}

	/**
	 * Replacement for <code>Class.forName()</code> that also returns Class
	 * instances for primitives (like "int") and array class names (like
	 * "String[]").
	 * 
	 * @param name
	 *            the name of the Class
	 * @param classLoader
	 *            the class loader to use
	 * @return Class instance for the supplied name
	 * @see java.lang.Class#forName(String, boolean, ClassLoader)
	 * @see java.lang.Thread#getContextClassLoader()
	 */
	public static Class forName(String name, ClassLoader classLoader) {
		try {
			Class clazz = resolvePrimitiveClassName(name);
			if (clazz != null) {
				return clazz;
			}
			if (name.endsWith(ARRAY_SUFFIX)) {
				// special handling for array class names
				String elementClassName = name.substring(0, name.length()
						- ARRAY_SUFFIX.length());
				Class elementClass = ClassUtils.forName(elementClassName,
						classLoader);
				return Array.newInstance(elementClass, 0).getClass();
			}

			return Class.forName(name, true, classLoader);
		} catch (ClassNotFoundException e) {
			throw ExceptionUtils.convertThrowableToBaseException(e);
		}
	}

	public static String getMethodNameWithoutGetterSetter(String methodName) {
		String methodNewName = methodName;
		if (methodName.startsWith("get") || methodName.startsWith("set")) {
			methodNewName = methodName.substring(3);
		}

		return methodNewName.substring(0, 1).toLowerCase()
				+ methodNewName.substring(1);
	}

	/**
	 * Resolve the given class name as primitive class, if appropriate.
	 * 
	 * @param name
	 *            the name of the potentially primitive class
	 * @return the primitive class, or <code>null</code> if the name does not
	 *         denote a primitive class
	 */
	public static Class resolvePrimitiveClassName(String name) {
		// Most class names will be quite long, considering that they
		// SHOULD sit in a package, so a length check is worthwhile.
		if (name.length() <= 8) {
			// could be a primitive - likely
			for (int i = 0; i < PRIMITIVE_CLASSES.length; i++) {
				Class clazz = PRIMITIVE_CLASSES[i];
				if (clazz.getName().equals(name)) {
					return clazz;
				}
			}
		}
		return null;
	}

	public static String getClassToPath(Class target, String subfix) {

		return StringUtils.replaceAll(target.getPackage().getName() + ".", ".",
				"/")
				+ ClassUtils.getShortClassName(target) + subfix;
	}

	/** Suffix for array class names */
	public static final String ARRAY_SUFFIX = "[]";

	/** All primitive classes */
	private static Class[] PRIMITIVE_CLASSES = { boolean.class, byte.class,
			char.class, short.class, int.class, long.class, float.class,
			double.class };

	public static ClassLoader getClassLoader() {
		return Thread.currentThread().getContextClassLoader();
	}

	public static String getFieldCollectionItemClassName(Field field) {

		return StringUtils.splitClosure(field.getGenericType().toString(), "<",
				">")[0];
	}

	private ClassUtils() {

	}

	public static void main(String[] args) {
		System.out.println(ClassUtils.getClassToPath(ClassUtils.class,
				"_modeldriven.properties"));
	}

}
