package com.rapid.j2ee.framework.core.utils;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.OutputStream;
import java.io.RandomAccessFile;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;

import com.rapid.j2ee.framework.core.exception.ExceptionUtils;
import com.rapid.j2ee.framework.core.utils.support.FileAntPatternFilter;

public class FileSplitUtils {

	public static void splitBySize(String file, int blockSize) {
		try {

			blockSize = blockSize * 1024;

			long length = FileUtils.getFileSize(new File(file));

			int count = (length % blockSize == 0) ? (int) (length / blockSize)
					: (int) (length / blockSize) + 1;

			long offset = 0L;

			for (int i = 0; i < count - 1; i++) {
				offset = write(file, i, offset, blockSize);
			}
			if (length - offset > 0) {
				// 将剩余的都写入最后一份
				write(file, count - 1, offset, (int) (length - offset));
			}
		} catch (Exception e) {
			throw ExceptionUtils.convertThrowableToBaseException(e);
		}
	}

	/**
	 * <p>
	 * 拆分文件
	 * </p>
	 * 
	 * @param file
	 *            源文件
	 * @param count
	 *            拆分的文件个数
	 * @throws Exception
	 */
	public static void splitByCount(String file, int count) {
		try {
			long length = FileUtils.getFileSize(new File(file));

			int filePartlySize = (int) (length / count);

			long offset = 0L;

			for (int i = 0; i < count - 1; i++) {
				offset = write(file, i, offset, filePartlySize);
			}
			if (length - offset > 0) {
				// 将剩余的都写入最后一份
				write(file, count - 1, offset, (int) (length - offset));
			}
		} catch (Exception e) {
			throw ExceptionUtils.convertThrowableToBaseException(e);
		}
	}

	/**
	 * <p>
	 * 指定每份文件的范围写入不同文件
	 * </p>
	 * 
	 * @param file
	 *            源文件
	 * @param index
	 *            文件顺序标识
	 * @param begin
	 *            开始指针位置
	 * @param end
	 *            结束指针位置
	 * @return
	 * @throws Exception
	 */
	private static long write(String file, int index, long begin, int blockSize)
			throws Exception {

		RandomAccessFile readFile = null;
		RandomAccessFile writeFile = null;

		try {

			readFile = new RandomAccessFile(new File(file), "r");

			FileUtils.delete(new File(file + "_" + index + ".tmp")
					.getAbsolutePath());

			writeFile = new RandomAccessFile(new File(file + "_" + index
					+ ".tmp"), "rw");

			byte[] bufferBytes = new byte[blockSize];

			int indexPosition = 0;
			// 从指定位置读取
			readFile.seek(begin);

			if ((indexPosition = readFile.read(bufferBytes)) != -1) {
				writeFile.write(bufferBytes, 0, indexPosition);
			}

			return readFile.getFilePointer();

		} finally {

			try {
				readFile.close();
			} catch (Exception e) {

			}

			try {
				writeFile.close();
			} catch (Exception e) {

			}
		}
	}

	/**
	 * <p>
	 * 合并文件
	 * </p>
	 * 
	 * @param file
	 *            指定合并后的文件
	 * @param tempFiles
	 *            分割前的文件名
	 * @param tempCount
	 *            文件个数
	 * @throws Exception
	 */
	public static void merge(File mergedTargetFile, List<File> files) {

		OutputStream fileOutputStream = null;

		try {
			fileOutputStream = new FileOutputStream(mergedTargetFile);
			for (File splitedFile : files) {
				FileUtils.copy(new FileInputStream(splitedFile),
						fileOutputStream, false);
			}
		} catch (Exception e) {
			throw ExceptionUtils.convertThrowableToBaseException(e);
		} finally {
			try {
				fileOutputStream.close();
			} catch (Exception e) {

			}
		}

	}

	public static void mergeFilesWithStartFileName(File mergedTargetFile,
			File orignalFile) {

		List<File> orginalSplitedFiles = FileUtils.listFiles(orignalFile
				.getParentFile(), new FileAntPatternFilter(orignalFile
				.getName()
				+ "_*"));

		if (TypeChecker.isEmpty(orginalSplitedFiles)) {
			return;
		}

		Collections.sort(orginalSplitedFiles, new Comparator<File>() {

			public int compare(File file1, File file2) {

				return NumberUtils.parseInt(StringUtils.remove(StringUtils
						.substringAfter(file1.getName(), "_"), ".tmp"))
						-

						NumberUtils.parseInt(StringUtils.remove(StringUtils
								.substringAfter(file2.getName(), "_"), ".tmp"));

			}

		});

		merge(mergedTargetFile, orginalSplitedFiles);

		deleteSplitedFiles(orginalSplitedFiles);
	}

	private static void deleteSplitedFiles(List<File> orginalSplitedFiles) {

		for (File file : orginalSplitedFiles) {
			try {

				file.delete();
			} catch (Exception exp) {
				return;
			}
		}
	}

}
