package com.rapid.j2ee.framework.core.utils;

import java.io.BufferedInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileFilter;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.RandomAccessFile;
import java.util.ArrayList;
import java.util.List;

import org.springframework.util.Assert;

import com.rapid.j2ee.framework.core.exception.ExceptionUtils;
import com.rapid.j2ee.framework.core.utils.support.Callable;

public final class FileUtils {

	public long getFileSizes(File f) {

		try {
			if (!f.exists()) {
				return 0;
			}
			FileInputStream fis = null;
			fis = new FileInputStream(f);
			return fis.available();
		} catch (Exception e) {
			return 0;
		}
	}

	public static boolean makeDir(String file) {
		return makeDir(new File(file));
	}

	public static boolean makeDir(File file) {

		if (file == null) {
			return false;
		}

		if (file.getName().contains(".")) {

			if (file.getParentFile() != null) {
				return file.getParentFile().mkdirs();
			}

			return false;
		}

		return file.mkdirs();
	}

	public static File findFileWithParents(String path, String name) {

		if (TypeChecker.isEmpty(path) || TypeChecker.isEmpty(name)) {
			return null;
		}

		File temp = new File(path, name);

		while (!temp.exists() || !temp.isFile()) {

			if (temp.getParentFile() == null
					|| temp.getParentFile().getParentFile() == null) {
				temp = null;
				break;
			}

			temp = new File(temp.getParentFile().getParentFile(), name);
		}

		return temp;

	}

	/**
	 * Returns an input stream for reading the specified resource.
	 * 
	 * @param url
	 * @param classLoader
	 * @return
	 * @throws FileNotFoundException
	 */
	public static InputStream getResourceAsStream(String url, Class clazz)
			throws FileNotFoundException {

		File file = new File(url);

		if (!file.exists()) {

			InputStream is = null;

			if (clazz == null) {

				is = Thread.currentThread().getContextClassLoader()
						.getResourceAsStream(url);

				if (!TypeChecker.isNull(is)) {
					return is;
				}

				throw new FileNotFoundException(url + " cannot be found under "
						+ Thread.currentThread().getContextClassLoader());
			}

			if (!TypeChecker.isNull(clazz.getClassLoader())) {
				is = clazz.getClassLoader().getResourceAsStream(url);
			}

			if (TypeChecker.isNull(is)) {
				is = clazz.getResourceAsStream(url);
			}

			if (!TypeChecker.isNull(is)) {
				return is;
			}

			throw new FileNotFoundException(url + " cannot be found under "
					+ clazz);

		}
		return new FileInputStream(file);
	}

	public static InputStream getResourceAsStream(String url)
			throws FileNotFoundException {
		return getResourceAsStream(url, null);
	}

	public static ByteArrayOutputStream getByteArrayOutputStream(String filename)
			throws IOException {
		ByteArrayOutputStream baos = new ByteArrayOutputStream();
		InputStream fis = getResourceAsStream(filename);
		BufferedInputStream bis = new BufferedInputStream(fis);

		byte[] bytes = new byte[4 * 1024];
		while (bis.read(bytes) > 0) {
			baos.write(bytes);
			bytes = new byte[4 * 1024];
		}
		bis.close();
		fis.close();
		return baos;
	}

	public static List<File> listFiles(File file, FileFilter filter) {

		List<File> files = new ArrayList<File>();

		if (file.isFile()) {

			files.add(file);

			return files;
		}

		File[] fileArrays = null;

		if (filter != null) {
			fileArrays = file.listFiles(filter);
		} else {
			fileArrays = file.listFiles();
		}

		if (TypeChecker.isEmpty(fileArrays)) {
			return files;
		}

		for (int i = 0; i < fileArrays.length; i++) {
			files.addAll(listFiles(fileArrays[i], filter));
		}

		return files;

	}

	public static void executeEachFile(File file, Callable<File> call,
			Object... args) {

		if (file.isFile()) {

			call.call(file, args);

			return;
		}

		File[] fileArrays = file.listFiles();

		if (TypeChecker.isEmpty(fileArrays)) {

			return;
		}

		for (int i = 0; i < fileArrays.length; i++) {
			executeEachFile(fileArrays[i], call, args);
		}

	}

	public static List<File> listFiles(File file) {

		return listFiles(file, null);

	}

	public static File[] listSimpleFiles(File file) {

		if (!file.isDirectory() || !file.exists()) {
			return ObjectUtils.EMPTY_FILE_ARRAYS;
		}

		File[] files = file.listFiles();

		return TypeChecker.isNull(files) ? ObjectUtils.EMPTY_FILE_ARRAYS
				: files;
	}

	public static void delete(String file) {
		delete(new File(file));
	}

	public static void delete(File file) {

		if (file == null) {
			return;
		}

		if (TypeChecker.isEmpty(file.listFiles())) {
			file.delete();
			return;
		}

		File[] files = file.listFiles();

		for (int i = 0, j = files.length; i < j; i++) {

			delete(files[i].getAbsoluteFile());

		}
		file.delete();
	}

	public static void renameFiles(File file, FileFilter filter, String targExt) {
		List files = listFiles(file, filter);

		for (int i = 0, j = files.size(); i < j; i++) {

			File orgFile = (File) files.get(i);

			String fileName = orgFile.toString();

			if (fileName.toLowerCase().lastIndexOf(".") >= 0) {
				fileName = fileName.substring(0, fileName.toLowerCase()
						.lastIndexOf("."))
						+ "." + targExt;
				orgFile.renameTo(new File(fileName));
			}

		}
	}

	public static File getFile(String filePath, String fileName) {
		return new File(filePath, fileName);
	}

	public static String getFullFilePathName(String filePath, String fileName) {

		filePath = StringUtils.trimToEmpty(filePath);
		fileName = StringUtils.trimToEmpty(fileName);

		String fileFullPath = filePath + fileName;

		int fileSeparatorCount = 0;

		if (filePath.endsWith("/") || filePath.endsWith("\\")) {
			fileSeparatorCount++;
		}

		if (fileName.startsWith("/") || filePath.startsWith("\\")) {
			fileSeparatorCount++;
		}

		if (fileSeparatorCount == 0) {
			if (!TypeChecker.isEmpty(filePath)
					&& !TypeChecker.isEmpty(fileName)) {
				fileFullPath = filePath + "/" + fileName;
			}
		}

		if (fileSeparatorCount == 2) {
			if (!TypeChecker.isEmpty(filePath)
					&& !TypeChecker.isEmpty(fileName)) {
				fileName = fileName.substring(1);
				fileFullPath = filePath + "/" + fileName;
			}
		}

		return fileFullPath;
	}

	public static boolean isFile(String filePath, String fileName) {
		try {

			File file = new File(getFullFilePathName(filePath, fileName));
			return file.exists() && file.isFile();
		} catch (Exception e) {
			return false;
		}
	}

	public static String[] getFileNameAndExtension(String fileName) {
		fileName = StringUtils.trimToEmpty(fileName);
		if (TypeChecker.isEmpty(fileName)) {
			return new String[] { "", "" };
		}
		int index = fileName.lastIndexOf(".");

		if (index < 0) {
			return new String[] { fileName, "" };
		}

		return new String[] { fileName.substring(0, index),
				fileName.substring(index + 1) };
	}

	public static long getFileSize(File file) {
		try {
			if (TypeChecker.isNull(file) || !file.exists() || !file.isFile()) {
				return 0;
			}
			RandomAccessFile randomAccessFile = new RandomAccessFile(file, "r");

			long length = randomAccessFile.length();
			randomAccessFile.close();

			return length;
		} catch (Exception e) {
			return 0;
		}
	}

	public static File getUploadFile(String dir, String fileName) {
		return new File(dir + File.separator + fileName);
	}

	public static long getUploadFileSize(String dir, String fileName) {
		return getFileSize(getUploadFile(dir, fileName));
	}

	public static String[] getViewFileSizeUnit(File file) {

		return getViewFileSizeUnit(getFileSize(file));

	}

	public static String[] getViewFileSizeUnit(long fileSize) {
		double size = fileSize * 1.0;
		String unit = "bytes";
		if (size < FILE_KB_SIZE) {
			// do nothing...
		} else if (size >= FILE_KB_SIZE && size < FILE_MB_SIZE) {
			size = size * 1.0 / FILE_KB_SIZE;
			unit = "KB";
		} else if (size >= FILE_MB_SIZE) {
			size = size * 1.0 / FILE_MB_SIZE;
			unit = "MB";
		}

		return new String[] { NumberUtils.formatCurrency(size), unit };

	}

	public static String renameFileExtension(String orginalFileName,
			String targetFileName) {
		return StringUtils.replace(orginalFileName, "."
				+ FileUtils.getFileNameAndExtension(orginalFileName)[1], "."
				+ FileUtils.getFileNameAndExtension(targetFileName)[1]);
	}

	public static void mergeMultipleResource(OutputStream os,
			InputStream... inputStreams) {

		try {
			for (InputStream is : inputStreams) {
				copy(is, os, false);
			}
		} finally {

			try {
				os.close();
			} catch (Exception e) {

			}
		}

	}

	/**
	 * Copy the contents of the given InputStream to the given OutputStream.
	 * Closes both streams when done.
	 * 
	 * @param in
	 *            the stream to copy from
	 * @param out
	 *            the stream to copy to
	 * @return the number of bytes copied
	 * @throws IOException
	 *             in case of I/O errors
	 */
	public static int copy(InputStream in, OutputStream out,
			boolean closedOutputStream) {
		Assert.notNull(in, "No InputStream specified");
		Assert.notNull(out, "No OutputStream specified");
		try {
			int byteCount = 0;
			byte[] buffer = new byte[BUFFER_SIZE];
			int bytesRead = -1;
			while ((bytesRead = in.read(buffer)) != -1) {
				out.write(buffer, 0, bytesRead);
				byteCount += bytesRead;
			}
			out.flush();
			return byteCount;
		} catch (Exception ep) {
			throw ExceptionUtils.convertThrowableToBaseException(ep);
		} finally {
			try {
				in.close();
			} catch (IOException ex) {
			}
			try {

				if (closedOutputStream) {

					out.close();
				}
			} catch (IOException ex) {
			}
		}
	}

	public static String getFullFilePathsName(String fileName,
			String... filePaths) {

		if (TypeChecker.isEmpty(filePaths)) {
			return "";
		}

		return getFullFilePaths(filePaths) + fileName;
	}

	public static String getFullFilePaths(String... filePaths) {

		if (TypeChecker.isEmpty(filePaths)) {
			return "";
		}

		StringBuffer filePath = new StringBuffer(filePaths.length * 50);

		for (String path : filePaths) {

			if (TypeChecker.isEmpty(path)) {
				continue;
			}

			if (!path.endsWith("/") && !path.endsWith("\\")) {
				filePath.append(path);
				filePath.append("/");
			} else {
				filePath.append(StringUtils.replace(path, "\\", "/"));
			}
		}

		return filePath.toString();
	}

	public static File newTempFile(String prefix, String suffix) {

		try {
			return File.createTempFile(prefix + System.currentTimeMillis(),
					suffix);
		} catch (IOException e) {
			return new File(prefix + System.currentTimeMillis() + suffix);
		}
	}

	public static File newTempDataFile() {

		return newTempFile("temp", ".data");

	}

	public static boolean close(InputStream is) {
		try {
			is.close();
		} catch (Exception e) {
			return false;
		}

		return true;
	}

	public static boolean close(OutputStream os) {
		boolean flag = true;
		try {
			os.flush();
		} catch (Exception e) {
			flag = false;
		}

		try {
			os.close();
		} catch (Exception e) {
			flag = false;
		}

		return flag;
	}

	public static boolean flush(OutputStream os) {
		boolean flag = true;
		try {
			os.flush();
		} catch (Exception e) {
			flag = false;
		}

		return flag;
	}

	private FileUtils() {

	}

	public static long FILE_KB_SIZE = 1024;

	public static long FILE_MB_SIZE = FILE_KB_SIZE * 1024;

	public static long FILE_GB_SIZE = FILE_MB_SIZE * 1024;

	public static final int BUFFER_SIZE = 1024 * 2;

	public static void main(String[] args) {
		System.out.println(newTempDataFile());
	}

}