package com.rapid.j2ee.framework.core.utils;

import java.lang.reflect.AccessibleObject;
import java.lang.reflect.Field;

/**
 * @author John 2005-03-30
 * @description ObjectAnalyzer Class provides convenient way to debug a certain
 *              bean object. To analyze a bean object is to display all info of
 *              its fields and values on the console.
 */
public class ObjectAnalyzer {
	public static final String SEPARATOR = " ";

	public static String toString(Object obj) {
		try {
			if (TypeChecker.isNull(obj)) {
				return "NULL";
			}

			if (obj instanceof String) {
				return String.valueOf(obj);
			}

			StringBuffer sb = new StringBuffer(50);
			// add Object header info to Buffer Cache
			addHeaderInfoToBuffer(sb, obj);
			// add Object field name and value to Buffer Cache
			addFieldsAndValuesToBuffer(sb, obj.getClass(), obj);
			// add Object Bottom info to Buffer Cache
			addBottomInfo(sb);
			return sb.toString();
		} catch (Exception e) {
			return String.valueOf(obj);
		}

	}

	private static void addHeaderInfoToBuffer(StringBuffer sb, Object obj) {
		Class clz = obj.getClass();
		String clzName = clz.getName(); // a whole class name including package.
		sb.append(getClzShortName(clzName));

		Class superClz = clz.getSuperclass();
		String superClzName = superClz.getName();

		if (isObjectTypeOfSuperClz(superClzName)) {
			// superClz is Object type
			sb.append(" {");
		} else {
			sb.append(" extends ");
			sb.append(getClzShortName(superClzName));
			sb.append(" {");

			// append super clz fields and values into Buffer Cache
			addFieldsAndValuesToBuffer(sb, superClz, obj);
		}

	}

	private static void addFieldsAndValuesToBuffer(StringBuffer sb, Class clz,
			Object obj) {
		// getDeclaredFields include private instance fields
		Field[] fields = clz.getDeclaredFields();
		try {
			AccessibleObject.setAccessible(fields, true);
		} catch (Exception e) {
			// ignore
		}

		Field f;
		for (int i = 0; i < fields.length; i++) {
			f = fields[i];
			sb.append(SEPARATOR);
			sb.append("		" + f.getName() + " = '");
			sb.append(reflectValue(f, obj));
			sb.append("'");
		}

	}

	private static void addBottomInfo(StringBuffer sb) {
		sb.append(SEPARATOR);
		sb.append("}");
	}

	private static boolean isObjectTypeOfSuperClz(String superClzName) {
		return Object.class.getName().equals(superClzName);
	}

	private static String getClzShortName(String longName) {
		return longName.substring(longName.lastIndexOf(".") + 1);
	}

	// reflect value through its field
	private static Object reflectValue(Field f, Object obj) {
		try {
			Object value = f.get(obj);

			if (TypeChecker.isNull(value)) {
				return "Null";
			} else {
				return value;
			}
		} catch (Exception e) {

		}

		return "???";
	}

}
