package com.rapid.j2ee.framework.core.utils;

import java.util.Vector;

import org.htmlparser.Attribute;
import org.htmlparser.Node;
import org.htmlparser.lexer.Lexer;
import org.htmlparser.nodes.RemarkNode;
import org.htmlparser.nodes.TagNode;

public final class SafeHtmlUtils {

	/**
	 * 处理有害的代码后返回（把有害的代码变成普通的文本）
	 * 
	 * @param html
	 * @return
	 */
	public static String getSafeHtml(String html) {
		try {

			if (TypeChecker.isNull(html)) {
				return "";
			}

			StringBuffer result = new StringBuffer(html.length() + 10);

			// 遍历所有元素
			Lexer lexer = new Lexer(html);

			for (Node node = null; (node = lexer.nextNode()) != null;) {

				result.append(getSaftHtmlByNode(node));
			}

			return StringUtils.trimToEmpty(result.toString());

		} catch (Exception e) {

			return html;
		}
	}

	private static String getSaftHtmlByNode(Node node) {

		// 节点有三种：TagNode（元素）、TextNode（文本）、RemarkNode（注释）

		if (node instanceof RemarkNode) {
			// 注释应替换
			return toSafeHtml(node.toHtml());
		}

		if (!(node instanceof TagNode)) {
			// Node是其他元素，不处理
			return node.toHtml();
		}

		// Node是元素
		TagNode tagNode = (TagNode) node;

		if (isSafeHtmlCompulsoryByTagNodes(tagNode)) {
			return toSafeHtml(node.toHtml());
		}

		return getSaftHtmlByAttributeNodes(tagNode);
	}

	/**
	 * 需要遍历查看每一个元素中的所有属性是否安全
	 * 
	 * @param tagNode
	 * @return
	 */
	private static String getSaftHtmlByAttributeNodes(TagNode tagNode) {

		// Copy Attributes 是为了防止数据循环冲突，当删除Attribute时。
		Vector<Attribute> attrList = new Vector<Attribute>(tagNode
				.getAttributesEx());

		for (Attribute attr : attrList) {

			if (!isJavascriptEventByAttribute(attr)) {
				continue;
			}

			tagNode.removeAttribute(attr.getName());

		}

		return tagNode.toHtml();

	}

	private static boolean isJavascriptEventByAttribute(Attribute attr) {

		String attrName = StringUtils.trimToEmpty(attr.getName()).toLowerCase();
		// onclick, onchange etc
		return attrName.toLowerCase().startsWith("on");
	}

	private static boolean isSafeHtmlCompulsoryByTagNodes(TagNode tagNode) {

		// 获取元素名，全是大写字母
		String name = tagNode.getTagName();

		return "SCRIPT".equals(name) || "STYLE".equals(name)
				|| "META".equals(name);

	}

	private static String toSafeHtml(String html) {

		if (TypeChecker.isEmpty(html)) {
			return html;
		}

		return html.replaceAll("<", "&lt;").replaceAll(">", "&gt;");
	}

	public static void main(String[] args) {
		String html = "text<>dd";
		System.out.println(SafeHtmlUtils.getSafeHtml(html));

	}

	private SafeHtmlUtils() {

	}
}
