package com.rapid.j2ee.framework.dispatcher;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.springframework.beans.BeansException;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.scheduling.concurrent.ThreadPoolTaskExecutor;
import org.springframework.util.Assert;

import com.rapid.j2ee.framework.core.exception.ExceptionUtils;
import com.rapid.j2ee.framework.core.utils.TypeChecker;
import com.rapid.j2ee.framework.dispatcher.interceptor.RequestDispatchInterceptor;
import com.rapid.j2ee.framework.dispatcher.listener.RequestDispatchListener;
import com.rapid.j2ee.framework.dispatcher.lookup.DispatchConfigurerLookup;
import com.rapid.j2ee.framework.dispatcher.resolve.RequestParameterResovler;

/**
 * 
 * HttpRequestDispachWebClientService Request URL 转发服务组件
 * 
 * @author John Hao
 * 
 */
public class RequestDispatchClient implements ApplicationContextAware,
		InitializingBean {

	/**
	 * Http Request 转发功能
	 * 
	 * @param HttpServletRequest
	 * @param HttpServletResponse
	 * @param HttpServletRequest
	 *            Request Parameters
	 * @param RequestParameterResovler --
	 *            进行Parameter的修改
	 */
	public void dispatch(HttpServletRequest request,
			HttpServletResponse response,
			Map<String, String[]> requestParameters) {

		RequestDispatchContext requestDispatchContext = new RequestDispatchContext();

		requestDispatchContext
				.setRequestDispatchedParameters(requestParameters);

		try {

			// Before 拦截器
			if (!this.before(requestDispatchContext, request, response,
					requestParameters)) {
				return;
			}

			// 获得HttpRequest 转发URL Mapper配置信息
			RequestNetClient httpRequestNetClient = new RequestNetClient(
					dispatchConfigurerLookup.lookup(request, requestParameters),
					requestDispatchContext);

			httpRequestNetClient
					.addHttpRequestParameterResovler(httpRequestParameterResovler);

			httpRequestNetClient
					.addRequestDispatchListeners(requestDispatchListeners);

			// 执行转发，并且保存转发内容
			httpRequestNetClient.dispatch(request, requestParameters, response);

		} catch (Throwable exp) {

			requestDispatchContext.setError(ExceptionUtils
					.printThrowableStack(exp));

			throw ExceptionUtils.convertThrowableToBaseException(exp);

		} finally {

			requestDispatchContext.end();

			// Complete 拦截器
			this.complete(requestDispatchContext, request, response,
					requestParameters);
		}

	}

	protected boolean before(
			RequestDispatchContext httpRequestDispatchInterceptionResult,
			HttpServletRequest request, HttpServletResponse response,
			Map<String, String[]> requestParameters) {

		boolean result = true;

		for (RequestDispatchInterceptor requestDispatchInterceptor : this.requestDispatchInterceptors) {
			if (!requestDispatchInterceptor.before(
					httpRequestDispatchInterceptionResult, request, response,
					requestParameters)) {
				result = false;
			}
		}

		return result;
	}

	private void complete(
			final RequestDispatchContext httpRequestDispatchInterceptionResult,
			final HttpServletRequest request,
			final HttpServletResponse response,
			final Map<String, String[]> requestParameters) {

		if (TypeChecker.isEmpty(requestDispatchInterceptors)) {
			return;
		}

		this.getThreadPoolTaskExecutor().execute(new Runnable() {

			public void run() {

				for (RequestDispatchInterceptor httpRequestDispatchInterceptor : requestDispatchInterceptors) {
					httpRequestDispatchInterceptor.complete(
							httpRequestDispatchInterceptionResult, request,
							response, requestParameters);
				}

			}

		});

	}

	private ThreadPoolTaskExecutor getThreadPoolTaskExecutor() {

		if (TypeChecker.isNull(threadPoolTaskExecutor)) {

			this.threadPoolTaskExecutor = applicationContext
					.getBean(ThreadPoolTaskExecutor.class);

		}

		return this.threadPoolTaskExecutor;
	}

	public void setApplicationContext(ApplicationContext applicationContext)
			throws BeansException {

		this.applicationContext = applicationContext;
	}

	public void setHttpRequestParameterResovler(
			RequestParameterResovler httpRequestParameterResovler) {
		this.httpRequestParameterResovler = httpRequestParameterResovler;
	}

	public RequestDispatchClient() {

	}

	public void afterPropertiesSet() throws Exception {

		Assert.notNull(dispatchConfigurerLookup,
				"Please provide a DispatchConfigurerLookup ");
	}

	public void setDispatchConfigurerLookup(
			DispatchConfigurerLookup dispatchConfigurerLookup) {
		this.dispatchConfigurerLookup = dispatchConfigurerLookup;
	}

	private ApplicationContext applicationContext;

	// 提供URL Mapper的配置
	private DispatchConfigurerLookup dispatchConfigurerLookup;

	@Autowired(required = false)
	private List<RequestDispatchInterceptor> requestDispatchInterceptors = new ArrayList<RequestDispatchInterceptor>();

	@Autowired(required = false)
	private List<RequestDispatchListener> requestDispatchListeners = new ArrayList<RequestDispatchListener>();

	private RequestParameterResovler httpRequestParameterResovler = RequestParameterResovler.Default_Parameter_Resolver;

	private ThreadPoolTaskExecutor threadPoolTaskExecutor;
}
