package com.rapid.j2ee.framework.dispatcher;

import java.io.Writer;
import java.nio.charset.Charset;
import java.util.List;
import java.util.Map;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.http.Header;
import org.apache.http.util.EntityUtils;
import org.springframework.util.FileCopyUtils;

import com.rapid.j2ee.framework.core.exception.ExceptionUtils;
import com.rapid.j2ee.framework.core.io.http.HttpPostClient;
import com.rapid.j2ee.framework.core.io.http.HttpResponseEntity;
import com.rapid.j2ee.framework.core.io.xml.XmlWriter;
import com.rapid.j2ee.framework.core.utils.CollectionsUtil;
import com.rapid.j2ee.framework.core.utils.NumberUtils;
import com.rapid.j2ee.framework.core.utils.ObjectUtils;
import com.rapid.j2ee.framework.core.utils.StringUtils;
import com.rapid.j2ee.framework.core.utils.TypeChecker;
import com.rapid.j2ee.framework.dispatcher.listener.RequestDispatchListener;
import com.rapid.j2ee.framework.dispatcher.lookup.DispatchConfigurer;
import com.rapid.j2ee.framework.dispatcher.resolve.RequestParameterResovler;

public class RequestNetClient {

	private static final Log LOGGER = LogFactory
			.getLog(RequestNetClient.class);

	private DispatchConfigurer dispatchConfigurer;

	private RequestParameterResovler httpRequestParameterResovler = RequestParameterResovler.Default_Parameter_Resolver;

	private RequestDispatchContext requestDispatchContext;

	private List<RequestDispatchListener> requestDispatchListeners = ObjectUtils.EMPTY_LIST;

	public RequestNetClient(DispatchConfigurer dispatchConfigurer,
			RequestDispatchContext requestDispatchContext) {
		this.dispatchConfigurer = dispatchConfigurer;
		this.requestDispatchContext = requestDispatchContext;
		requestDispatchContext.setDispatcherConfigurer(dispatchConfigurer);

	}

	public void addRequestDispatchListeners(
			List<RequestDispatchListener> requestDispatchListeners) {
		this.requestDispatchListeners = requestDispatchListeners;
	}

	public void addHttpRequestParameterResovler(
			RequestParameterResovler httpRequestParameterResovler) {
		this.httpRequestParameterResovler = httpRequestParameterResovler;
	}

	public void dispatch(HttpServletRequest request,
			Map<String, String[]> requestParameters,
			HttpServletResponse response) {

		this.requestDispatchContext.setDispatchedRequestContent(this.doDispatchExecute(
				request, requestParameters, response));
	}

	private String doDispatchExecute(HttpServletRequest request,
			Map<String, String[]> requestParameters,
			HttpServletResponse response) {

		try {

			HttpResponseEntity httpResponseEntity = this
					.buildHttpClientConnection(request, requestParameters);

			prepareResponseStatusAndHeaders(httpResponseEntity, response,
					requestParameters);

			if (dispatchConfigurer.isResourceDownloaded()) {

				return writeStreamToResponse(request, httpResponseEntity,
						response);

			}

			return this.captureRequestContentBeforeDispatch(request,
					requestParameters, httpResponseEntity, response);

		} catch (Exception e) {

			LOGGER.error("The request dispatcher execute error! Url:"
					+ dispatchConfigurer.getUrl(), e);

			throw ExceptionUtils.convertThrowableToBaseException(e);
		}

	}

	private String captureRequestContentBeforeDispatch(
			HttpServletRequest request,
			Map<String, String[]> requestParameters,
			HttpResponseEntity httpResponseEntity, HttpServletResponse response)
			throws Exception {

		String dispatchedResponseContent = EntityUtils.toString(
				httpResponseEntity, Charset.forName(dispatchConfigurer
						.getCharset()));

		this.executeRequestDispatchListener(request, requestParameters, this
				.matchesHttpRequestDispatchListeners(),
				dispatchedResponseContent);

		Writer writer = response.getWriter();

		writer.write(dispatchedResponseContent);

		writer.flush();

		writer.close();

		return dispatchedResponseContent;

	}

	private void executeRequestDispatchListener(HttpServletRequest request,
			Map<String, String[]> requestParameters,
			List<RequestDispatchListener> listeners, String content) {

		for (RequestDispatchListener listener : listeners) {
			listener.execute(request, requestParameters, dispatchConfigurer,
					content);
		}
	}

	private List<RequestDispatchListener> matchesHttpRequestDispatchListeners() {

		String key = this.dispatchConfigurer.getSystemAppName() + "_"
				+ dispatchConfigurer.getFunction() + "_"
				+ StringUtils.trimToEmpty(dispatchConfigurer.getSubFunction());

		List<RequestDispatchListener> requestDispatchMatchesListener = CollectionsUtil
				.find(requestDispatchListeners,
						"${systemAppName+ '_'+ function + '_' + subFunction}",
						key);

		if (TypeChecker.isEmpty(requestDispatchMatchesListener)) {

			key = this.dispatchConfigurer.getSystemAppName() + "_"
					+ dispatchConfigurer.getFunction();

			requestDispatchMatchesListener = CollectionsUtil.find(
					requestDispatchListeners,
					"${systemAppName+ '_'+ function + '_' + subFunction}", key);

		}

		return TypeChecker.isEmpty(requestDispatchMatchesListener) ? ObjectUtils.EMPTY_LIST
				: requestDispatchMatchesListener;
	}

	private String writeStreamToResponse(HttpServletRequest request,
			HttpResponseEntity httpResponseEntity, HttpServletResponse response)
			throws Exception {

		FileCopyUtils.copy(httpResponseEntity.getContent(), response
				.getOutputStream());

		XmlWriter xmlWriter = new XmlWriter();

		xmlWriter.addRootElement("output");

		xmlWriter.addElement("body");

		xmlWriter.addElementText("[Download Resource:"
				+ this.getResourceNameDownloaded(request, httpResponseEntity)
				+ "]");

		return xmlWriter.toXmlText();
	}

	private String getResourceNameDownloaded(HttpServletRequest request,
			HttpResponseEntity httpEntity) {

		String attachedFileName = httpEntity
				.getHeaderValue("Content-disposition");

		if (TypeChecker.isEmpty(attachedFileName)) {
			return "";
		}

		return StringUtils.remove(attachedFileName, "inline;filename=");

	}

	private HttpResponseEntity buildHttpClientConnection(
			HttpServletRequest request, Map<String, String[]> requestParameters) {

		HttpPostClient client = new HttpPostClient(dispatchConfigurer.getUrl(),
				dispatchConfigurer.getCharset(), (int) (requestParameters
						.size() * 1.25 + 1));

		LOGGER.info("The dispatched url:" + dispatchConfigurer.getUrl());

		this.prepareHttpParameters(client, request, requestParameters);

		return client.execute();
	}

	private void prepareHttpParameters(HttpPostClient client,
			HttpServletRequest request, Map<String, String[]> requestParameters) {

		requestParameters = httpRequestParameterResovler.resolve(request,
				requestParameters);

		requestDispatchContext
				.setRequestDispatchedParameters(requestParameters);

		for (String parameterName : requestParameters.keySet()) {

			for (String value : requestParameters.get(parameterName)) {

				LOGGER.info("Request Parameter- Name:" + parameterName
						+ " Value:" + value);

				client.addParameter(parameterName, StringUtils
						.trimToEmpty(value));
			}

		}

	}

	private void prepareResponseStatusAndHeaders(HttpResponseEntity httpEntity,
			HttpServletResponse response,
			Map<String, String[]> requestParameters) {

		try {
			response.reset();

			response.resetBuffer();

			this.requestDispatchContext.setResponseDispatchedHeaders(httpEntity
					.getHttpResponse().getAllHeaders());

			for (Header header : httpEntity.getHttpResponse().getAllHeaders()) {

				if ("Content-Length".equalsIgnoreCase(header.getName())) {

					response.setContentLength(NumberUtils.parseInt(header
							.getValue()));

					continue;
				}

				// HTTP分块传输编码允许服务器为动态生成的内容维持HTTP持久链接
				// 分块传输编码（Chunked transfer
				// encoding）是超文本传输协议（HTTP）中的一种数据传输机制，允许HTTP由网页服务器发送给客户端应用（
				// 通常是网页浏览器）的数据可以分成多个部分

				if ("Transfer-Encoding".equalsIgnoreCase(header.getName())) {
					continue;

				}

				if (dispatchConfigurer.isResourceDownloaded()) {

					// 附件信息
					if ("Content-disposition"
							.equalsIgnoreCase(header.getName())) {

						response.setHeader(header.getName(), header.getValue());

						return;
					}

					continue;
				}

				if ("Content-Type".equalsIgnoreCase(header.getName())) {

					response.setContentType(header.getValue());

					continue;
				}

				if (requestParameters.containsKey("excludedHeaderNames")) {
					if ((requestParameters.get("excludedHeaderNames"))[0]
							.indexOf(header.getName()) >= 0) {
						continue;
					}
				}

				response.setHeader(header.getName(), header.getValue());

			}

		} finally {

			if (httpEntity.getStatusCode() != 200) {

				response.setStatus(httpEntity.getStatusCode());

			}
		}

	}

}
