package com.rapid.j2ee.framework.lucene.store.listener;

import java.io.File;
import java.util.List;

import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.lucene.document.Document;
import org.apache.lucene.document.Field;
import org.apache.lucene.document.IntField;
import org.apache.lucene.document.TextField;
import org.apache.lucene.index.IndexWriter;
import org.apache.lucene.index.Term;
import org.springframework.util.Assert;

import com.rapid.j2ee.framework.lucene.LuceneConstants;
import com.rapid.j2ee.framework.lucene.store.context.ResourceAttribute;
import com.rapid.j2ee.framework.lucene.store.context.ResourceContextAttribute;
import com.rapid.j2ee.framework.lucene.stream.ResourceReaderCycleListener;

public class LuceneReaderCycleListener implements ResourceReaderCycleListener,
		LuceneConstants {

	private IndexWriter indexWriter;

	private ResourceAttribute resourceAttribute;

	private List<ResourceContextAttribute> resourceContextAttributes;

	private static final Log Logger = LogFactory
			.getLog(LuceneReaderCycleListener.class);

	private String fsDirectory;

	public LuceneReaderCycleListener(IndexWriter indexWriter) {
		this.indexWriter = indexWriter;
	}

	public void setResource(String fsDirectory,
			ResourceAttribute resourceAttribute,
			List<ResourceContextAttribute> resourceContextAttributes) {

		this.fsDirectory = fsDirectory;
		this.resourceAttribute = resourceAttribute;
		this.resourceContextAttributes = resourceContextAttributes;

	}

	private void checkResourceAvailable() {

		Assert.hasText(fsDirectory,
				"Please provide a existed Lucene FS Directory!");

		Assert.isTrue(new File(fsDirectory).isDirectory()
				&& new File(fsDirectory).canWrite(),
				"Please provide a writable Lucene FS Directory! "
						+ this.fsDirectory);

		Assert.notNull(resourceAttribute,
				"Please provide a Resource Attribute Object!");

		Assert.notNull(resourceAttribute.getResourceType(),
				"Please provide a Resource Type Attribute Object!");

		Assert.notNull(resourceAttribute.getResourceIdContextAttribute(),
				"Please provide a Resource Id Attribute  Context!");

		if (resourceAttribute.getResourceType().isFileResource()) {

			Assert.notNull(resourceAttribute.getResourceFile(),
					"Please provide a Resource File!");
		}

	}

	public boolean open() throws Exception {

		this.checkResourceAvailable();

		if (ifResourceIndexExists()) {

			if (!resourceAttribute.isOverwrite()) {
				return false;
			}

			Term term = new Term(this.resourceAttribute
					.getResourceIdContextAttribute().getAttributeName(),
					this.resourceAttribute.getResourceIdContextAttribute()
							.getAttributeValue());

			indexWriter.deleteDocuments(term);

			Logger.info("Remove Resource Indexes :"
					+ resourceAttribute.getResourceFile()
					+ " for Key:"
					+ resourceAttribute.getResourceIdContextAttribute()
							.getAttributeValue());

		}

		return true;

	}

	public void read(int currentPage, int totalCount, String content)
			throws Exception {

		Logger.info("Create Index For Page:" + currentPage + " Total Page:"
				+ totalCount);

		content = StringUtils.trimToEmpty(content);

		if (content.length() == 0) {
			return;
		}

		// make a new, ey document
		Document doc = new Document();

		doc.add(new IntField(Document_Lucene_Store_FieldName_Page, currentPage,
				Field.Store.YES));

		doc.add(new IntField(Document_Lucene_Store_FieldName_TotalPage,
				totalCount, Field.Store.YES));

		if (resourceContextAttributes != null) {

			for (ResourceContextAttribute resourceContextAttribute : resourceContextAttributes) {

				Field field = resourceContextAttribute.getAttributeType()
						.getField(resourceContextAttribute);
				field.setBoost(resourceContextAttribute.getBoost());

				doc.add(field);
			}
		}

		// Resource Identity Information
		ResourceContextAttribute resourceIdContextAttribute = resourceAttribute
				.getResourceIdContextAttribute();

		Field field = resourceIdContextAttribute.getAttributeType().getField(
				resourceIdContextAttribute);

		field.setBoost(resourceIdContextAttribute.getBoost());

		doc.add(field);

		content = StringUtils.replace(content, "\n", " ");
		content = StringUtils.replace(content, "\t", " ");
		content = StringUtils.replace(content, "\r", " ");
		content = StringUtils.replace(content, ".......", " ");

		Field contentField = new TextField(
				resourceAttribute.getAttributeName(), content,
				resourceAttribute.getFieldStore());

		contentField.setBoost(resourceAttribute.getBoost());

		doc.add(contentField);

		try {
			indexWriter.addDocument(doc);
		} catch (Exception e) {
			Logger.error("Cannot Create Index Page:" + currentPage, e);
		}

	}

	public boolean ifResourceIndexExists() {

		return getResourceIndexExistMarkFile().exists();

	}

	private File getResourceIndexExistMarkFile() {

		String resourceIndexName = resourceAttribute
				.getResourceIdContextAttribute().getAttributeValue()
				+ ".idx";

		Logger.info("Resource Index Existed MarkFile:" + resourceIndexName);

		return new File(this.fsDirectory, resourceIndexName);
	}

	public void close() throws Exception {

		indexWriter.commit();

		String resourceIndexName = resourceAttribute
				.getResourceIdContextAttribute().getAttributeValue()
				+ ".idx";

		new File(this.fsDirectory, resourceIndexName).createNewFile();

	}

	public void remove() throws Exception {

		if (!this.ifResourceIndexExists()) {
			return;
		}

		Term term = new Term(this.resourceAttribute
				.getResourceIdContextAttribute().getAttributeName(),
				this.resourceAttribute.getResourceIdContextAttribute()
						.getAttributeValue());

		indexWriter.deleteDocuments(term);

		this.getResourceIndexExistMarkFile().delete();

	}

}
