package com.rapid.j2ee.framework.mvc.security.logger.publish;

import org.springframework.beans.BeansException;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;

import com.opensymphony.xwork2.ActionInvocation;
import com.rapid.j2ee.framework.core.exception.ApplicationException;
import com.rapid.j2ee.framework.core.exception.BaseException;
import com.rapid.j2ee.framework.core.exception.ExceptionUtils;
import com.rapid.j2ee.framework.core.reflect.ConstructorUtils;
import com.rapid.j2ee.framework.core.utils.ObjectAnalyzer;
import com.rapid.j2ee.framework.core.utils.StringUtils;
import com.rapid.j2ee.framework.core.utils.TypeChecker;
import com.rapid.j2ee.framework.mvc.constants.OperationResultConstants;
import com.rapid.j2ee.framework.mvc.security.logger.AuditActionLogger;
import com.rapid.j2ee.framework.mvc.security.logger.AuditActionLoggerConfigurer;
import com.rapid.j2ee.framework.mvc.security.logger.LoggerStatus;
import com.rapid.j2ee.framework.mvc.security.logger.listener.AuditActionLoggerApplicationEvent;
import com.rapid.j2ee.framework.mvc.security.utils.MvcSecurityActionContextUtils;
import com.rapid.j2ee.framework.mvc.utils.ActionContextUtils;
import com.rapid.j2ee.framework.mvc.web.interceptor.MvcWebActionController;

public class AbstractAuditActionLoggerPublisher implements
		AuditActionLoggerPublisher, ApplicationContextAware {

	public void end(AuditActionLogger auditActionLogger, Throwable error) {

		if (TypeChecker.isNull(auditActionLogger)) {
			return;
		}

		this.toEndAuditActionLogger(auditActionLogger, error);

		this.traceLogger(auditActionLogger);

	}

	private void traceLogger(AuditActionLogger record) {

		if (ignoreAuditActionLogger) {
			return;
		}

		AuditActionLoggerApplicationEvent auditActionLoggerApplicationEvent = new AuditActionLoggerApplicationEvent(
				this.applicationContext, record, MvcSecurityActionContextUtils
						.getWebUser());

		applicationContext.publishEvent(auditActionLoggerApplicationEvent);

	}

	private void toEndAuditActionLogger(AuditActionLogger record,
			Throwable error) {
		try {

			if (!TypeChecker.isNull(error)) {

				BaseException exception = ExceptionUtils
						.convertThrowableToBaseException(error);

				record.setErrorReason(StringUtils.getStringContextWithinLength(
						exception.getResultCode() + "/"
								+ exception.getResultMessage() + "/"
								+ exception.getMessage(), 2.5, 2500, "......"));

				if (OperationResultConstants.isFailureCode(exception
						.getResultCode())) {

					record.setLoggerStatus(LoggerStatus.Failure);

					String errorMsg = StringUtils.getStringContextWithinLength(
							ExceptionUtils.printThrowableStack(error), 2.5,
							2500, "......");

					if (errorMsg.startsWith(ApplicationException.class
							.getName())) {
						errorMsg = errorMsg.split("\r|\n")[0];
					}

					record.setErrorReason(errorMsg);
				}

			}

			record.setMethodParameters(this.getMethodTraceParameterValues());

			record.setMethod(ActionContextUtils.getActionMethodName());

			record.endLogger();

		} catch (Throwable e) {

		}
	}

	public AuditActionLogger start(ActionInvocation actionInvocation,
			AuditActionLoggerConfigurer auditActionLoggerConfigurer) {

		if (ignoreAuditActionLogger) {
			return null;
		}

		try {

			AuditActionLogger auditActionLogger = (AuditActionLogger) ConstructorUtils
					.newInstance(auditActionLoggerConfigurer
							.getAuditActionLoggerClass());

			auditActionLogger.setApplicationName(auditActionLoggerConfigurer
					.getApplicationName());

			auditActionLogger.setActionName(actionInvocation.getAction()
					.getClass().getName());

			auditActionLogger
					.setMethod(MvcWebActionController
							.getMethodNameByRequest(ActionContextUtils
									.getHttpRequest()));

			auditActionLogger.setLoggerStatus(LoggerStatus.Success);

			return auditActionLogger;
		} catch (Throwable e) {
			return null;
		}
	}

	protected String getMethodTraceParameterValues() {
		String traceParameterNames = ActionContextUtils
				.getMethodTraceParameterNames();

		if (TypeChecker.isEmpty(traceParameterNames)) {
			return "";
		}

		StringBuffer paramValues = new StringBuffer(500);

		for (String name : StringUtils.splitBySeparators(traceParameterNames,
				",", ";")) {
			if (TypeChecker.isEmpty(name)) {
				continue;
			}

			paramValues.append(name + "="
					+ ActionContextUtils.getHttpParameter(name, ""));
			paramValues.append("&");
		}

		return paramValues.toString();
	}

	public void setApplicationContext(ApplicationContext applicationContext)
			throws BeansException {

		this.applicationContext = applicationContext;
	}

	public void setIgnoreAuditActionLogger(String ignoreAuditActionLogger) {
		this.ignoreAuditActionLogger = TypeChecker
				.isSpecialTrue(ignoreAuditActionLogger);
	}

	private boolean ignoreAuditActionLogger;

	private ApplicationContext applicationContext;

}
