package com.rapid.j2ee.framework.mvc.security.logic;

import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import com.rapid.j2ee.framework.core.cryptology.CryptologyFactory;
import com.rapid.j2ee.framework.core.cryptology.CryptologyType;
import com.rapid.j2ee.framework.core.exception.ApplicationException;
import com.rapid.j2ee.framework.core.utils.DateTimeUtils;
import com.rapid.j2ee.framework.core.utils.RandomUtils;
import com.rapid.j2ee.framework.core.utils.StringUtils;
import com.rapid.j2ee.framework.core.utils.TypeChecker;
import com.rapid.j2ee.framework.core.utils.UUIDGenerator;
import com.rapid.j2ee.framework.mvc.constants.OperationResultConstants;
import com.rapid.j2ee.framework.mvc.exception.SessionTimeoutApplicationException;
import com.rapid.j2ee.framework.mvc.security.domain.WebUser;
import com.rapid.j2ee.framework.mvc.security.domain.WebUserSetter;
import com.rapid.j2ee.framework.mvc.security.domain.WebUserTokenSession;
import com.rapid.j2ee.framework.mvc.security.utils.MvcSecurityActionContextUtils;
import com.rapid.j2ee.framework.mvc.utils.ActionContextUtils;

public abstract class UserTokenContainerManager implements
		UserTokenContainerManagerOAuth2 {

	public UserTokenContainerManager() {
		webUserTokenSessionMap = new HashMap<String, WebUserTokenSession>();
		keepWebUserTokenSessionActiveTime = 10 * 1000 * 60;

	}

	public void registry() {

		WebUserTokenSession webUserTokenSession = this
				.doRegistry(MvcSecurityActionContextUtils.getWebUser());

		this.doUpdateCascadeWebUser(webUserTokenSession);

		synchronized (webUserTokenSessionMap) {
			webUserTokenSessionMap.put(webUserTokenSession.getTokenId(),
					webUserTokenSession);
		}

		MvcSecurityActionContextUtils
				.saveWebUserTokenSession(webUserTokenSession);
	}

	public void printTokenSessionMap() {

		// System.out
		// .println("--------------------------Token Session Informations
		// Start---------------------------------");
		//
		// System.out.println("Active Token Session Count:"
		// + webUserTokenSessionMap.size());
		// System.out.println();
		// for (String key : webUserTokenSessionMap.keySet()) {
		//
		// System.out.println("Token Id ：" + key);
		// System.out.println("Session Object:"
		// + webUserTokenSessionMap.get(key));
		// System.out.println();
		// }
		//
		// System.out
		// .println("--------------------------Token Session Informations
		// End---------------------------------");

	}

	public void logout(String tokenId) {

		WebUserTokenSession webUserTokenSession = null;

		synchronized (webUserTokenSessionMap) {
			webUserTokenSession = webUserTokenSessionMap.remove(tokenId);
		}

		// Logout
		this.doLogout(tokenId, webUserTokenSession);

	}

	public boolean doTokenEligibilityValidate() {

		try {
			return this.doTokenEligibilityProcess();

		} finally {
			// Passive UnActiveUser
			this.releaseInactionWebUserTokenSessions();
		}

	}

	private synchronized void releaseInactionWebUserTokenSessions() {

		Set<String> copyKeySet = new HashSet<String>(webUserTokenSessionMap
				.keySet());

		for (String tokenId : copyKeySet) {

			WebUserTokenSession webUserTokenSession = webUserTokenSessionMap
					.get(tokenId);

			if (this.isWebUserTokenSessionInactive(webUserTokenSession)) {
				this.webUserTokenSessionMap.remove(tokenId);
			}

		}

	}

	private boolean isWebUserTokenSessionInactive(
			WebUserTokenSession webUserTokenSession) {
		return (System.currentTimeMillis() - webUserTokenSession
				.getTransactionDate().getTime()) > keepWebUserTokenSessionActiveTime;
	}

	private boolean doTokenEligibilityProcess() {

		if (!this.hasTokenId()) {
			return false;
		}

		WebUserTokenSession webUserTokenSession = this
				.getWebUserTokenSessionByTokenId(ActionContextUtils
						.getHttpParameter(this.parameterTokenName));

		if (TypeChecker.isNull(webUserTokenSession)) {
			return false;
		}

		// Check if expired....
		if (!TypeChecker.isNull(webUserTokenSession.getTokenIdExpiredDate())) {

			if (DateTimeUtils.compareDate(webUserTokenSession
					.getTokenIdExpiredDate(), DateTimeUtils.getCurrentDate()) < 0) {

				// Token Id Expired and session is time out!
				// Logout...
				this.logout(webUserTokenSession.getTokenId());

				return false;
			}

		}

		String oldTokenId = webUserTokenSession.getTokenId();

		boolean updateTransactionSuccess = this
				.updateWebUserTokenTransaction(webUserTokenSession);

		this.updateWebUserTokenIdCacheMemory(updateTransactionSuccess,
				oldTokenId, webUserTokenSession);

		return updateTransactionSuccess;
	}

	private void doUpdateCascadeWebUser(WebUserTokenSession webUserTokenSession) {

		if (webUserTokenSession.getWebUser() instanceof WebUserSetter) {

			WebUserSetter webUserSetter = (WebUserSetter) webUserTokenSession
					.getWebUser();

			webUserSetter.setTokenId(webUserTokenSession.getTokenId());
			webUserSetter.setPlatformId(webUserTokenSession.getPlatformId());

		}

		MvcSecurityActionContextUtils.setWebUser(webUserTokenSession
				.getWebUser(), false);

		MvcSecurityActionContextUtils
				.saveWebUserTokenSession(webUserTokenSession);
	}

	private void updateWebUserTokenIdCacheMemory(
			boolean updateTransactionSuccess, String oldTokenId,
			WebUserTokenSession webUserTokenSession) {

		if (!updateTransactionSuccess) {
			return;
		}

		if (!StringUtils.equalsIgnoreCase(oldTokenId, webUserTokenSession
				.getTokenId())) {

			synchronized (webUserTokenSessionMap) {
				webUserTokenSessionMap.remove(oldTokenId);
				webUserTokenSessionMap.put(webUserTokenSession.getTokenId(),
						webUserTokenSession);
			}
		}

		this.doUpdateCascadeWebUser(webUserTokenSession);

	}

	protected synchronized WebUserTokenSession getWebUserTokenSessionByTokenId(
			String tokenId) {

		boolean webUserTokenSessionCached = this.webUserTokenSessionMap
				.containsKey(tokenId);

		WebUserTokenSession webUserTokenSession = webUserTokenSessionMap
				.get(tokenId);

		if (!webUserTokenSessionCached) {

			webUserTokenSession = this.retrieveWebUserTokenSession(tokenId);

		}

		if (TypeChecker.isNull(webUserTokenSession)) {
			return null;
		}

		webUserTokenSession.setTransactionDate(DateTimeUtils.getCurrentDate());

		webUserTokenSession
				.setCountUsed(webUserTokenSession.getCountUsed() + 1);

		webUserTokenSessionMap.put(tokenId, webUserTokenSession);

		return webUserTokenSession;

	}

	public WebUserTokenSession getWebUserTokenSessionByAuthorityCode() {

		return getWebUserTokenSessionByAuthorityCode(ActionContextUtils
				.getHttpParameter(this.parameterAuthorityCodeName),
				ActionContextUtils
						.getHttpParameter(this.parameterPlatformIdName),
				ActionContextUtils
						.getHttpParameter(this.parameterPlatformSecretName));
	}

	public abstract WebUserTokenSession getWebUserTokenSessionByAuthorityCode(
			String authorityCode, String platformId, String platformSecret);

	public WebUserTokenSession getWebUserTokenSessionByAuthorityCode(
			String authorityCode) {

		return getWebUserTokenSessionByAuthorityCode(authorityCode,
				ActionContextUtils
						.getHttpParameter(this.parameterPlatformIdName),
				ActionContextUtils
						.getHttpParameter(this.parameterPlatformSecretName));
	}

	public WebUserTokenSession renewTokenId() {

		String platformId = this
				.decodePlatformIdByRefreshTokenId(ActionContextUtils
						.getHttpParameter(this.parameterRefreshTokenName));

		if (TypeChecker.isEmpty(platformId)) {
			throw new ApplicationException(
					OperationResultConstants.FAILED_OAUTH2_REFRESH_TOKEN_WRONG);
		}

		return renewTokenId(platformId, ActionContextUtils
				.getHttpParameter(this.parameterRefreshTokenName));
	}

	protected abstract WebUserTokenSession renewTokenId(String platformId,
			String refreshTokenId);

	private boolean hasTokenId() {

		return !TypeChecker.isEmpty(ActionContextUtils
				.getHttpParameter(this.parameterTokenName));
	}

	protected abstract WebUserTokenSession doRegistry(WebUser webUser);

	protected boolean updateWebUserTokenTransaction(
			WebUserTokenSession webUserTokenSession) {

		String currentTokenId = webUserTokenSession.getTokenId();
		// 操作时间
		webUserTokenSession.setTransactionDate(DateTimeUtils.getCurrentDate());

		// 同时如果使用数据库，可以校验Token 是否时效，因为内存中的数据没有更新

		if (webUserTokenSession.getCountUsed() > webServiceUserTokenUsedMaxCount) {
			webUserTokenSession.setTokenId(this.getGeneratedNextTokenId());
			webUserTokenSession.setCountUsed(0);
		}

		return this.doUpdateWebUserTokenTransaction(currentTokenId,
				webUserTokenSession);
	}

	public String getGeneratedNextAuthorityCode() {
		return UUIDGenerator.nextUUID()
				+ RandomUtils
						.nextAlphabetNumberLetter(Authority_Code_Suffix_Length);
	}

	public String getGeneratedNextTokenId() {
		return "TK_" + UUIDGenerator.nextUUID() + Token_Inform_Separator
				+ RandomUtils.nextAlphabetNumberLetter(Token_Id_Suffix_Length);
	}

	public String getGeneratedNextRefreshTokenId(String platformId) {
		return "RT_"
				+ UUIDGenerator.nextUUID()
				+ Token_Inform_Separator
				+ CryptologyFactory.getReversibleCryptology(CryptologyType.ELM,
						SecretEnglishLetters_FileName).encrypt(platformId);
	}

	public String decodePlatformIdByRefreshTokenId(String refreshTokenId) {

		return CryptologyFactory.getReversibleCryptology(CryptologyType.ELM,
				SecretEnglishLetters_FileName).decrypt(
				StringUtils.substringAfterLast(refreshTokenId,
						Token_Inform_Separator));
	}

	public void setWebServiceUserTokenUsedMaxCount(
			int webServiceUserTokenUsedMaxCount) {
		this.webServiceUserTokenUsedMaxCount = webServiceUserTokenUsedMaxCount;
	}

	protected abstract boolean doUpdateWebUserTokenTransaction(
			String currentTokenId, WebUserTokenSession webUserTokenSession);

	protected abstract WebUserTokenSession retrieveWebUserTokenSession(
			String tokenId);

	protected abstract void doLogout(String tokenId,
			WebUserTokenSession webUserTokenSession);

	public void setParameterAuthorityCodeName(String parameterAuthorityCodeName) {
		this.parameterAuthorityCodeName = parameterAuthorityCodeName;
	}

	public void setParameterPlatformIdName(String parameterPlatformIdName) {
		this.parameterPlatformIdName = parameterPlatformIdName;
	}

	public void setParameterPlatformSecretName(
			String parameterPlatformSecretName) {
		this.parameterPlatformSecretName = parameterPlatformSecretName;
	}

	public void setParameterRefreshTokenName(String parameterRefreshTokenName) {
		this.parameterRefreshTokenName = parameterRefreshTokenName;
	}

	public void setKeepWebUserTokenSessionActiveMinute(
			int keepWebUserTokenSessionActiveMinute) {
		this.keepWebUserTokenSessionActiveTime = keepWebUserTokenSessionActiveMinute * 1000 * 60;

	}

	public void setParameterTokenName(String parameterTokenName) {
		this.parameterTokenName = parameterTokenName;
	}

	private int keepWebUserTokenSessionActiveTime;

	protected Map<String, WebUserTokenSession> webUserTokenSessionMap;

	private int webServiceUserTokenUsedMaxCount = Integer.MAX_VALUE;

	private String parameterTokenName = "tokenId";

	private String parameterRefreshTokenName = "refreshTokenId";

	private String parameterAuthorityCodeName = "authorityCode";

	private String parameterPlatformIdName = "platformId";

	private String parameterPlatformSecretName = "platformSecret";

	private static final int Token_Id_Suffix_Length = 30;

	private static final int Authority_Code_Suffix_Length = 5;

	private static final String Token_Inform_Separator = "_John_";

	private static final String SecretEnglishLetters_FileName = "SecretEnglishLettersRapidSecurityCore.properties";

}
