package com.rapid.j2ee.framework.mvc.web.interceptor;

import java.lang.reflect.Method;

import javax.servlet.http.HttpServletRequest;

import org.apache.commons.fileupload.FileUploadThreadLocal;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.opensymphony.xwork2.ActionInvocation;
import com.rapid.j2ee.framework.core.exception.ApplicationException;
import com.rapid.j2ee.framework.core.exception.SystemException;
import com.rapid.j2ee.framework.core.reflect.ConstructorUtils;
import com.rapid.j2ee.framework.core.reflect.InvokeUtils;
import com.rapid.j2ee.framework.core.spring.SpringApplicationContextHolder;
import com.rapid.j2ee.framework.core.utils.StringUtils;
import com.rapid.j2ee.framework.core.utils.TypeChecker;
import com.rapid.j2ee.framework.mvc.constants.OperationResultConstants;
import com.rapid.j2ee.framework.mvc.constants.SupportActionConstants;
import com.rapid.j2ee.framework.mvc.exception.ActionMethodLostApplicationException;
import com.rapid.j2ee.framework.mvc.security.MvcSecurityAuthorityFilter;
import com.rapid.j2ee.framework.mvc.security.logger.AuditActionLogger;
import com.rapid.j2ee.framework.mvc.security.logger.AuditActionLoggerConfigurer;
import com.rapid.j2ee.framework.mvc.security.logic.MvcSecurityCleanUpInterceptorHandler;
import com.rapid.j2ee.framework.mvc.utils.ActionContextUtils;
import com.rapid.j2ee.framework.mvc.web.MvcBaseActionSupport;
import com.rapid.j2ee.framework.mvc.web.MvcBaseActionSupportContainer;
import com.rapid.j2ee.framework.mvc.web.annotation.MvcMethodUrlMatcherValidator;
import com.rapid.j2ee.framework.mvc.web.annotation.MvcMethodConfigurer;
import com.rapid.j2ee.framework.mvc.web.context.MvcMethodContext;
import com.rapid.j2ee.framework.mvc.web.error.SupportActionExceptionHandler;

public class MvcWebActionController {

	protected ActionInvocation actionInvocation;

	protected MvcSecurityAuthorityFilter mvcSecurityAuthorityFilter;

	protected SupportActionExceptionHandler supportActionExceptionHandler;

	protected AuditActionLoggerConfigurer auditActionLoggerConfigurer;

	protected MvcSecurityCleanUpInterceptorHandler mvcSecurityCleanUpInterceptorHandler;

	private AuditActionLogger auditActionLogger = null;

	private Throwable error = null;

	public MvcWebActionController(
			ActionInvocation actionInvocation,
			MvcSecurityAuthorityFilter mvcSecurityAuthorityFilter,
			SupportActionExceptionHandler supportActionExceptionHandler,
			AuditActionLoggerConfigurer auditActionLoggerConfigurer,
			MvcSecurityCleanUpInterceptorHandler mvcSecurityCleanUpInterceptorHandler) {

		SpringApplicationContextHolder.inject(this);

		this.actionInvocation = actionInvocation;
		this.mvcSecurityAuthorityFilter = mvcSecurityAuthorityFilter;
		this.supportActionExceptionHandler = supportActionExceptionHandler;
		this.auditActionLoggerConfigurer = auditActionLoggerConfigurer;

		this.mvcSecurityCleanUpInterceptorHandler = mvcSecurityCleanUpInterceptorHandler;
	}

	private void startAuditActionLogger() {

		if (TypeChecker.isNull(auditActionLoggerConfigurer)) {
			return;
		}

		auditActionLogger = auditActionLoggerConfigurer
				.getAuditActionLoggerPublisher().start(actionInvocation,
						auditActionLoggerConfigurer);
	}

	private void endAuditActionLogger() {

		try {
			if (TypeChecker.isNull(auditActionLoggerConfigurer)) {
				return;
			}
			auditActionLoggerConfigurer.getAuditActionLoggerPublisher().end(
					this.auditActionLogger, error);
		} catch (Exception e) {
			e.printStackTrace();
		}
	}

	public String invoke() {

		String actionMethodHandlerName = getMethodNameByRequest(ActionContextUtils
				.getHttpRequest())
				+ "/"
				+ this.actionInvocation.getAction().getClass().getSimpleName();

		String returnPage = null;

		try {

			logger
					.info("\n\n--------------------------Rapid Framework Invokes:"
							+ actionMethodHandlerName
							+ " Start----------------------------------------------------\n\n");

			mvcSecurityCleanUpInterceptorHandler.start(actionInvocation,
					actionMethodHandlerName);

			this.startAuditActionLogger();

			prepareMvcSecurityParameters();

			this.doValidateAfterPreparation();

			this.doMvcSecurityAuthorityAccept();

			this.doValidateAfterMvcSecurityAuthority();

			returnPage = actionInvocation.invoke();

			logger.info("Handle Dispatcher Page:" + returnPage);

			return returnPage;

		} catch (Throwable e) {

			logger.info("\n\n");

			error = e;

			returnPage = supportActionExceptionHandler.handleException(e);

			logger.info("Handle Dispatcher Page:" + returnPage);

			return returnPage;

		} finally {

			mvcSecurityCleanUpInterceptorHandler.end(actionInvocation,
					actionMethodHandlerName);

			this.endAuditActionLogger();

			logger
					.info("\n\n--------------------------Rapid Framework Invokes:"
							+ actionMethodHandlerName
							+ " End. The Returned Page Name:"
							+ StringUtils.trimToEmpty(returnPage)
							+ "----------------------------------------------------\n\n");
		}
	}

	private void doValidateAfterMvcSecurityAuthority() {

		MvcMethodUrlMatcherValidator methodUrlMatcherValidator = ActionContextUtils
				.getActionMethod().getAnnotation(
						MvcMethodUrlMatcherValidator.class);

		if (!TypeChecker.isNull(methodUrlMatcherValidator)) {

			if (!StringUtils.match(ActionContextUtils.getHttpRequest()
					.getRequestURI(), methodUrlMatcherValidator.value())) {
				throw new ApplicationException(
						OperationResultConstants.FAILED_NO_PERMISSION_ACCESSIBLE_ERROR);
			}
		}

	}

	// MvcBaseActionSupport

	private void prepareMvcSecurityParameters() {

		ActionContextUtils
				.setMvcBaseActionSupport((MvcBaseActionSupport) actionInvocation
						.getAction());

		ActionContextUtils.setActionClass(actionInvocation.getAction()
				.getClass());

		ActionContextUtils.getMvcBaseActionSupport().setServletRequest(
				ActionContextUtils.getHttpRequest());

		ActionContextUtils.getMvcBaseActionSupport().setServletResponse(
				ActionContextUtils.getHttpResponse());

		this.setBrowserCharset();

		this.setActionMethodInformation();

		this.setMvcMethodContextInformation();

	}

	protected void doMvcSecurityAuthorityAccept() {

		if (TypeChecker.isNull(mvcSecurityAuthorityFilter)) {
			return;
		}

		mvcSecurityAuthorityFilter.accept(actionInvocation);
	}

	protected void setActionMethodInformation() {

		String method = getMethodNameByRequest(ActionContextUtils
				.getHttpRequest());

		if (TypeChecker.isEmpty(method)) {
			throw new ActionMethodLostApplicationException();
		}

		ActionContextUtils.setActionMethodName(method);

		ActionContextUtils.setActionMethod(findActionMethod(actionInvocation
				.getAction().getClass(), method,
				SupportActionConstants.Action_Invoke_Perfix_Method_Names));

	}

	protected void setBrowserCharset() {

		if (actionInvocation.getAction() instanceof MvcBaseActionSupportContainer) {
			ActionContextUtils
					.setBrowserCharset((MvcBaseActionSupportContainer) actionInvocation
							.getAction());
		}
	}

	protected void setMvcMethodContextInformation() {

		ActionContextUtils.setMvcMethodConfigurer(ActionContextUtils
				.getActionMethod().getAnnotation(MvcMethodConfigurer.class));

		MvcMethodConfigurer mvcMethod = ActionContextUtils
				.getMvcMethodConfigurer();

		if (TypeChecker.isNull(mvcMethod)) {
			return;
		}

		if (!MvcMethodContext.class.isAssignableFrom(mvcMethod.value())) {

			new IllegalArgumentException("Your provided class ["
					+ mvcMethod.value().getSimpleName() + "] must implement "
					+ MvcMethodContext.class);
		}

		logger.info("\n\n------- MvcMethodContext:" + mvcMethod.value()
				+ " , ViewerType " + mvcMethod.type() + "-------\n\n");

		MvcMethodContext mvcContext = (MvcMethodContext) ConstructorUtils
				.newInstance(mvcMethod.value());

		ActionContextUtils.setMvcMethodContext(mvcContext);

	}

	protected void doValidateAfterPreparation() {

	}

	private Method findActionMethod(Class actionClass, String methodName,
			String... prefixs) {

		if (TypeChecker.isEmpty(prefixs)) {
			return InvokeUtils.findMethod(actionClass, methodName);
		}

		Method method = InvokeUtils.findMethod(actionClass, methodName, null,
				true);

		if (!TypeChecker.isNull(method)) {
			return method;
		}

		for (String prefix : prefixs) {

			String methodNameTemp = prefix
					+ StringUtils.upperStartChar(methodName);

			logger.info("Find Action Temp Method :" + methodNameTemp);

			method = InvokeUtils.findMethod(actionClass, methodNameTemp, null,
					true);

			if (!TypeChecker.isNull(method)) {
				return method;
			}
		}

		throw new SystemException("Sorry, the method [" + methodName
				+ "] cannot be found in " + actionClass.getName());
	}

	public static String getMethodNameByRequest(HttpServletRequest request) {

		return StringUtils
				.getFirstItemVoidEmptyInArrays(
						ActionContextUtils
								.getRequestHeader(
										request,
										SupportActionConstants.SupportAction_Request_Method_Name_As_Key2),
						ActionContextUtils
								.getRequestHeader(
										request,
										SupportActionConstants.SupportAction_Request_Method_Name_As_Key1),
						request
								.getParameter(SupportActionConstants.SupportAction_Request_Method_Name_As_Key1),
						request
								.getParameter(SupportActionConstants.SupportAction_Request_Method_Name_As_Key2),
						FileUploadThreadLocal.getInstance().getMethod(),
						SupportActionConstants.SupportAction_Request_Method_Name_As_Default);

	}

	private final Log logger = LogFactory.getLog(this.getClass());

}
