package com.rapid.j2ee.framework.orm.jdbc;

import java.sql.CallableStatement;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;

import javax.sql.DataSource;

import org.springframework.dao.DataAccessException;
import org.springframework.jdbc.core.CallableStatementCallback;
import org.springframework.jdbc.core.CallableStatementCreator;
import org.springframework.jdbc.core.ParameterDisposer;
import org.springframework.jdbc.core.PreparedStatementCallback;
import org.springframework.jdbc.core.PreparedStatementCreator;
import org.springframework.jdbc.core.ResultSetExtractor;
import org.springframework.jdbc.core.SqlProvider;
import org.springframework.jdbc.support.JdbcUtils;
import org.springframework.jdbc.support.SQLErrorCodeSQLExceptionTranslator;
import org.springframework.jdbc.support.SQLExceptionTranslator;
import org.springframework.util.Assert;

public class JdbcTransactionTemplate {

	private DataSource dataSource;

	private SQLExceptionTranslator exceptionTranslator;

	public JdbcTransactionTemplate(DataSource dataSource) {
		this.dataSource = dataSource;
	}

	public JdbcTransactionTemplate() {

	}

	public <T> T query(PreparedStatementCreator psc,
			final PreparedStatementSetterCommit pss,
			final ResultSetExtractor<T> rse) throws DataAccessException {

		Assert.notNull(rse, "ResultSetExtractor must not be null");

		return execute(psc, new PreparedStatementCallback<T>() {
			public T doInPreparedStatement(PreparedStatement ps)
					throws SQLException {
				ResultSet rs = null;
				try {
					if (pss != null) {
						pss.setValues(ps);

						if (pss.commit()) {
							TransactionSynchronizationManager.commit();
						}
					} else {
						TransactionSynchronizationManager.commit();
					}
					
					rs = ps.executeQuery();
					
					return rse.extractData(rs);
				} finally {
					JdbcUtils.closeResultSet(rs);
					if (pss instanceof ParameterDisposer) {
						((ParameterDisposer) pss).cleanupParameters();
					}
				}
			}
		});
	}

	public <T> T query(String sql, PreparedStatementSetterCommit pss,
			ResultSetExtractor<T> rse) throws DataAccessException {
		
		return query(new SimplePreparedStatementCreator(sql), pss, rse);
	}

	// -------------------------------------------------------------------------
	// Methods dealing with callable statements
	// -------------------------------------------------------------------------

	protected <T> T execute(CallableStatementCreator csc,
			CallableStatementCallback<T> action) throws DataAccessException {

		Assert.notNull(csc, "CallableStatementCreator must not be null");
		Assert.notNull(action, "Callback object must not be null");

		Connection con = this.getConnection();

		CallableStatement cs = null;

		try {
			cs = csc.createCallableStatement(con);

			return action.doInCallableStatement(cs);

		} catch (SQLException ex) {

			if (csc instanceof ParameterDisposer) {
				((ParameterDisposer) csc).cleanupParameters();
			}

			csc = null;
			JdbcUtils.closeStatement(cs);
			cs = null;
			TransactionSynchronizationManager.releaseConnection();
			con = null;

			throw getExceptionTranslator().translate(
					"CallableStatementCallback", getSql(csc), ex);
		} finally {

			if (csc instanceof ParameterDisposer) {
				((ParameterDisposer) csc).cleanupParameters();
			}
			JdbcUtils.closeStatement(cs);

		}
	}

	public <T> T execute(PreparedStatementCreator psc,
			PreparedStatementCallback<T> action) throws DataAccessException {

		Assert.notNull(psc, "PreparedStatementCreator must not be null");
		Assert.notNull(action, "Callback object must not be null");

		Connection con = this.getConnection();
		PreparedStatement ps = null;
		try {

			ps = psc.createPreparedStatement(con);

			return action.doInPreparedStatement(ps);
		} catch (SQLException ex) {
			// Release Connection early, to avoid potential connection pool
			// deadlock
			// in the case when the exception translator hasn't been initialized
			// yet.
			if (psc instanceof ParameterDisposer) {
				((ParameterDisposer) psc).cleanupParameters();
			}
			String sql = getSql(psc);
			psc = null;
			JdbcUtils.closeStatement(ps);
			ps = null;
			TransactionSynchronizationManager.releaseConnection();
			con = null;
			throw getExceptionTranslator().translate(
					"PreparedStatementCallback", sql, ex);
		} finally {
			if (psc instanceof ParameterDisposer) {
				((ParameterDisposer) psc).cleanupParameters();
			}
			JdbcUtils.closeStatement(ps);

		}
	}

	protected int update(final PreparedStatementCreator psc,
			final PreparedStatementSetterCommit pss) throws DataAccessException {

		return execute(psc, new PreparedStatementCallback<Integer>() {
			public Integer doInPreparedStatement(PreparedStatement ps)
					throws SQLException {
				try {
					if (pss != null) {
						pss.setValues(ps);

						if (pss.commit()) {
							TransactionSynchronizationManager.commit();
						}
					} else {
						TransactionSynchronizationManager.commit();
					}

					return ps.executeUpdate();
				} finally {
					if (pss instanceof ParameterDisposer) {
						((ParameterDisposer) pss).cleanupParameters();
					}
				}
			}
		});
	}

	public int update(PreparedStatementCreator psc) throws DataAccessException {
		return update(psc, (PreparedStatementSetterCommit) null);
	}

	public int update(String sql, PreparedStatementSetterCommit pss)
			throws DataAccessException {
		
		return update(new SimplePreparedStatementCreator(sql), pss);
	}

	/**
	 * Determine SQL from potential provider object.
	 * 
	 * @param sqlProvider
	 *            object that's potentially a SqlProvider
	 * @return the SQL string, or <code>null</code>
	 * @see SqlProvider
	 */
	private static String getSql(Object sqlProvider) {
		if (sqlProvider instanceof SqlProvider) {
			return ((SqlProvider) sqlProvider).getSql();
		} else {
			return null;
		}
	}

	protected Connection getConnection() {

		return TransactionSynchronizationManager.getConnection(this.dataSource);
	}

	private SQLExceptionTranslator getExceptionTranslator() {

		if (this.exceptionTranslator == null) {
			this.exceptionTranslator = new SQLErrorCodeSQLExceptionTranslator(
					dataSource);

		}

		return this.exceptionTranslator;
	}

	public void setDataSource(DataSource dataSource) {
		this.dataSource = dataSource;
	}

	/**
	 * Simple adapter for PreparedStatementCreator, allowing to use a plain SQL
	 * statement.
	 */
	private static class SimplePreparedStatementCreator implements
			PreparedStatementCreator, SqlProvider {

		private final String sql;

		public SimplePreparedStatementCreator(String sql) {
			Assert.notNull(sql, "SQL must not be null");
			this.sql = sql;
		}

		public PreparedStatement createPreparedStatement(Connection con)
				throws SQLException {
			return con.prepareStatement(this.sql);
		}

		public String getSql() {
			return this.sql;
		}
	}

}
