package com.rapid.j2ee.framework.orm.medium.configurer;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.springframework.core.io.Resource;

import com.rapid.j2ee.framework.core.exception.ExceptionUtils;
import com.rapid.j2ee.framework.core.io.xml.XNode;
import com.rapid.j2ee.framework.core.io.xml.XPathParser;
import com.rapid.j2ee.framework.core.reflect.ConstructorUtils;
import com.rapid.j2ee.framework.core.utils.ResourceUtils;
import com.rapid.j2ee.framework.core.utils.StringUtils;
import com.rapid.j2ee.framework.core.utils.TypeChecker;
import com.rapid.j2ee.framework.core.utils.support.DateTimeFormat;
import com.rapid.j2ee.framework.orm.medium.field.FieldColumn;
import com.rapid.j2ee.framework.orm.medium.getter.BigDecimalOutResultSetAcceptor;
import com.rapid.j2ee.framework.orm.medium.getter.DateOutResultSetAcceptor;
import com.rapid.j2ee.framework.orm.medium.getter.FloatOutResultSetAcceptor;
import com.rapid.j2ee.framework.orm.medium.getter.NStringOutResultSetAcceptor;
import com.rapid.j2ee.framework.orm.medium.getter.OutResultSetAcceptor;
import com.rapid.j2ee.framework.orm.medium.getter.StringOutResultSetAcceptor;
import com.rapid.j2ee.framework.orm.medium.getter.TimestampOutResultSetAcceptor;
import com.rapid.j2ee.framework.orm.medium.mapper.ColumnFieldMapper;
import com.rapid.j2ee.framework.orm.medium.mapper.ColumnNamingResolver;
import com.rapid.j2ee.framework.orm.medium.mapper.ColumnOrderResovler;
import com.rapid.j2ee.framework.orm.medium.mapper.DefaultColumnNamingRevisor;
import com.rapid.j2ee.framework.orm.medium.mapper.EliminateColumnUnderLineFieldMapper;
import com.rapid.j2ee.framework.orm.medium.statement.BigDecimalInPreparedStatementAcceptor;
import com.rapid.j2ee.framework.orm.medium.statement.DateInPreparedStatementAcceptor;
import com.rapid.j2ee.framework.orm.medium.statement.FloatInPreparedStatementAcceptor;
import com.rapid.j2ee.framework.orm.medium.statement.InPreparedStatementAcceptor;
import com.rapid.j2ee.framework.orm.medium.statement.NStringInPreparedStatementAcceptor;
import com.rapid.j2ee.framework.orm.medium.statement.OracleNStringInPreparedStatementAcceptor;
import com.rapid.j2ee.framework.orm.medium.statement.StringInPreparedStatementAcceptor;
import com.rapid.j2ee.framework.orm.medium.statement.TimestampInPreparedStatementAcceptor;
import com.rapid.j2ee.framework.orm.medium.table.DefaultTableColumnTypeMapper;
import com.rapid.j2ee.framework.orm.medium.table.OracleTableColumnTypeMapper;
import com.rapid.j2ee.framework.orm.medium.table.TableColumnTypeMapper;

public class MeduimMapperConfigurer {

	private String mapperLocation;

	private Database database;

	private ColumnFieldMapper columnFieldMapper = null;

	private ColumnNamingResolver columnNamingResovler = null;

	private ColumnOrderResovler columnOrderResovler = ColumnOrderResovler.ColumnOrderResovler_Default;

	private TableColumnTypeMapper tableColumnTypeMapper = null;

	private static final ColumnFieldMapper DEFAULT_COLUMNFIELD_MAPPER = new EliminateColumnUnderLineFieldMapper();

	private static final ColumnNamingResolver DEFAULT_COLUMNNAMING_REVISOR = new DefaultColumnNamingRevisor();

	private static final TableColumnTypeMapper DEFAULT_TABLECOLUMNTYPE_MAPPER = new DefaultTableColumnTypeMapper();

	private Map<String, OutResultSetAcceptor> resultSetAcceptors = null;

	private Map<String, InPreparedStatementAcceptor> preparedStatementAcceptors = null;

	// auto-increase-id
	private String autoIncreaseIdSql = null;

	private Resource mapperLocationResource;

	public MeduimMapperConfigurer(String mapperLocation,
			Resource mapperLocationResource) {

		this.mapperLocation = mapperLocation;
		this.mapperLocationResource = mapperLocationResource;

		this.columnFieldMapper = DEFAULT_COLUMNFIELD_MAPPER;
		this.columnNamingResovler = DEFAULT_COLUMNNAMING_REVISOR;
		this.tableColumnTypeMapper = DEFAULT_TABLECOLUMNTYPE_MAPPER;
		resultSetAcceptors = new HashMap<String, OutResultSetAcceptor>();
		preparedStatementAcceptors = new HashMap<String, InPreparedStatementAcceptor>();

		dateTimeFormat = DateTimeFormat.YYYY_MM_DD_HH_MM_SS.getFormat();

		prepare();

		load();

	}

	private void prepare() {

		preparedStatementAcceptors.put("STRING",
				new StringInPreparedStatementAcceptor());
		preparedStatementAcceptors.put("BIGDECIMAL",
				new BigDecimalInPreparedStatementAcceptor());

		preparedStatementAcceptors.put("ORACLEINTEGER",
				new BigDecimalInPreparedStatementAcceptor());
		
		preparedStatementAcceptors.put("INTEGER",
				new BigDecimalInPreparedStatementAcceptor());

		preparedStatementAcceptors.put("FLOAT",
				new FloatInPreparedStatementAcceptor());
		
		preparedStatementAcceptors.put("TIMESTAMP",
				new TimestampInPreparedStatementAcceptor());
		
		preparedStatementAcceptors.put("DATE",
				new DateInPreparedStatementAcceptor());
		
		preparedStatementAcceptors.put("NSTRING",
				new NStringInPreparedStatementAcceptor());
		
		preparedStatementAcceptors.put("ORACLENSTRING",
				new OracleNStringInPreparedStatementAcceptor());

		resultSetAcceptors.put("STRING", new StringOutResultSetAcceptor());

		resultSetAcceptors.put("BIGDECIMAL",
				new BigDecimalOutResultSetAcceptor());

		resultSetAcceptors.put("INTEGER", new BigDecimalOutResultSetAcceptor());

		resultSetAcceptors.put("FLOAT", new FloatOutResultSetAcceptor());
		
		resultSetAcceptors.put("DOUBLE", new FloatOutResultSetAcceptor());

		resultSetAcceptors
				.put("TIMESTAMP", new TimestampOutResultSetAcceptor());
		
		resultSetAcceptors.put("DATE", new DateOutResultSetAcceptor());
		
		resultSetAcceptors.put("NSTRING", new NStringOutResultSetAcceptor());
		

	}

	private void load() {
		try {

			Resource res = mapperLocationResource;

			if (TypeChecker.isNull(res)) {
				res = ResourceUtils.getResource(this.mapperLocation);
			}

			XPathParser parser = new XPathParser(res.getInputStream());

			parseDatabaseInfo(parser.evalNode("configuration/database"));

			parseColumnMappers(parser.evalNode("configuration/columnmappers"));

			parseColumntypers(parser.evalNodes("configuration/columntypers"));

			parseAutoIncreaseIdSql(parser
					.evalNode("configuration/autoincreaseid"));

		} catch (Exception e) {
			throw ExceptionUtils.convertThrowableToBaseException(e);
		}
	}

	public void parseAutoIncreaseIdSql(XNode node) {

		if (TypeChecker.isNull(node)) {
			return;
		}

		autoIncreaseIdSql = node.getStringAttribute("sql");

	}

	public void parseColumntypers(List<XNode> nodes) {

		if (TypeChecker.isEmpty(nodes)) {
			return;
		}

		for (XNode node : nodes) {
			if ("datetime".equalsIgnoreCase(node.getStringAttribute("type"))
					|| "date".equalsIgnoreCase(node.getStringAttribute("type"))) {
				dateTimeFormat = node.getStringAttribute("format");
			}
		}

	}

	public String getDateTimeFormat() {
		return this.dateTimeFormat;
	}

	private String dateTimeFormat;

	private void parseColumnMappers(XNode node) {

		if (TypeChecker.isNull(node)) {
			return;
		}

		if (node.isEvalNode("resultSetAcceptors/resultSetAcceptor")) {
			List<XNode> nodes = node
					.evalNodes("resultSetAcceptors/resultSetAcceptor");

			for (XNode temp : nodes) {
				resultSetAcceptors.put(temp.getStringAttribute("name"),
						(OutResultSetAcceptor) ConstructorUtils
								.newInstance(temp.getStringAttribute("type")));
			}
		}

		if (node
				.isEvalNode("preparedStatementAcceptors/preparedStatementAcceptor")) {
			List<XNode> nodes = node
					.evalNodes("preparedStatementAcceptors/preparedStatementAcceptor");

			for (XNode temp : nodes) {

				preparedStatementAcceptors.put(temp.getStringAttribute("name"),
						(InPreparedStatementAcceptor) ConstructorUtils
								.newInstance(temp.getStringAttribute("type")));
			}
		}

	}

	private void parseDatabaseInfo(XNode node) {

		this.database = new Database(node);

		if (TypeChecker.isNull(node)) {
			return;
		}

		parseColumnFieldMapperByDatabase(node);
	}

	private void parseColumnFieldMapperByDatabase(XNode node) {

		if (node.isEvalNode("columnFieldMapper")) {

			columnFieldMapper = (ColumnFieldMapper) ConstructorUtils
					.newInstance(node.evalNode("columnFieldMapper")
							.getStringAttribute("type"));
		}

		if (node.isEvalNode("columnNamingResolver")) {
			columnNamingResovler = (ColumnNamingResolver) ConstructorUtils
					.newInstance(node.evalNode("columnNamingResolver")
							.getStringAttribute("type"));
		}

		if (node.isEvalNode("columnOrderResolver")) {
			columnOrderResovler = (ColumnOrderResovler) ConstructorUtils
					.newInstance(node.evalNode("columnOrderResolver")
							.getStringAttribute("type"));
		}

	}

	public ColumnFieldMapper getColumnFieldMapper() {

		return columnFieldMapper;
	}

	public ColumnNamingResolver getColumnNamingResovler() {

		return columnNamingResovler;
	}

	public TableColumnTypeMapper getTableColumnTypeMapper() {
		return new OracleTableColumnTypeMapper();
	}

	public OutResultSetAcceptor getOutResultSetAcceptor(FieldColumn column) {

		String type = StringUtils.trimToEmpty(
				column.getColumn().getColumnType().getInTypeName())
				.toUpperCase();

		String dbType = (this.database.getType() + type).toUpperCase();

		if (resultSetAcceptors.containsKey(dbType)) {

			return resultSetAcceptors.get(dbType);
		}

		if (resultSetAcceptors.containsKey(type)) {
			return resultSetAcceptors.get(type);
		}

		type = StringUtils.remove(type, "NULL");

		dbType = StringUtils.remove(dbType, "NULL");

		if (resultSetAcceptors.containsKey(dbType)) {

			return resultSetAcceptors.get(dbType);
		}

		if (resultSetAcceptors.containsKey(type)) {
			return resultSetAcceptors.get(type);
		}



		throw new IllegalArgumentException(
				"For result set, Unsupported Column Type " + type + " or "
						+ dbType);
	}

	public InPreparedStatementAcceptor getInPreparedStatementAcceptor(
			FieldColumn column) {

		String type = StringUtils.trimToEmpty(
				column.getColumn().getColumnType().getInTypeName())
				.toUpperCase();

		String dbType = (this.database.getType() + type).toUpperCase();

		if (preparedStatementAcceptors.containsKey(dbType)) {

			return preparedStatementAcceptors.get(dbType);
		}

		if (preparedStatementAcceptors.containsKey(type)) {
			return preparedStatementAcceptors.get(type);
		}

		type = StringUtils.remove(type, "NULL");

		dbType = StringUtils.remove(dbType, "NULL");

		if (preparedStatementAcceptors.containsKey(dbType)) {

			return preparedStatementAcceptors.get(dbType);
		}

		if (preparedStatementAcceptors.containsKey(type)) {
			return preparedStatementAcceptors.get(type);
		}



		throw new IllegalArgumentException(
				"For Statement, Unsupported Column Type [" + dbType + "] or   "
						+ type);
	}

	public void clear() {

		try {
			resultSetAcceptors.clear();
		} catch (Exception e) {

		}

		resultSetAcceptors = null;

		try {

			this.preparedStatementAcceptors.clear();
		} catch (Exception e) {

		}

		preparedStatementAcceptors = null;
	}

	private class Database {

		private Database(XNode node) {
			this.type = "";
			if (TypeChecker.isNull(node)) {
				return;
			}
			type = node.getStringAttribute("type");
		}

		private String type;

		public String getType() {
			return type;
		}

	}

	public ColumnOrderResovler getColumnOrderResovler() {
		return columnOrderResovler;
	}

	public String getAutoIncreaseIdSql() {
		return autoIncreaseIdSql;
	}

}
