package com.rapid.j2ee.framework.orm.pkg;

import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;

import javax.sql.DataSource;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.dao.DataAccessException;
import org.springframework.dao.DuplicateKeyException;
import org.springframework.jdbc.core.ResultSetExtractor;
import org.springframework.util.Assert;

import com.rapid.j2ee.framework.core.utils.StringUtils;
import com.rapid.j2ee.framework.core.utils.TypeChecker;
import com.rapid.j2ee.framework.orm.jdbc.JdbcTransactionTemplate;
import com.rapid.j2ee.framework.orm.jdbc.PreparedStatementSetterCommit;
import com.rapid.j2ee.framework.orm.jdbc.TransactionSynchronizationManager;

public class PrimaryKeyJdbcGenerator {

	private static final Log LOGGER = LogFactory
			.getLog(PrimaryKeyJdbcGenerator.class);

	private String primaryKeyTableName;

	private JdbcTransactionTemplate jdbcTransactionTemplate;

	private String updateSeqFuncSql;

	public PrimaryKeyJdbcGenerator(DataSource dataSource,
			String primaryKeyTableName, String updateSeqFuncSql) {
		jdbcTransactionTemplate = new JdbcTransactionTemplate(dataSource);
		this.primaryKeyTableName = primaryKeyTableName;
		this.updateSeqFuncSql = updateSeqFuncSql;

	}

	/**
	 * Getter a primary key from dynamic system table designed for generation pk
	 * 
	 * @param tableName --
	 *            Table Name
	 * @return String -- primary key
	 * 
	 */
	public Long getPrimaryKeyNextVal(String tableName, String param,
			int stepNumber) {

		LOGGER.info("Long getPrimaryKeyNextVal(Table Name:" + tableName
				+ " Param:" + param + " Step Number:" + stepNumber
				+ ") called...");

		long maxValue = getMaxNextPrimaryKey(tableName, param, stepNumber);

		LOGGER.info("Primary Key Table Name :" + this.primaryKeyTableName
				+ " Max Value:" + maxValue);

		if (maxValue != Long.MIN_VALUE) {
			return maxValue;

		}

		maxValue = getMaxNextPrimaryKeyBeforeCreate(tableName, param,
				stepNumber);

		if (maxValue == Long.MIN_VALUE) {

			throw new DuplicateKeyException(
					"Please check if dupliated primary key increasement configuration in "
							+ primaryKeyTableName + ". Table Name:" + tableName
							+ " Param:" + param);
		}

		return maxValue;

	}

	private Long getMaxNextPrimaryKeyBeforeCreate(final String tableName,
			final String param, final int stepNumber) {
		try {
			createInitialPrimaryKeyTransaction(tableName, param);
			return getMaxNextPrimaryKey(tableName, param, stepNumber);
		} finally {
			TransactionSynchronizationManager.releaseConnection();
		}
	}

	private void createInitialPrimaryKeyTransaction(final String tableName,
			final String param) {

		this.jdbcTransactionTemplate.update(getInsertPrimaryKeySQL(),
				new PreparedStatementSetterCommit() {

					public void setValues(PreparedStatement ps)
							throws SQLException {
						ps.setString(1, tableName);
						ps.setString(2, param);
						ps.setLong(3, 0);

					}

					public boolean commit() {

						return false;
					}

				});

	}

	private Long getMaxNextPrimaryKey(final String tableName,
			final String param, final int stepNumber) {
		try {
			jdbcTransactionTemplate.update(getUpdatePrimaryKeySQL(),
					new PreparedStatementSetterCommit() {

						public void setValues(PreparedStatement ps)
								throws SQLException {
							ps.setInt(1, stepNumber);
							ps.setString(2, tableName);
							ps.setString(3, param);
						}

						public boolean commit() {

							return false;
						}

					});

			return jdbcTransactionTemplate.query(getSearchPrimaryKeySQL(),
					new PreparedStatementSetterCommit() {

						public void setValues(PreparedStatement ps)
								throws SQLException {

							ps.setString(1, tableName);
							ps.setString(2, param);

						}

						public boolean commit() {

							return true;
						}
					}, new ResultSetExtractor<Long>() {

						public Long extractData(ResultSet rs)
								throws SQLException, DataAccessException {

							if (rs.next()) {
								return rs.getLong(1);
							}

							if (rs.next()) {
								throw new DuplicateKeyException(
										"Please check if dupliated primary key increasement configuration in "
												+ primaryKeyTableName
												+ ". Table Name:" + tableName
												+ " Param:" + param);
							}

							return Long.MIN_VALUE;
						}

					}

			);

		} finally {
			TransactionSynchronizationManager.releaseConnection();
		}

	}

	public void restoreReservedPrimarykey(final String tableName,
			final String param, final long currentVal, final int step) {

		try {
			int changedRow = jdbcTransactionTemplate.update(
					getUpdatePrimaryKeySQL(),
					new PreparedStatementSetterCommit() {

						public void setValues(PreparedStatement ps)
								throws SQLException {
							ps.setLong(1, currentVal);
							ps.setLong(2, currentVal + step);
							ps.setString(3, tableName);
							ps.setString(4, param);
						}

						public boolean commit() {

							return true;
						}

					});

			if (changedRow > 0) {
				System.out.println("Successfully Restore next value is "
						+ currentVal + " on record's table:" + tableName
						+ " param:" + param + " in " + this.primaryKeyTableName
						+ " Table.");
			}

		} finally {
			TransactionSynchronizationManager.releaseConnection();
		}
	}

	private String getUpdatePrimaryKeySQL() {

		if (!TypeChecker.isEmpty(updateSeqFuncSql)) {

			Assert.isTrue(
					StringUtils.getOccuredCounts(updateSeqFuncSql, "?") == 1,
					"Please include only one ? as input parameter in your function! ["
							+ updateSeqFuncSql + "]");

			return "UPDATE " + primaryKeyTableName + " SET NEXTVAL= "
					+ updateSeqFuncSql + " WHERE TABLENAME=? AND PARAM=?";
		}

		return "UPDATE " + primaryKeyTableName
				+ " SET NEXTVAL= NEXTVAL+? WHERE TABLENAME=? AND PARAM=?";
	}

	private String getSearchPrimaryKeySQL() {

		return "SELECT NEXTVAL FROM " + primaryKeyTableName
				+ "  WHERE TABLENAME=? AND PARAM=?";
	}

	private String getInsertPrimaryKeySQL() {
		return "INSERT INTO " + primaryKeyTableName
				+ "(TABLENAME, PARAM, NEXTVAL) VALUES (?,?,?)";
	}

}
