package com.rapid.j2ee.framework.quartz;

import java.util.Date;
import java.util.Map;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.quartz.Job;
import org.quartz.JobDetail;
import org.quartz.JobExecutionContext;
import org.quartz.JobExecutionException;
import org.quartz.SchedulerException;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;

import com.rapid.j2ee.framework.core.spring.SpringApplicationContextHolder;
import com.rapid.j2ee.framework.core.utils.NumberUtils;
import com.rapid.j2ee.framework.core.utils.TypeChecker;
import com.rapid.j2ee.framework.core.utils.UUIDGenerator;
import com.rapid.j2ee.framework.quartz.interceptor.QuartzJobInterceptorContainer;
import com.rapid.j2ee.framework.quartz.interceptor.QuartzJobInterceptor.JobStatus;
import com.rapid.j2ee.framework.quartz.log.QuartzJobLogger;

public abstract class AbstractQuartzJob implements Job {

	protected Log logger = LogFactory.getLog(this.getClass());

	public void execute(JobExecutionContext context)
			throws JobExecutionException {

		SpringApplicationContextHolder.inject(this);

		String jobClassName = this.getClass().getSimpleName();

		logger.info("<!------- void execute/ " + jobClassName
				+ " Start! Job Execution Context：" + context + "------->");

		long startTime = System.currentTimeMillis();

		Throwable error = null;

		QuartzJobLogger quartzJobLogger = new QuartzJobLogger("QuartzJob_"
				+ context.getJobDetail().getKey() + "_" + jobClassName
				+ UUIDGenerator.nextUUID(), context);

		JobStatus jobStatus = JobStatus.Success;

		try {

			quartzJobInterceptorContainer.initQuartzJobBean(this);

			jobStatus = quartzJobInterceptorContainer.beforeJobExecute(
					quartzJobLogger, context,
					getApplicationContextFromScheduleContext(context),
					getScheduledFireTime(context), context.getJobDetail());

			if (jobStatus != JobStatus.Success) {

				return;
			}

			this.doExecuteJob(quartzJobLogger, context,
					getApplicationContextFromScheduleContext(context),
					getScheduledFireTime(context), context.getJobDetail());

		} catch (Throwable e) {

			e.printStackTrace();

			logger.error("<!------- execute job error!------->", e);

			jobStatus = JobStatus.Fail;

			error = e;

			if (e instanceof JobExecutionException) {
				throw (JobExecutionException) e;
			}

		} finally {

			logger.error("<!------- execute " + jobStatus + " end !------->");

			quartzJobLogger.endQuartzJobLogger(context, error, jobStatus);

			quartzJobInterceptorContainer.completeJobExecute(quartzJobLogger,
					context, startTime, applicationContext,
					getScheduledFireTime(context), context.getJobDetail(),
					error, jobStatus);

			quartzJobInterceptorContainer.finalizeJobExecute();
		}

	}

	protected abstract void doExecuteJob(QuartzJobLogger quartzJobLogger,
			JobExecutionContext context,
			ApplicationContext applicationContextFromScheduleContext,
			Date scheduledFireTime, JobDetail jobDetail) throws Throwable;

	protected Date getScheduledFireTime(JobExecutionContext context) {

		if (!context.getJobDetail().getJobDataMap().containsKey(
				"re-run-firedate")) {
			return context.getScheduledFireTime();
		}

		long reRunFireDate = NumberUtils.parseLong(context.getJobDetail()
				.getJobDataMap().getString("re-run-firedate"));

		context.getJobDetail().getJobDataMap().remove("re-run-firedate");

		if (reRunFireDate == 0) {
			return context.getScheduledFireTime();
		}

		return new Date(reRunFireDate);
	}

	private ApplicationContext getApplicationContextFromScheduleContext(
			JobExecutionContext context) {

		try {
			if (!TypeChecker.isNull(applicationContext)) {
				return applicationContext;
			}

			Map<String, Object> scheduleContext = context.getScheduler()
					.getContext().getWrappedMap();

			for (Object value : scheduleContext.values()) {
				if (value instanceof ApplicationContext) {
					return (ApplicationContext) value;
				}
			}

		} catch (SchedulerException e) {
			logger.error("execute job error!", e);
		}

		return null;

	}

	@Autowired
	private QuartzJobInterceptorContainer quartzJobInterceptorContainer;

	@Autowired
	private ApplicationContext applicationContext;
}
