package com.rocogz.syy.order.client;

import com.rocogz.syy.common.PageTable;
import com.rocogz.syy.common.response.Response;
import com.rocogz.syy.order.constant.OtherInfoOption;
import com.rocogz.syy.order.constant.trace.BussiSystemEnum;
import com.rocogz.syy.order.constant.trace.BussiTypeEnum;
import com.rocogz.syy.order.constant.trace.CallApiEnum;
import com.rocogz.syy.order.dto.CancelOrderResultDto;
import com.rocogz.syy.order.dto.OrderCashPayDto;
import com.rocogz.syy.order.dto.after.*;
import com.rocogz.syy.order.dto.goods.CreateGoodsOrderDto;
import com.rocogz.syy.order.dto.goods.GoodsAdminSearchParamDto;
import com.rocogz.syy.order.dto.goods.GoodsOrderChangeStatusDto;
import com.rocogz.syy.order.dto.pay.WxPayRefundParamDto;
import com.rocogz.syy.order.dto.pay.WxPayUnifiedOrderParamDto;
import com.rocogz.syy.order.dto.pay.WxPayUnifiedOrderResultDto;
import com.rocogz.syy.order.dto.trace.ApiRequestLogStatisticsResponseDto;
import com.rocogz.syy.order.entity.after.OrderAfter;
import com.rocogz.syy.order.entity.after.OrderAfterLog;
import com.rocogz.syy.order.entity.goods.GoodsOrder;
import com.rocogz.syy.order.entity.orders.OrderLog;
import com.rocogz.syy.order.entity.orders.SaleOrder;
import com.rocogz.syy.order.entity.pay.WxPayRecord;
import com.rocogz.syy.order.entity.pay.WxPayRefundRecord;
import com.rocogz.syy.order.entity.trace.OrderErrorTrace;
import com.rocogz.syy.order.entity.trace.OrderThirdTrace;
import org.springframework.cloud.openfeign.SpringQueryMap;
import org.springframework.web.bind.annotation.*;

import java.time.LocalDateTime;
import java.util.List;

/**
 * 订单接口
 *
 * @author：zhangmin
 * @since：2020/3/25
 */
public interface IOrderClientService {

    /**
     * 创建订单
     *
     * @param order
     */
    @PostMapping("/api/order/create")
    Response createOrder(@RequestBody SaleOrder order);

    /**
     * 获取订单主信息
     *
     * @param orderCode 订单编号
     */
    @GetMapping("/api/order/getOrderMainInfo/{orderCode}")
    Response<SaleOrder> getOrderMainInfo(@PathVariable("orderCode") String orderCode);

    /**
     * 查询订单主信息,如果需要其他信息也一起加载
     *
     * @param orderCode     订单编号
     * @param otherInfoList 需要加载的其他信息选项,用来实现按需加载,如果需要全部其他信息,则传入 ALL
     **/
    @GetMapping("/api/order/getOrderWithOtherInfo/{orderCode}")
    Response<SaleOrder> getOrderWithOtherInfo(@PathVariable("orderCode") String orderCode,
                                              @RequestParam("otherInfoList") List<OtherInfoOption> otherInfoList);

      /**
     * 车主端小程序 搜索订单
     * @param userCode
     * @return
     */
    @GetMapping("/api/order/weappSearch")
    PageTable<SaleOrder> weappSearch(@RequestParam("userCode") String userCode,
                                     @RequestParam(value = "status", required = false) String status,
                                     @RequestParam(value = "orderTypeList", required = false) List<String> orderTypeList,
                                     @RequestParam(value = "limit", defaultValue = "10") Integer limit,
                                     @RequestParam(value = "page", defaultValue = "1") Integer page);



    /**
     * 逻辑删除订单
     *
     * @param orderCode 订单号
     * @param userCode  如果是weapp端调用,则需要传入用户编号,用来判断是否是车主自己的订单
     * @return
     */
    @GetMapping("/api/order/delete/{orderCode}")
    Response<String> deleteLogical(@PathVariable("orderCode") String orderCode,
                                   @RequestParam(value = "userCode", required = false) String userCode);

    /**
     * 关闭 超时未支付的订单
     *
     * @param orderCode 订单号
     * @return 如果关单成功, 则返回需要退回的用户券码
     */
    @GetMapping("/api/order/cancelOrder/{orderCode}")
    Response<CancelOrderResultDto> cancelOrder(@PathVariable("orderCode") String orderCode);

    /**
     * 订单的操作日志列表
     *
     * @param orderCode 订单编号
     * @return 返回结果按照id倒序排列
     */
    @GetMapping("/api/order/log/{orderCode}")
    Response<List<OrderLog>> findLogList(@PathVariable("orderCode") String orderCode);

    /**
     * 定时把过了过期时间的订单设置为已过期
     *
     * @return
     */
    @GetMapping("/api/order/scheduleExpireOrder")
    Response<String> scheduleExpireOrder();

    /**
     * 定时查询 需要现金支付并支付超时的订单号(主要解决rabbitmq延迟任务中消息丢失,mq宕机等问题导致的数据不一致行问题)
     *
     * @return
     */
    @GetMapping("/api/order/scheduleFindCashPayTimeoutOrderCodes")
    Response<List<String>> scheduleFindCashPayTimeOutOrderCodes();

    /**
     * 更新订单状态 & 保存操作日志
     * 注意：此接口只针对单个订单的状态更新
     *
     * @param logList （orderCode 与 afterStatus 必须传入）
     */
    @PostMapping("/api/order/writeOff/updateOrderStatus")
    Response<String> updateStatusWithSaveLog(@RequestBody List<OrderLog> logList);


    //服务调用 跟踪   相关接口 start   ===========>

    @GetMapping("/api/order/trace/error")
    PageTable<OrderErrorTrace> searchErrorTrace(@RequestParam(value = "coreCode", required = false) String coreCode,
                                                @RequestParam(value = "bussType", required = false) BussiTypeEnum bussType,
                                                @RequestParam(value = "callApi", required = false) CallApiEnum callApi,
                                                @RequestParam(value = "system", required = false) BussiSystemEnum system,
                                                @RequestParam(value = "createStartTime", required = false) LocalDateTime createStartTime,
                                                @RequestParam(value = "createEndTime", required = false) LocalDateTime createEndTime,
                                                @RequestParam(value = "limit", defaultValue = "20") int limit,
                                                @RequestParam(value = "page", defaultValue = "1") int page);

    @PostMapping("/api/order/trace/error/save")
    Response<String> saveErrorTrace(@RequestBody OrderErrorTrace callTrace);

    @GetMapping("/api/order/trace/thirdApi")
    PageTable<OrderThirdTrace> searchThirdCallTrace(@RequestParam(value = "coreCode", required = false) String coreCode,
                                                    @RequestParam(value = "callApiPath", required = false) String callApiPath,
                                                    @RequestParam(value = "sourcePlatform", required = false) String sourcePlatform,
                                                    @RequestParam(value = "targetPlatform", required = false) String targetPlatform,
                                                    @RequestParam(value = "createStartTime", required = false) LocalDateTime createStartTime,
                                                    @RequestParam(value = "createEndTime", required = false) LocalDateTime createEndTime,
                                                    @RequestParam(value = "limit", defaultValue = "20") int limit,
                                                    @RequestParam(value = "page", defaultValue = "1") int page);

    @PostMapping("/api/order/trace/thirdApi/save")
    Response<String> saveThirdCallTrace(@RequestBody OrderThirdTrace callTrace);

    /**
     * 主要是配合在折线图中会选择查询的接口，需要进行去重操作，只返回，callApiPath
     *
     * @param callApiPath
     * @param startTime   必填写
     * @param endTime     必填写
     * @param page
     * @param limit
     * @return
     */
    @GetMapping("/api/order/trace/thirdApi/pageQueryDistinctApiRequestInterface")
    PageTable<OrderThirdTrace> pageQueryDistinctApiRequestInterface(@RequestParam(value = "callApiPath", required = false) String callApiPath,
                                                                    @RequestParam(value = "startTime") LocalDateTime startTime,
                                                                    @RequestParam(value = "endTime") LocalDateTime endTime,
                                                                    @RequestParam(value = "page", defaultValue = "1") Integer page,
                                                                    @RequestParam(value = "limit", defaultValue = "20") Integer limit);

    /**
     * 只返回callStartTime，spendMillis,配合折线图组装数据
     *
     * @param callApiPath 类名
     * @param startTime   查询开始时间
     * @param endTime     查询结束时间
     * @return
     */
    @GetMapping("/api/order/trace/thirdApi/getStartTimeAndUsedTimeMillisList")
    Response<List<OrderThirdTrace>> getStartTimeAndUsedTimeMillisList(@RequestParam(value = "callApiPath", required = false) String callApiPath,
                                                                      @RequestParam(value = "startTime") LocalDateTime startTime,
                                                                      @RequestParam(value = "endTime") LocalDateTime endTime);

    /**
     * 返回耗时最大的10个接口，和平均耗时最大的10个接口
     *
     * @param startTime 查询开始时间，以startTime作为对比参考
     * @param endTime   查询结束时间，以startTime作为对比参考
     * @return
     */
    @GetMapping("/api/order/trace/thirdApi/statisticsMaxAndAverageUserTime")
    Response<ApiRequestLogStatisticsResponseDto> statisticsMaxAndAverageUserTimeApiList(@RequestParam(value = "startTime") LocalDateTime startTime,
                                                                                        @RequestParam(value = "endTime") LocalDateTime endTime);
    //微信支付记录 相关接口 start   ===========>

    /**
     * Query wx pay record by roco trade no and status response.
     *
     * @param rocoTradeNo  the roco trade no
     * @param wxTradeState the wx trade state
     * @return the response
     */
    @GetMapping("/api/order/wx-pay-record/queryByRocoTradeNoAndStatus")
    Response<WxPayRecord> queryWxPayRecordByRocoTradeNoAndStatus(@RequestParam("rocoTradeNo") String rocoTradeNo,
                                                                 @RequestParam(value = "wxTradeState", required = false) String wxTradeState);

    /**
     * Update wx pay record by id response.
     *
     * @param wxPayRecord the wx pay record
     * @return the response
     */
    @PostMapping("/api/order/wx-pay-record/updateById")
    Response updateWxPayRecordById(@RequestBody WxPayRecord wxPayRecord);

    /**
     * 后台管理订单支付记录分页查询列表  yinshihao加
     *
     * @param rocoTradeNo
     * @param wxTransactionId
     * @param wxUserOpenid
     * @param rocoResCode
     * @param rocoTradeType
     * @param wxTradeState
     * @param createTimeStart
     * @param createTimeEnd
     * @param wxTimeEndStart
     * @param wxTimeEndEnd
     * @param limit
     * @param page
     * @return
     */
    @GetMapping("/api/order/wx-pay-record/queryPage")
    PageTable<WxPayRecord> wxPayRecordPageTable(@RequestParam(value = "rocoTradeNo", required = false) String rocoTradeNo,
                                                @RequestParam(value = "wxTransactionId", required = false) String wxTransactionId,
                                                @RequestParam(value = "wxUserOpenid", required = false) String wxUserOpenid,
                                                @RequestParam(value = "rocoResCode", required = false) String rocoResCode,
                                                @RequestParam(value = "rocoTradeType", required = false) String rocoTradeType,
                                                @RequestParam(value = "wxTradeState", required = false) String wxTradeState,
                                                @RequestParam(value = "createTimeStart", required = false) String createTimeStart,
                                                @RequestParam(value = "createTimeEnd", required = false) String createTimeEnd,
                                                @RequestParam(value = "wxTimeEndStart", required = false) String wxTimeEndStart,
                                                @RequestParam(value = "wxTimeEndEnd", required = false) String wxTimeEndEnd,
                                                @RequestParam(value = "limit", defaultValue = "20") int limit,
                                                @RequestParam(value = "page", defaultValue = "1") int page);

    /**
     * 后台管理导出订单支付记录  yinshihao加
     *
     * @param rocoTradeNo
     * @param wxTransactionId
     * @param wxUserOpenid
     * @param rocoResCode
     * @param rocoTradeType
     * @param wxTradeState
     * @param createTimeStart
     * @param createTimeEnd
     * @param wxTimeEndStart
     * @param wxTimeEndEnd
     * @return
     */
    @GetMapping("/api/order/wx-pay-record/wxPayRecordList")
    Response<List<WxPayRecord>> wxPayRecordList(@RequestParam(value = "rocoTradeNo", required = false) String rocoTradeNo,
                                                @RequestParam(value = "wxTransactionId", required = false) String wxTransactionId,
                                                @RequestParam(value = "wxUserOpenid", required = false) String wxUserOpenid,
                                                @RequestParam(value = "rocoResCode", required = false) String rocoResCode,
                                                @RequestParam(value = "rocoTradeType", required = false) String rocoTradeType,
                                                @RequestParam(value = "wxTradeState", required = false) String wxTradeState,
                                                @RequestParam(value = "createTimeStart", required = false) String createTimeStart,
                                                @RequestParam(value = "createTimeEnd", required = false) String createTimeEnd,
                                                @RequestParam(value = "wxTimeEndStart", required = false) String wxTimeEndStart,
                                                @RequestParam(value = "wxTimeEndEnd", required = false) String wxTimeEndEnd);

    /**
     * 后台根据订单编码查询微信支付及退款详情 yinshihao加
     *
     * @param rocoTradeNo
     * @return
     */
    @GetMapping("/api/order/wx-pay-record/getWxPayRecordByTradeNo")
    Response<WxPayRecord> getWxPayRecordByTradeNo(@RequestParam(value = "rocoTradeNo") String rocoTradeNo);
    //微信支付记录 相关接口 end   ===========>

    //微信支付API 相关接口 start   ===========>

    /**
     * 统一下单(详见https://pay.weixin.qq.com/wiki/doc/api/jsapi.php?chapter=9_1)
     * 在发起微信支付前，需要调用统一下单接口，获取"预支付交易会话标识"
     * 接口地址：https://api.mch.weixin.qq.com/pay/unifiedorder
     *
     * @param paramDto the param dto
     * @return the response
     */
    @PostMapping("/api/order/pay/wx/unifiedOrder")
    Response<WxPayUnifiedOrderResultDto> unifiedWxPayOrder(@RequestBody WxPayUnifiedOrderParamDto paramDto);

    /**
     * 微信支付下单成功后未立即支付，再次调起支付接口
     *
     * @param outTradeNo roco车服的订单编号
     * @return response
     */
    @GetMapping("/api/order/pay/wx/repayOrder")
    Response<WxPayUnifiedOrderResultDto> repayWxPayOrder(@RequestParam("outTradeNo") String outTradeNo);

    /**
     * <pre>
     * 关闭订单
     * 应用场景
     * 以下情况需要调用关单接口：
     * 1. 商户订单支付失败需要生成新单号重新发起支付，要对原订单号调用关单，避免重复支付；
     * 2. 系统下单后，用户支付超时，系统退出不再受理，避免用户继续，请调用关单接口。
     * 注意：订单生成后不能马上调用关单接口，最短调用时间间隔为5分钟。
     * 接口地址：https://api.mch.weixin.qq.com/pay/closeorder
     * 是否需要证书：   不需要。
     * </pre>
     *
     * @param outTradeNo 商户系统内部的订单号
     * @return the response
     */
    @GetMapping("/api/order/pay/wx/closeOrder/{outTradeNo}")
    Response closeWxPayOrder(@PathVariable("outTradeNo") String outTradeNo);

    /**
     * <pre>
     * 微信支付-申请退款
     * 详见 https://pay.weixin.qq.com/wiki/doc/api/jsapi.php?chapter=9_4
     * 接口链接：https://api.mch.weixin.qq.com/secapi/pay/refund
     * </pre>
     *
     * @param refundParam 请求对象
     * @return the response
     */
    @PostMapping("/api/order/pay/wx/refund")
    Response<String> refund(@RequestBody WxPayRefundParamDto refundParam);

    //微信支付API 相关接口 end   ===========>

    //微信支付退款记录 相关接口 start   ===========>

    /**
     * Query wx pay refund record by roco refund no response.
     *
     * @param rocoRefundNo the roco refund no
     * @return the response
     */
    @GetMapping("/api/order/wx-pay-refund-record/queryByRocoRefundNo")
    Response<WxPayRefundRecord> queryWxPayRefundRecordByRocoRefundNo(@RequestParam("rocoRefundNo") String rocoRefundNo);

    /**
     * Update wx pay refund record by id response.
     *
     * @param wxPayRefundRecord the wx pay refund record
     * @return the response
     */
    @PostMapping("/api/order/wx-pay-refund-record/updateById")
    Response updateWxPayRefundRecordById(@RequestBody WxPayRefundRecord wxPayRefundRecord);

    /**
     * 后台管理订单支付退款记录分页查询列表  yinshihao加
     *
     * @param rocoTradeNo
     * @param rocoRefundNo
     * @param wxRefundId
     * @param wxTransactionId
     * @param wxUserOpenid
     * @param rocoResCode
     * @param rocoTradeType
     * @param wxRefundStatus
     * @param wxRefundRequestSource
     * @param createTimeStart
     * @param createTimeEnd
     * @param wxRefundSuccessTimeStart
     * @param wxRefundSuccessTimeEnd
     * @param limit
     * @param page
     * @return
     */
    @GetMapping("/api/order/wx-pay-refund-record/queryPage")
    PageTable<WxPayRefundRecord> wxPayRefundRecordPageTable(@RequestParam(value = "rocoTradeNo", required = false) String rocoTradeNo,
                                                            @RequestParam(value = "rocoRefundNo", required = false) String rocoRefundNo,
                                                            @RequestParam(value = "wxRefundId", required = false) String wxRefundId,
                                                            @RequestParam(value = "wxTransactionId", required = false) String wxTransactionId,
                                                            @RequestParam(value = "wxUserOpenid", required = false) String wxUserOpenid,
                                                            @RequestParam(value = "rocoResCode", required = false) String rocoResCode,
                                                            @RequestParam(value = "rocoTradeType", required = false) String rocoTradeType,
                                                            @RequestParam(value = "wxRefundStatus", required = false) String wxRefundStatus,
                                                            @RequestParam(value = "wxRefundRequestSource", required = false) String wxRefundRequestSource,
                                                            @RequestParam(value = "createTimeStart", required = false) String createTimeStart,
                                                            @RequestParam(value = "createTimeEnd", required = false) String createTimeEnd,
                                                            @RequestParam(value = "wxRefundSuccessTimeStart", required = false) String wxRefundSuccessTimeStart,
                                                            @RequestParam(value = "wxRefundSuccessTimeEnd", required = false) String wxRefundSuccessTimeEnd,
                                                            @RequestParam(value = "limit", defaultValue = "20") int limit,
                                                            @RequestParam(value = "page", defaultValue = "1") int page);

    /**
     * 后台管理导出订单支付退款记录列表  yinshihao加
     *
     * @param rocoTradeNo
     * @param rocoRefundNo
     * @param wxRefundId
     * @param wxTransactionId
     * @param wxUserOpenid
     * @param rocoResCode
     * @param rocoTradeType
     * @param wxRefundStatus
     * @param wxRefundRequestSource
     * @param createTimeStart
     * @param createTimeEnd
     * @param wxRefundSuccessTimeStart
     * @param wxRefundSuccessTimeEnd
     * @return
     */
    @GetMapping("/api/order/wx-pay-refund-record/wxPayRefundRecordList")
    Response<List<WxPayRefundRecord>> wxPayRefundRecordList(@RequestParam(value = "rocoTradeNo", required = false) String rocoTradeNo,
                                                            @RequestParam(value = "rocoRefundNo", required = false) String rocoRefundNo,
                                                            @RequestParam(value = "wxRefundId", required = false) String wxRefundId,
                                                            @RequestParam(value = "wxTransactionId", required = false) String wxTransactionId,
                                                            @RequestParam(value = "wxUserOpenid", required = false) String wxUserOpenid,
                                                            @RequestParam(value = "rocoResCode", required = false) String rocoResCode,
                                                            @RequestParam(value = "rocoTradeType", required = false) String rocoTradeType,
                                                            @RequestParam(value = "wxRefundStatus", required = false) String wxRefundStatus,
                                                            @RequestParam(value = "wxRefundRequestSource", required = false) String wxRefundRequestSource,
                                                            @RequestParam(value = "createTimeStart", required = false) String createTimeStart,
                                                            @RequestParam(value = "createTimeEnd", required = false) String createTimeEnd,
                                                            @RequestParam(value = "wxRefundSuccessTimeStart", required = false) String wxRefundSuccessTimeStart,
                                                            @RequestParam(value = "wxRefundSuccessTimeEnd", required = false) String wxRefundSuccessTimeEnd);

    //微信支付退款记录 相关接口 end   ===========>


    // ***************************** 商品订单 ranxiangfu start

    /**
     * 搜索商品订单
     *
     * @param searchParamDto the search param dto
     * @return page table
     */
    @GetMapping("/api/order/goods-order/adminSearch")
    PageTable<SaleOrder> adminSearchGoodsOrder(@SpringQueryMap GoodsAdminSearchParamDto searchParamDto);

    /**
     * 创建商品订单
     *
     * @param orderDto the order dto
     * @return the response
     */
    @PostMapping("/api/order/goods-order/create")
    Response<SaleOrder> createGoodsOrder(@RequestBody CreateGoodsOrderDto orderDto);

    /**
     * 根据订单编码查询提现订单信息.
     *
     * @param orderCode the order code
     * @return the withdrawals record by order code
     */
    @GetMapping("/api/order/goods-order/getByOrderCode")
    Response<GoodsOrder> getGoodsOrderByOrderCode(@RequestParam("orderCode") String orderCode);

    /**
     * 更改商品订单状态
     *
     * @param changeStatusDto the change status dto
     * @return withdrawals summary
     */
    @PostMapping("/api/order/goods-order/changeStatus")
    Response changeGoodsOrderStatus(@RequestBody GoodsOrderChangeStatusDto changeStatusDto);

    /**
     * 现金支付回调接口
     *
     * @param cashPayDto the cash pay dto
     * @return withdrawals summary
     */
    @PostMapping("/api/order/goods-order/cashPay")
    Response<String> payGoodsOrderByCash(@RequestBody OrderCashPayDto cashPayDto);

    /**
     * List goods order by order codes response.
     *
     * @param orderCodeList the order code list
     * @return the response
     */
    @GetMapping("/api/order/goods-order/listGoodsOrderByOrderCodes")
    Response<List<GoodsOrder>> listGoodsOrderByOrderCodes(@RequestParam("orderCodeList") List<String> orderCodeList);
    // ***************************** 商品订单 ranxiangfu end


    // -----------------------------liangyongtong 微信支付业务流水 start-----------------------------

    /**
     * 处理超时未支付订单失效
     *
     * @return
     */
    @GetMapping("/api/order/pay/wx/paymentflow/failure")
    Response taskToFailureForOrder();
    // -----------------------------liangyongtong 微信支付业务流水 end-----------------------------


    //-------------------------------huhailong 订单售后-----------

    /**
     * 管理后台查询，导出
     *
     * @param searchParamDto
     * @return
     */
    @GetMapping("/api/order/after/admin/adminAfterSearch")
    PageTable<OrderAfter> adminAfterSearch(@SpringQueryMap AdminAfterOrderSearchParamDto searchParamDto);

    /**
     * 订单审批
     *
     * @param pass
     * @param code
     * @param remark
     * @param updateUser
     * @return
     */
    @GetMapping("/api/order/after/admin/afterApproveCallBlack")
    Response afterApproveCallBlack(@RequestParam(value = "pass") String pass,
                                   @RequestParam(value = "code") String code,
                                   @RequestParam(value = "remark",required = false) String remark,
                                   @RequestParam(value = "updateUser") String updateUser);

    /**
     * 创建售后单,因为涉及到审批记录，所以后台审批记录应该是要在聚合中实现。
     *
     * @param adminCreateParamDto
     * @return 售后单号
     */
    @PostMapping("/api/order/after/admin/adminCreateAfter")
    Response<String> adminCreateAfter(@RequestBody AfterOrderAdminCreateParamDto adminCreateParamDto);

    /**
     * 退款，有两种情况：仅退款和退货退款
     *
     * @param refundParamDto
     * @return
     */
    @PostMapping("/api/order/after/admin/adminAfterRefund")
    Response adminAfterRefund(@RequestBody AfterOrderRefundParamDto refundParamDto);

    /**
     * 拒绝 受理
     *
     * @param refuseParamDto
     * @return
     */
    @PostMapping("/api/order/after/admin/adminAfterRefuse")
    Response adminAfterRefuse(@RequestBody AfterOrderRefuseParamDto refuseParamDto);

    /**
     * 售后填写快递 提交
     *
     * @param logisticsParamDto
     * @return
     */
    @PostMapping("/api/order/after/admin/adminAfterExchangeLogistics")
    Response adminAfterExchangeLogistics(@RequestBody AfterOrderAdminDeliveryLogisticsParamDto logisticsParamDto);


    /**
     * 得到订单售后日志
     *
     * @return
     */
    @GetMapping("/api/order/after/admin/getOrderAfterLog")
    Response<List<OrderAfterLog>> getOrderAfterLog(@RequestParam("afterCode") String afterCode);

    /**
     * 得到售后单主体，商品，售后的图片
     *
     * @param afterCode
     * @return
     */
    @GetMapping("/api/order/after/admin/adminAfterDetail")
    Response<OrderAfter> adminAfterDetail(@RequestParam("afterCode") String afterCode);


    /**
     * 主要是根据用户编号，售后单号 判断是否有权限
     * @param afterCode
     * @param userCode
     * @return
     */
    @GetMapping("/api/order/after/mall/getMainOrderAfter")
     Response<OrderAfter> getMainOrderAfter(@RequestParam("afterCode") String afterCode,
                                            @RequestParam("userCode")String userCode);

    /**
     * 创建售后单
     * @param createParamDto
     * @return
     */
     @PostMapping("/api/order/after/mall/mallCreateAfter")
     Response<String> mallCreateAfter(@RequestBody AfterOrderMallCreateParamDto createParamDto);


    /**
     * 用户填写了退货/换货的物流数据
     * @param logisticsParamDto
     * @return
     */
    @PostMapping("/api/order/after/mall/mallAfterReturnLogistics")
    Response mallAfterReturnLogistics(@RequestBody AfterOrderLogisticsParamDto logisticsParamDto);


    /**
     * 用户进行了物流售后确认
     * @param paramDto
     * @return
     */
    @PostMapping("/api/order/after/mall/mallAfterLogisticsConfirm")
    Response mallAfterLogisticsConfirm(@RequestBody AfterOrderOpeartorBaseParamDto paramDto);



    /**
     * 商城端搜索
     * @param userCode 会员编码
     * @param limit
     * @param page
     * @return
     */
    @GetMapping("/api/order/after/mall/mallAfterSearch")
    PageTable<OrderAfter> mallAfterSearch(@RequestParam("userCode") String userCode,
                                          @RequestParam(value = "limit", defaultValue = "10") Integer limit,
                                          @RequestParam(value = "page", defaultValue = "1") Integer page);


    /**
     * 得到售后单主体，商品，售后的图片,售后的处理节点
     * @param afterCode
     * @return
     */
    @GetMapping("/api/order/after/mall/mallAfterDetail")
    Response<OrderAfter> mallAfterDetail(@RequestParam("afterCode") String afterCode);


    //-------------------------------huhailong 订单售后end-----------


}

