package com.rocogz.syy.order.client;

import com.rocogz.syy.common.PageTable;
import com.rocogz.syy.common.response.Response;
import com.rocogz.syy.order.constant.CreateOrderModeEnum;
import com.rocogz.syy.order.constant.OrderStatisOption;
import com.rocogz.syy.order.constant.OtherInfoOption;
import com.rocogz.syy.order.dto.*;
import com.rocogz.syy.order.dto.after.*;
import com.rocogz.syy.order.dto.goods.*;
import com.rocogz.syy.order.dto.invoice.SelectAndUpdateStatusParam;
import com.rocogz.syy.order.dto.invoice.SubmitInvoiceParamDto;
import com.rocogz.syy.order.dto.jms.NotifyPayResultToSamsungMsg;
import com.rocogz.syy.order.dto.pay.*;
import com.rocogz.syy.order.dto.settle.order.SearchSettleItemParamDto;
import com.rocogz.syy.order.dto.settle.order.SettleOrderSearchDto;
import com.rocogz.syy.order.dto.settle.order.SettleSearchOrderItemResultDto;
import com.rocogz.syy.order.entity.after.OrderAfter;
import com.rocogz.syy.order.entity.after.OrderAfterLog;
import com.rocogz.syy.order.entity.after.OrderAfterProcessPrompt;
import com.rocogz.syy.order.entity.goods.GoodsOrder;
import com.rocogz.syy.order.entity.invoice.OrderInvoice;
import com.rocogz.syy.order.entity.orders.*;
import com.rocogz.syy.order.entity.pay.*;
import com.rocogz.syy.order.entity.qrcode.OrderSamsungQrcode;
import com.rocogz.syy.order.entity.settle.OrderSettleInfo;
import com.rocogz.syy.order.entity.settle.OrderSettleOrder;
import com.rocogz.syy.order.entity.settle.OrderSettleOrderLog;
import org.springframework.cloud.openfeign.SpringQueryMap;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import javax.validation.Valid;
import java.util.List;
import java.util.Map;

/**
 * 订单接口
 *
 * @author：zhangmin
 * @since：2020/3/25
 */
public interface IOrderClientService {

    /**
     * 创建订单
     *
     * @param order
     */
    @PostMapping("/api/order/create")
    Response createOrder(@RequestBody SaleOrder order);

    /**
     * 获取订单主信息
     *
     * @param orderCode 订单编号
     */
    @GetMapping("/api/order/getOrderMainInfo/{orderCode}")
    Response<SaleOrder> getOrderMainInfo(@PathVariable("orderCode") String orderCode);

    /**
     * 查询订单主信息,如果需要其他信息也一起加载
     *
     * @param orderCode     订单编号
     * @param otherInfoList 需要加载的其他信息选项,用来实现按需加载,如果需要全部其他信息,则传入 ALL
     **/
    @GetMapping("/api/order/getOrderWithOtherInfo/{orderCode}")
    Response<SaleOrder> getOrderWithOtherInfo(@PathVariable("orderCode") String orderCode,
                                              @RequestParam("otherInfoList") List<OtherInfoOption> otherInfoList);

    /**
     * 通过子订单号查询子订单详情：主订单，支付信息，收获地址信息一起加载
     * @param orderItemCode
     * @return
     */
    @GetMapping("/api/order/getOrderItemDetail/{orderItemCode}")
    Response<SaleOrder> getOrderItemDetail(@PathVariable("orderItemCode") String orderItemCode);

    /**
     * 只是获取子订单信息
     * @param orderItemCode
     * @return
     */
    @GetMapping("/api/order/getOrderItemBasic/{orderItemCode}")
    Response<GoodsOrder> getOrderItemBasicInfo(@PathVariable("orderItemCode") String orderItemCode);


    /**
     * 搜索福袋主订单
     */
    @GetMapping("/api/order/adminSearchSuiteMainOrder")
    PageTable<SaleOrder> adminSearchSuiteMainOrder(@SpringQueryMap GoodsAdminSearchParamDto searchParamDto);

    /**
     * 车主端小程序 搜索订单
     *
     * @param userCode
     * @return
     */
    @GetMapping("/api/order/weappSearch")
    PageTable<SaleOrder> weappSearch(@RequestParam("userCode") String userCode,
                                     @RequestParam(value = "status", required = false) String status,
                                     @RequestParam(value = "limit", defaultValue = "10") Integer limit,
                                     @RequestParam(value = "page", defaultValue = "1") Integer page);


    /**
     * 根据订单编号查询实物订单
     *
     * @param orderCodes 订单编号
     * @return
     */
    @PostMapping("/api/order/findOrderListByOrderCodes")
    Response<List<SaleOrder>> findOrderListByOrderCodes(@RequestBody List<String> orderCodes);


    /**
     * 统计订单数量，用来限制会员购买虚拟商品数量
     * @param orderCountParamDto orderCountParamDto
     */
    @PostMapping("/api/order/count")
    public Response<Map<String,OrderSummaryStatistic>> countOrder(@RequestBody List<OrderStatisOption> statisOptionList);

    /**
     * 统计每种状态订单的数量
     *
     * @param param 查询参数
     * @return
     */
    @PostMapping("/api/order/orderCountGroupByStatus")
    Response<List<OrderCountGroupByStatusDto>> orderCountGroupByStatus(@RequestBody OrderCountGroupByStatusParamDto param);

    /**
     * 按照手机Imei号统计订单下单数量，不包括取消的订单, 专享服务商品1个imei号最多购买1个专享订单
     * @param userCode 会员编号
     */
    @PostMapping("/api/order/countGroupByIMEI")
    Response<Map<String,OrderCountGroupByIMEIDto>> countOrderGroupByIMEI(@RequestBody OrderCountGroupByImeiParamDto param);

    /**
     * 逻辑删除订单
     *
     * @param orderCode 订单号
     * @param userCode  如果是weapp端调用,则需要传入用户编号,用来判断是否是车主自己的订单
     * @return
     */
    @GetMapping("/api/order/delete/{orderCode}")
    Response<String> deleteLogical(@PathVariable("orderCode") String orderCode,
                                   @RequestParam(value = "userCode", required = false) String userCode);

    /**
     * 关闭 超时未支付的订单
     *
     * @param orderCode 订单号
     * @return 如果关单成功, 则返回需要退回的用户券码
     */
    @GetMapping("/api/order/cancelOrder/{orderCode}")
    Response<CancelOrderResultDto> cancelOrder(@PathVariable("orderCode") String orderCode);


    /**
     * 更新订单的取消申请状态
     * @param orderCode 主订单号
     * @param newApplyStatus 取消申请状态 {@link OrderConstant.DictApplyStatus}
     * @return
     */
    @GetMapping("/api/order/updateCancelApplyStatus")
    Response updateOrderCancelApplyStatus(@RequestParam("orderCode") String orderCode,
                                          @RequestParam("newApplyStatus") String newApplyStatus);


    /**
     * 取消订单申请 审批操作后callback
     * @param code 主订单编号
     * @param pass 审核操作是否成功
     * @return
     */
    @GetMapping("/api/order/cancelOrderApproveCallBlack")
    Response cancelOrderApproveCallBlack(@RequestParam(value = "code") String code,
                                         @RequestParam(value = "pass") String pass);

    /**
     * 查询所有未开票的线下订单
     *
     * @param unionId 会员的unionId
     * @param limit   分页参数
     * @param page    分页参数
     * @return
     */
    @GetMapping("/api/order/getUninvoicedOfflineOrder")
    PageTable<SaleOrder> getUninvoicedOfflineOrder(@RequestParam(name = "unionId") String unionId,
                                                   @RequestParam(value = "limit",defaultValue = "10") int limit,
                                                   @RequestParam(value = "page",defaultValue = "1") int page);

    /**
     * 在后台管理对订单发起售后流程
     *
     * @param orderCode 订单号
     * @return response
     */
    @PostMapping("/api/order/orderAfterApplyCashRefund")
    Response<CancelOrderResultDto> orderAfterApplyCashRefund(@RequestBody OrderAfterRefundRequestParamDto requestParamDto);


        /**
         * 当会员注册时,更新迁移的三星订单会员账号信息
         * @param memberInfoDto
         * @return
         */
    @PostMapping("/api/order/updateUserBySamsungMemIdOnRegist")
    Response<String> updateUserInfoBySamsungMemIdOnRegist(@RequestBody SamsungMemberInfoDto memberInfoDto);

    /**
     * 订单的操作日志列表
     *
     * @param orderCode 订单编号
     * @return 返回结果按照id倒序排列
     */
    @GetMapping("/api/order/log/{orderCode}")
    Response<List<OrderLog>> findLogList(@PathVariable("orderCode") String orderCode);

    /**
     * 判断福袋商品是否被下过单
     *
     * @param suiteGoodCode 福袋商品的商品编码
     * @return
     */
    @GetMapping("/api/order/determineSuiteGoodsHasOrdered")
    Response<Boolean> determineSuiteGoodsHasOrdered(@RequestParam(name = "suiteGoodCode") String suiteGoodCode);

    /**
     * 判断普通商品是否被下过单
     *
     * @param generalGoodsCode 普通商品的商品编码
     * @return
     */
    @GetMapping("/api/order/goods-order/determineGeneralGoodsHasOrdered")
    Response<Boolean> determineGeneralGoodsHasOrdered(@RequestParam(name = "generalGoodsCode") String generalGoodsCode);

    /**
     * 根据code查询GoodsOrder，顺便带出SalesOrder信息
     *
     * @param orderCodes 主订单编号集合
     * @return
     */
    @PostMapping("/api/order/goods-order/findWithSaleOrderByCodes")
    Response<List<GoodsOrder>> findGoodsOrderWithSaleOrderByCodes(@RequestBody List<String> orderCodes);

    /**
     * 定时把过了过期时间的订单设置为 已取消
     *
     * @return
     */
    @GetMapping("/api/order/scheduleFindExpireOrderCodes")
    Response<List<String>> scheduleFindExpireOrderCodes();


    /**
     * 定时查找 发货后 7天 未确认收货的实物订单, 如果>7天 未收货 系统 自动确认收货订单
     * @return 返回自动确认收货的实物订单号列表
     */
    @GetMapping("/api/order/scheduleFindAndAutoCompleteOrder")
    Response<List<String>> scheduleFindAndAutoCompleteOrder();

    /**
     * 更新订单状态 & 保存操作日志
     * 注意：此接口只针对单个订单的状态更新
     *
     * @param logList （orderCode 与 afterStatus 必须传入）
     */
    @PostMapping("/api/order/writeOff/updateOrderStatus")
    Response<String> updateStatusWithSaveLog(@RequestBody List<OrderLog> logList);


    //微信支付记录 相关接口 start   ===========>

    /**
     * Query wx pay record by flow code and status response.
     *
     * @param flowCode     the flow code
     * @param wxTradeState the wx trade state
     * @return the response
     */
    @GetMapping("/api/order/wx-pay-record/queryByFlowCodeAndStatus")
    Response<WxPayRecord> queryWxPayRecordByFlowCodeAndStatus(@RequestParam("flowCode") String flowCode,
                                                                     @RequestParam(value = "wxTradeState", required = false) String wxTradeState);

    /**
     * Update wx pay record by id response.
     *
     * @param wxPayRecord the wx pay record
     * @return the response
     */
    @PostMapping("/api/order/wx-pay-record/updateById")
    Response updateWxPayRecordById(@RequestBody WxPayRecord wxPayRecord);

    /**
     * 后台管理订单支付记录分页查询列表  yinshihao加
     *
     * @param rocoTradeNo
     * @param wxTransactionId
     * @param wxUserOpenid
     * @param rocoResCode
     * @param rocoTradeType
     * @param wxTradeState
     * @param createTimeStart
     * @param createTimeEnd
     * @param wxTimeEndStart
     * @param wxTimeEndEnd
     * @param limit
     * @param page
     * @return
     */
    @GetMapping("/api/order/wx-pay-record/queryPage")
    PageTable<WxPayRecord> wxPayRecordPageTable(@RequestParam(value = "rocoTradeNo", required = false) String rocoTradeNo,
                                                @RequestParam(value = "wxTransactionId", required = false) String wxTransactionId,
                                                @RequestParam(value = "wxUserOpenid", required = false) String wxUserOpenid,
                                                @RequestParam(value = "rocoResCode", required = false) String rocoResCode,
                                                @RequestParam(value = "rocoTradeType", required = false) String rocoTradeType,
                                                @RequestParam(value = "wxTradeState", required = false) String wxTradeState,
                                                @RequestParam(value = "createTimeStart", required = false) String createTimeStart,
                                                @RequestParam(value = "createTimeEnd", required = false) String createTimeEnd,
                                                @RequestParam(value = "wxTimeEndStart", required = false) String wxTimeEndStart,
                                                @RequestParam(value = "wxTimeEndEnd", required = false) String wxTimeEndEnd,
                                                @RequestParam(value = "limit", defaultValue = "20") int limit,
                                                @RequestParam(value = "page", defaultValue = "1") int page);

    /**
     * 后台管理导出订单支付记录  yinshihao加
     *
     * @param rocoTradeNo
     * @param wxTransactionId
     * @param wxUserOpenid
     * @param rocoResCode
     * @param rocoTradeType
     * @param wxTradeState
     * @param createTimeStart
     * @param createTimeEnd
     * @param wxTimeEndStart
     * @param wxTimeEndEnd
     * @return
     */
    @GetMapping("/api/order/wx-pay-record/wxPayRecordList")
    Response<List<WxPayRecord>> wxPayRecordList(@RequestParam(value = "rocoTradeNo", required = false) String rocoTradeNo,
                                                @RequestParam(value = "wxTransactionId", required = false) String wxTransactionId,
                                                @RequestParam(value = "wxUserOpenid", required = false) String wxUserOpenid,
                                                @RequestParam(value = "rocoResCode", required = false) String rocoResCode,
                                                @RequestParam(value = "rocoTradeType", required = false) String rocoTradeType,
                                                @RequestParam(value = "wxTradeState", required = false) String wxTradeState,
                                                @RequestParam(value = "createTimeStart", required = false) String createTimeStart,
                                                @RequestParam(value = "createTimeEnd", required = false) String createTimeEnd,
                                                @RequestParam(value = "wxTimeEndStart", required = false) String wxTimeEndStart,
                                                @RequestParam(value = "wxTimeEndEnd", required = false) String wxTimeEndEnd);

    /**
     * 后台根据订单编码查询微信支付及退款详情 yinshihao加
     * 2020-11-26 ranxaingfu修改: 查支付记录时, 不查退款记录; 查退款记录时, 只查支付成功的微信支付记录
     *
     * @param rocoTradeNo     the roco trade no
     * @param wxTransactionId the wx transaction id
     * @param wxRefundId      the wx refund id
     * @return wx pay record by trade no
     */
    @GetMapping("/api/order/wx-pay-record/getWxPayRecord")
    Response<WxPayRecord> getWxPayRecord(@RequestParam(value = "rocoTradeNo") String rocoTradeNo,
                                         @RequestParam(value = "wxTransactionId", required = false) String wxTransactionId,
                                         @RequestParam(value = "wxRefundId", required = false) String wxRefundId);
    //微信支付记录 相关接口 end   ===========>

    //微信支付API 相关接口 start   ===========>

    /**
     * 统一下单(详见https://pay.weixin.qq.com/wiki/doc/api/jsapi.php?chapter=9_1)
     * 在发起微信支付前，需要调用统一下单接口，获取"预支付交易会话标识"
     * 接口地址：https://api.mch.weixin.qq.com/pay/unifiedorder
     *
     * @param paramDto the param dto
     * @return the response
     */
    @PostMapping("/api/order/pay/wx/unifiedOrder")
    Response<WxPayUnifiedOrderResultDto> unifiedWxPayOrder(@RequestBody WxPayUnifiedOrderParamDto paramDto);

    /**
     * 微信支付下单成功后未立即支付，再次调起支付接口
     *
     * @param paramDto the param dto
     * @return response response
     */
    @PostMapping("/api/order/pay/wx/repayOrder")
    Response<WxPayUnifiedOrderResultDto> repayWxPayOrder(@RequestBody WxPayRepayOrderParamDto paramDto);

    /**
     * <pre>
     * 关闭订单
     * 应用场景
     * 以下情况需要调用关单接口：
     * 1. 商户订单支付失败需要生成新单号重新发起支付，要对原订单号调用关单，避免重复支付；
     * 2. 系统下单后，用户支付超时，系统退出不再受理，避免用户继续，请调用关单接口。
     * 注意：订单生成后不能马上调用关单接口，最短调用时间间隔为5分钟。
     * 接口地址：https://api.mch.weixin.qq.com/pay/closeorder
     * 是否需要证书：   不需要。
     * </pre>
     *
     * @param outTradeNo 商户系统内部的订单号
     * @return the response
     */
    @GetMapping("/api/order/pay/wx/closeOrder/{outTradeNo}")
    Response closeWxPayOrder(@PathVariable("outTradeNo") String outTradeNo);

    /**
     * <pre>
     * 微信支付-申请退款
     * 详见 https://pay.weixin.qq.com/wiki/doc/api/jsapi.php?chapter=9_4
     * 接口链接：https://api.mch.weixin.qq.com/secapi/pay/refund
     * </pre>
     *
     * @param refundParam 请求对象
     * @return the response
     */
    @PostMapping("/api/order/pay/wx/refund")
    Response<String> refund(@RequestBody CashPayRefundParamDto refundParam);

    /**
     * 检查微信订单是否支付成功.
     *
     * @param appid         the appid
     * @param flowCode      the flow code
     * @param transactionId the transaction id
     * @return the response
     */
    @GetMapping("/api/order/pay/wx/checkOrderWxPaySuccess")
    Response<Boolean> checkOrderWxPaySuccess(@RequestParam("appid") String appid,
                                             @RequestParam(value = "flowCode", required = false) String flowCode,
                                             @RequestParam(value = "transactionId", required = false) String transactionId);


    //微信支付API 相关接口 end   ===========>

    //微信支付退款记录 相关接口 start   ===========>

    /**
     * Query wx pay refund record by roco refund no response.
     *
     * @param rocoRefundNo the roco refund no
     * @return the response
     */
    @GetMapping("/api/order/wx-pay-refund-record/queryByRocoRefundNo")
    Response<WxPayRefundRecord> queryWxPayRefundRecordByRocoRefundNo(@RequestParam("rocoRefundNo") String rocoRefundNo);

    /**
     * Query wx pay refund record by refund flow code response.
     *
     * @param refundFlowCode the refund flow code
     * @return the response
     */
    @GetMapping("/api/order/wx-pay-refund-record/queryByRefundFlowCode")
    Response<WxPayRefundRecord> queryWxPayRefundRecordByRefundFlowCode(@RequestParam("refundFlowCode") String refundFlowCode);

    /**
     * Update wx pay refund record by id response.
     *
     * @param wxPayRefundRecord the wx pay refund record
     * @return the response
     */
    @PostMapping("/api/order/wx-pay-refund-record/updateById")
    Response updateWxPayRefundRecordById(@RequestBody WxPayRefundRecord wxPayRefundRecord);

    /**
     * 后台管理订单支付退款记录分页查询列表  yinshihao加
     *
     * @param rocoTradeNo
     * @param rocoRefundNo
     * @param wxRefundId
     * @param wxTransactionId
     * @param wxUserOpenid
     * @param rocoResCode
     * @param rocoTradeType
     * @param wxRefundStatus
     * @param wxRefundRequestSource
     * @param createTimeStart
     * @param createTimeEnd
     * @param wxRefundSuccessTimeStart
     * @param wxRefundSuccessTimeEnd
     * @param limit
     * @param page
     * @return
     */
    @GetMapping("/api/order/wx-pay-refund-record/queryPage")
    PageTable<WxPayRefundRecord> wxPayRefundRecordPageTable(@RequestParam(value = "rocoTradeNo", required = false) String rocoTradeNo,
                                                            @RequestParam(value = "rocoRefundNo", required = false) String rocoRefundNo,
                                                            @RequestParam(value = "wxRefundId", required = false) String wxRefundId,
                                                            @RequestParam(value = "wxTransactionId", required = false) String wxTransactionId,
                                                            @RequestParam(value = "wxUserOpenid", required = false) String wxUserOpenid,
                                                            @RequestParam(value = "rocoResCode", required = false) String rocoResCode,
                                                            @RequestParam(value = "rocoTradeType", required = false) String rocoTradeType,
                                                            @RequestParam(value = "wxRefundStatus", required = false) String wxRefundStatus,
                                                            @RequestParam(value = "wxRefundRequestSource", required = false) String wxRefundRequestSource,
                                                            @RequestParam(value = "createTimeStart", required = false) String createTimeStart,
                                                            @RequestParam(value = "createTimeEnd", required = false) String createTimeEnd,
                                                            @RequestParam(value = "wxRefundSuccessTimeStart", required = false) String wxRefundSuccessTimeStart,
                                                            @RequestParam(value = "wxRefundSuccessTimeEnd", required = false) String wxRefundSuccessTimeEnd,
                                                            @RequestParam(value = "limit", defaultValue = "20") int limit,
                                                            @RequestParam(value = "page", defaultValue = "1") int page);

    /**
     * 后台管理导出订单支付退款记录列表  yinshihao加
     *
     * @param rocoTradeNo
     * @param rocoRefundNo
     * @param wxRefundId
     * @param wxTransactionId
     * @param wxUserOpenid
     * @param rocoResCode
     * @param rocoTradeType
     * @param wxRefundStatus
     * @param wxRefundRequestSource
     * @param createTimeStart
     * @param createTimeEnd
     * @param wxRefundSuccessTimeStart
     * @param wxRefundSuccessTimeEnd
     * @return
     */
    @GetMapping("/api/order/wx-pay-refund-record/wxPayRefundRecordList")
    Response<List<WxPayRefundRecord>> wxPayRefundRecordList(@RequestParam(value = "rocoTradeNo", required = false) String rocoTradeNo,
                                                            @RequestParam(value = "rocoRefundNo", required = false) String rocoRefundNo,
                                                            @RequestParam(value = "wxRefundId", required = false) String wxRefundId,
                                                            @RequestParam(value = "wxTransactionId", required = false) String wxTransactionId,
                                                            @RequestParam(value = "wxUserOpenid", required = false) String wxUserOpenid,
                                                            @RequestParam(value = "rocoResCode", required = false) String rocoResCode,
                                                            @RequestParam(value = "rocoTradeType", required = false) String rocoTradeType,
                                                            @RequestParam(value = "wxRefundStatus", required = false) String wxRefundStatus,
                                                            @RequestParam(value = "wxRefundRequestSource", required = false) String wxRefundRequestSource,
                                                            @RequestParam(value = "createTimeStart", required = false) String createTimeStart,
                                                            @RequestParam(value = "createTimeEnd", required = false) String createTimeEnd,
                                                            @RequestParam(value = "wxRefundSuccessTimeStart", required = false) String wxRefundSuccessTimeStart,
                                                            @RequestParam(value = "wxRefundSuccessTimeEnd", required = false) String wxRefundSuccessTimeEnd);

    //微信支付退款记录 相关接口 end   ===========>


    // ***************************** 商品订单 ranxiangfu start

    /**
     * 搜索商品订单
     *
     * @param searchParamDto the search param dto
     * @return page table
     */
    @GetMapping("/api/order/goods-order/adminSearch")
    PageTable<SaleOrder> adminSearchGoodsOrder(@SpringQueryMap GoodsAdminSearchParamDto searchParamDto);


    /**
     * 结算：搜索子订单列表,同时搜索 普通订单和福袋订单
     */
    @GetMapping("/api/order/goods-order/settleSearch")
    PageTable<SettleSearchOrderItemResultDto> settleSearchGoodsOrder(@SpringQueryMap SettleAdminSearchOrderItemParamDto searchParamDto);


    /**
     * 创建商品订单
     *
     * @param orderDto the order dto
     * @return the response
     */
    @PostMapping("/api/order/goods-order/create")
    Response<SaleOrder> createGoodsOrder(@RequestBody CreateGoodsOrderDto orderDto);

    /**
     * 根据订单编码查询提现订单信息.
     *
     * @param orderCode the order code
     * @return the withdrawals record by order code
     */
    @GetMapping("/api/order/goods-order/getByOrderCode")
    Response<GoodsOrder> getGoodsOrderByOrderCode(@RequestParam("orderCode") String orderCode);

    /**
     * 根据三星订单号获取实物订单信息
     */
    @GetMapping("/api/order/getOrderBySamsungOrderCode")
    Response<SaleOrder> getOrderBySamsungOrderCode(@RequestParam("samsungOrderCode") String samsungOrderCode,
                                                   @RequestParam(name = "createMode") CreateOrderModeEnum createMode);


    /**
     * 更改商品订单状态
     *
     * @param changeStatusDto the change status dto
     * @return withdrawals summary
     */
    @PostMapping("/api/order/goods-order/changeStatus")
    Response<UpdateStatusResultDto> changeGoodsOrderStatus(@RequestBody GoodsOrderChangeStatusDto changeStatusDto);

    /**
     * 更新实物订单的快递信息,订单状态更新为已发货（待发货或已发货状态都能更新物流信息）
     * @param changeStatusDto
     * @return 如果更新成功,返回订单之前的状态
     */
    @PostMapping("/api/order/goods-order/updateExpressInfo")
    Response<UpdateStatusResultDto> updateGoodOrderExpressInfo(@RequestBody GoodsOrderChangeStatusDto changeStatusDto);

    /**
     * 确认收获
     * @param confirmReceive
     */
    @PostMapping("/api/order/goods-order/confirmReceive")
    Response confirmReceiveChangeStatus(@RequestBody ConfirmReceiveChangeStatusDto confirmReceive);

    /**
     * 现金支付回调接口
     *
     * @param cashPayDto the cash pay dto
     * @return withdrawals summaryo
     */
    @PostMapping("/api/order/goods-order/cashPay")
    Response<String> payGoodsOrderByCash(@RequestBody OrderCashPayDto cashPayDto);

    /**
     * 积分支付订单
     */
    @PostMapping("/api/order/goods-order/centPay")
    Response<String> payGoodsOrderByCent(@RequestBody OrderCentPayDto centPayDto);


    /**
     * 更新支付结果 通知结果
     * @param notifyResult
     */
    @PostMapping("/api/order/goods-order/updateNotifySamsungPayResult")
    Response updateNotifySamsungPayResult(@RequestBody PayNotifyResultDto notifyResult);


    /**
     * 更新充值结果
     * @param resultDto
     * @return
     */
    @PostMapping("/api/order/goods-order/updateDepositResult")
    Response updateDepositResult(@RequestBody DepositResultDto resultDto);


    /**
     * 查询支付成功了但是待通知的 三星生活助手 订单
     * @param notifyResult
     */
    @GetMapping("/api/order/goods-order/findPendingNotifyGoodOrders")
    Response<List<SaleOrder>> findPendingNotifyGoodOrders();


    /**
     * 查询订单的 现金支付成功记录列表
     * @param orderCodes
     */
    @PostMapping("/api/order/goods-order/findOrderCashPayRecordList")
    Response<Map<String, OrderPaymentRecord>> findOrderCashPayRecordList(@RequestBody List<String> orderCodeList);


    /**
     * 手工推送订单 支付信息或券信息 到三星系统（只有支付成功以后的 管家服务商品 或  三星虚拟商品 才需要推送）
     */
    @GetMapping("/api/order/goods-order/manualPush")
    Response manualPushGoodOrder(@RequestParam("orderCode") String orderCode);


     /**
      * 充值订单充值失败后，尝试手工尝试充值
      */
     @GetMapping("/api/order/goods-order/manualDeposit")
     Response manualDeposit(@RequestParam("orderItemCode") String orderItemCode);


     @GetMapping("/api/order/goods-order/manualPushTest")
     Response<NotifyPayResultToSamsungMsg> manualPushGoodOrderTest(@RequestParam("orderCode") String orderCode);

    // ***************************** 商品订单 ranxiangfu end


    // -----------------------------liangyongtong 微信支付业务流水 start-----------------------------

    /**
     * 处理超时未支付订单失效
     *
     * @return
     */
    @GetMapping("/api/order/pay/wx/paymentflow/failure")
    Response taskToFailureForOrder();
    // -----------------------------liangyongtong 微信支付业务流水 end-----------------------------


    //-------------------------------huhailong 订单售后-----------

    /**
     * 管理后台查询，导出
     *
     * @param searchParamDto
     * @return
     */
    @GetMapping("/api/order/after/admin/adminAfterSearch")
    PageTable<OrderAfter> adminAfterSearch(@SpringQueryMap AdminAfterOrderSearchParamDto searchParamDto);

    /**
     * 订单审批
     *
     * @param pass
     * @param code
     * @param remark
     * @param updateUser
     * @return
     */
    @GetMapping("/api/order/after/admin/afterApproveCallBlack")
    Response afterApproveCallBlack(@RequestParam(value = "pass") String pass,
                                   @RequestParam(value = "code") String code,
                                   @RequestParam(value = "remark", required = false) String remark,
                                   @RequestParam(value = "updateUser") String updateUser);

    /**
     * 创建售后单,因为涉及到审批记录，所以后台审批记录应该是要在聚合中实现。
     *
     * @param adminCreateParamDto
     * @return 售后单号
     */
    @PostMapping("/api/order/after/admin/adminCreateAfter")
    Response<AfterOrderCreateResponseDto> adminCreateAfter(@RequestBody AfterOrderAdminCreateParamDto adminCreateParamDto);

    /**
     * 退款，有两种情况：仅退款和退货退款
     *
     * @param refundParamDto
     * @return
     */
    @PostMapping("/api/order/after/admin/adminAfterRefund")
    Response adminAfterRefund(@RequestBody AfterOrderRefundParamDto refundParamDto);

    /**
     * 拒绝 受理
     *
     * @param refuseParamDto
     * @return
     */
    @PostMapping("/api/order/after/admin/adminAfterRefuse")
    Response adminAfterRefuse(@RequestBody AfterOrderRefuseParamDto refuseParamDto);

    /**
     * 售后填写快递 提交
     *
     * @param logisticsParamDto
     * @return
     */
    @PostMapping("/api/order/after/admin/adminAfterExchangeLogistics")
    Response adminAfterExchangeLogistics(@RequestBody AfterOrderAdminDeliveryLogisticsParamDto logisticsParamDto);


    /**
     * 得到订单售后日志
     *
     * @return
     */
    @GetMapping("/api/order/after/admin/getOrderAfterLog")
    Response<List<OrderAfterLog>> getOrderAfterLog(@RequestParam("afterCode") String afterCode);

    /**
     * 得到售后单主体，商品，售后的图片
     *
     * @param afterCode
     * @return
     */
    @GetMapping("/api/order/after/admin/adminAfterDetail")
    Response<OrderAfter> adminAfterDetail(@RequestParam("afterCode") String afterCode);


    /**
     * 主要是根据用户编号，售后单号 判断是否有权限
     *
     * @param afterCode
     * @param userCode
     * @return
     */
    @GetMapping("/api/order/after/mall/getMainOrderAfter")
    Response<OrderAfter> getMainOrderAfter(@RequestParam("afterCode") String afterCode,
                                           @RequestParam("userCode") String userCode);

    /**
     * 创建售后单
     *
     * @param createParamDto
     * @return
     */
    @PostMapping("/api/order/after/mall/mallCreateAfter")
    Response<AfterOrderCreateResponseDto> mallCreateAfter(@RequestBody AfterOrderMallCreateParamDto createParamDto);


    /**
     * 用户填写了退货/换货的物流数据
     *
     * @param logisticsParamDto
     * @return
     */
    @PostMapping("/api/order/after/mall/mallAfterReturnLogistics")
    Response mallAfterReturnLogistics(@RequestBody AfterOrderLogisticsParamDto logisticsParamDto);


    /**
     * 用户进行了物流售后确认
     *
     * @param paramDto
     * @return
     */
    @PostMapping("/api/order/after/mall/mallAfterLogisticsConfirm")
    Response mallAfterLogisticsConfirm(@RequestBody AfterOrderOpeartorBaseParamDto paramDto);


    /**
     * 商城端搜索
     *
     * @param userCode 会员编码
     * @param limit
     * @param page
     * @return
     */
    @GetMapping("/api/order/after/mall/mallAfterSearch")
    PageTable<OrderAfter> mallAfterSearch(@RequestParam("userCode") String userCode,
                                          @RequestParam(value = "limit", defaultValue = "10") Integer limit,
                                          @RequestParam(value = "page", defaultValue = "1") Integer page);


    /**
     * 得到售后单主体，商品，售后的图片,售后的处理节点
     *
     * @param afterCode
     * @return
     */
    @GetMapping("/api/order/after/mall/mallAfterDetail")
    Response<OrderAfter> mallAfterDetail(@RequestParam("afterCode") String afterCode);


    /**
     * 获取未完成的售后数量
     *
     * @param userCode
     * @return
     */
    @GetMapping("/api/order/after/mall/mallNoFinishedAfterCount")
    Response<Integer> mallNoFinishedAfterCount(@RequestParam("userCode") String userCode);


    /**
     *  管理端售后节点的提示语
     * @return
     */
    @GetMapping("/api/order/after/admin/adminAfterPromptSearch")
    PageTable<OrderAfterProcessPrompt> adminAfterPromptSearch(@RequestParam(value = "limit") Integer limit,
                                                              @RequestParam(value = "page") Integer page);


    /**
     *  管理端更新售后节点提示语
     * @return
     */
    @PostMapping("/api/order/after/admin/adminUpdateAfterNodePrompt")
    Response  adminUpdateAfterNodePrompt(@RequestBody OrderAfterProcessPrompt processPrompt);


    //-------------------------------huhailong 订单售后end-----------

    //-------------------------------ranxiangfu 支付宝支付start-----------

    /**
     * Pay response.
     *
     * @param paramDto the param dto
     * @return the response
     */
    @PostMapping("/api/order/pay/alipay/pay")
    Response<String> alipayUnifiedOrder(@Valid @RequestBody AlipayUnifiedOrderParamDto paramDto);

    /**
     * Refund response.
     *
     * @param refundParam the refund param
     * @return the response
     */
    @PostMapping("/api/order/pay/alipay/refund")
    Response alipayRefundOrder(@Validated @RequestBody CashPayRefundParamDto refundParam);

    /**
     * Close order response.
     *
     * @param outTradeNo the out trade no
     * @return the response
     */
    @GetMapping("/api/order/pay/alipay/close/{outTradeNo}")
    Response alipayCloseOrder(@PathVariable("outTradeNo") String outTradeNo);
    //-------------------------------ranxiangfu 支付宝支付end-----------

    //-------------------------------ranxiangfu 支付宝支付记录start-----------
    /**
     * 根据流水号查询支付宝支付记录.
     *
     * @param flowCode the flow code
     * @return the alipay pay record by flow code
     */
    @GetMapping("/api/order/alipay-pay-record/getByFlowCode")
    Response<AlipayPayRecord> getAlipayPayRecordByFlowCode(@RequestParam("flowCode") String flowCode);

    /**
     * 更新或保存支付宝支付记录.
     *
     * @param alipayPayRecord the alipay pay record
     * @return the response
     */
    @PostMapping("/api/order/alipay-pay-record")
    Response saveOrUpdateAlipayPayRecord(@RequestBody AlipayPayRecord alipayPayRecord);
    //-------------------------------ranxiangfu 支付宝支付记录end-----------

    //-------------------------------ranxiangfu 支付宝退款记录start-----------
    /**
     * 根据退款流水号查询支付宝退款记录.
     *
     * @param flowCode the flow code
     * @return the alipay refund record by flow code
     */
    @GetMapping("/api/order/alipay-refund-record/getByFlowCode")
    Response<AlipayRefundRecord> getAlipayRefundRecordByFlowCode(@RequestParam("flowCode") String flowCode);

    /**
     * 保存或更新支付宝退款记录.
     *
     * @param alipayRefundRecord the alipay refund record
     * @return the response
     */
    @PostMapping("/api/order/alipay-refund-record")
    Response saveOrUpdateAlipayRefundRecord(@RequestBody AlipayRefundRecord alipayRefundRecord);
    //-------------------------------ranxiangfu 支付宝退款记录end-----------

    //-------------------------------ranxiangfu 支付流水start-----------
    /**
     * 根据支付流水号查询支付流水信息.
     *
     * @param flowCode the flow code
     * @return the order payment flow by flow code
     */

    @GetMapping("/api/order/order-payment-flow/getByFlowCode")
    Response<OrderPaymentFlow> getOrderPaymentFlowByFlowCode(@RequestParam("flowCode") String flowCode);

    /**
     * 更新或保存支付流水信息.
     *
     * @param orderPaymentFlow the order payment flow
     * @return the response
     */
    @PostMapping("/api/order/order-payment-flow")
    Response saveOrUpdateOrderPaymentFlow(@RequestBody OrderPaymentFlow orderPaymentFlow);
    //-------------------------------ranxiangfu 支付流水end-----------

    //-------------------------------ranxiangfu 退款流水start-----------
    /**
     * 根据流水号获取退款流水信息.
     *
     * @param flowCode the flow code
     * @return the order refund flow by flow code
     */
    @GetMapping("/api/order/order-refund-flow/getByFlowCode")
    Response<OrderRefundFlow> getOrderRefundFlowByFlowCode(@RequestParam("flowCode") String flowCode);

    /**
     * 保存或更新退款流水.
     *
     * @param orderRefundFlow the order refund flow
     * @return the response
     */
    @PostMapping("/api/order/order-refund-flow")
    Response saveOrUpdateOrderRefundFlow(@RequestBody OrderRefundFlow orderRefundFlow);
    //-------------------------------ranxiangfu 退款流水end-----------

    /************************************  zouzhicai 2020-11-25 10:37:02 二维码及参数记录表 START   ****************************************************************/
    /**
     * 保存二维码及相关参数
     * @param orderSamsungQrcode
     * @return
     */
    @PostMapping("/api/order/qrcode/saveQrcode")
    Response saveQrcode(@RequestBody OrderSamsungQrcode orderSamsungQrcode);

    /**
     * 根据场景码查询二维码及相关参数
     * @param sceneNo 订单号
     * @return response
     */
    @GetMapping("/api/order/qrcode/query/{sceneNo}")
    Response<OrderSamsungQrcode> getQrcode(@PathVariable("sceneNo") String sceneNo);

    /************************************  zouzhicai 2020-11-25 10:37:02 二维码及参数记录表 END   ****************************************************************/

    //-------------------------------huhailong 退款记录begin-----------

    /**
     * 获取订单退款记录
     *
     * @param orderCode
     * @return
     */
    @RequestMapping("/api/order/orderRefundByOrderCode")
     Response<OrderRefund> getOrderRefundByOrderCode(@RequestParam("orderCode") String orderCode);


    /**
     * 得到基础售后信息
     * @param afterCode
     * @return
     */
    @GetMapping("/api/order/after/admin/getBaseAfterInfoByAfterCode")
    Response<OrderAfter> getBaseAfterInfoByAfterCode(@RequestParam("afterCode") String afterCode);
    //-------------------------------huhailong 退款记录end-----------

    /**
     * 当虚拟卡密导入后,触发符合条件的订单发货消息
     * @param
     * @return
     */
     @PostMapping("/api/order/goods-order/sendVirtualGoodDeliverJms")
     Response sendVirtualGoodDeliverJms(@RequestBody List<String> skuCodeList);



    //-------------------------------zhangmin 三星订单导入报告 start -----------

    /**
     * 保存三星订单导入报告
     * @param batch
     * @return
     */
    @PostMapping("/api/order/importSamsung/report/save")
    Response saveSamsungOrderImportReport(@RequestBody ImportSamsungOrderBatch batch);

    @PostMapping("/api/order/importSamsung/report/update")
    Response updateSamsungOrderImportReport(@RequestBody ImportSamsungOrderBatch batch);

    /**
     * 查询所有导入报告
     * @param batch
     * @return
     */
    @GetMapping("/api/order/importSamsung/report/list")
    Response<List<ImportSamsungOrderBatch>> findImportReportList();


    @GetMapping("/api/order/importSamsung/report/{batchCode}")
    Response<ImportSamsungOrderBatch> getReportByBatchCode(@PathVariable("batchCode") String batchCode,
                                                           @RequestParam(value = "status", defaultValue = "0") Integer status,
                                                           @RequestParam(value = "isLoadItem", defaultValue = "true") Boolean isLoadItem);

    @PostMapping("/api/order/importSamsung/report/saveFailItemList")
    Response saveSamsungOrderImportFailItemList(@RequestBody List<ImportSamsungOrderFailItem> saveFailItemList);


    @PostMapping("/api/order/importSamsung/report/batchUpdateItemResult")
    Response batchUpdateSamsungOrderImportResultList(@RequestBody List<ImportSamsungOrderFailItem> updateImportResultList);

    //--------------------- 开票管理相关start -------------------/

    /**
     * 分页查询开票申请
     *
     * @param applyCode    申请编号
     * @param invoiceTitle 抬头
     * @param invoiceNo    税号
     * @param startTime    申请开始时间
     * @param endTime      申请结束时间
     * @param mobile       手机号
     * @param email        邮箱
     * @param invoiceType  发票类型
     * @param titleType    抬头类型
     * @param status       状态
     * @param limit        分页参数
     * @param page         分页参数
     * @return
     */
    @GetMapping("/api/order/invoice")
    PageTable<OrderInvoice> searchInvoice(@RequestParam(name = "applyCode", required = false) String applyCode,
                                          @RequestParam(name = "invoiceTitle", required = false) String invoiceTitle,
                                          @RequestParam(name = "invoiceNo", required = false) String invoiceNo,
                                          @RequestParam(name = "startTime", required = false) String startTime,
                                          @RequestParam(name = "endTime", required = false) String endTime,
                                          @RequestParam(name = "mobile", required = false) String mobile,
                                          @RequestParam(name = "email", required = false) String email,
                                          @RequestParam(name = "invoiceType", required = false) String invoiceType,
                                          @RequestParam(name = "titleType", required = false) String titleType,
                                          @RequestParam(name = "status", required = false) String status,
                                          @RequestParam(name = "limit", defaultValue = "20") int limit,
                                          @RequestParam(name = "page", defaultValue = "1") int page);

    /**
     * 根据unionId分页查询开票信息
     *
     * @param unionId unionId
     * @param limit   分页参数
     * @param page    分页参数
     * @return
     */
    @GetMapping("/api/order/invoice/searchInvoiceByUnionId")
    PageTable<OrderInvoice> searchInvoiceByUnionId(@RequestParam(name = "unionId") String unionId,
                                                          @RequestParam(name = "limit", defaultValue = "20") int limit,
                                                          @RequestParam(name = "page", defaultValue = "1") int page);

    /**
     * 根据申请编号查询发票申请详情
     *
     * @param applyCode 申请编号
     * @return
     */
    @GetMapping("/api/order/invoice/getInvoiceByApplyCode")
    Response<OrderInvoice> getInvoiceDetailByApplyCode(@RequestParam(name = "applyCode") String applyCode);


    /**
     * 提交开票请求
     * @param param 开票请求参数
     * @return
     */
    @PostMapping("/api/order/invoice/submitInvoice")
    Response<String> submitInvoice(@RequestBody SubmitInvoiceParamDto param);

    /**
     * 批量查询并更新状态，后台批量导出使用
     *
     * @return
     */
    @PostMapping("/api/order/invoice/batchSelectAndUpdateStatus")
    Response<List<OrderInvoice>> batchSelectAndUpdateInvoiceStatus(@RequestBody SelectAndUpdateStatusParam param);

    /**
     * 查询单条记录并更新状态，后台单条记录导出使用
     *
     * @return
     */
    @PostMapping("/api/order/invoice/singleSelectAndUpdateStatus")
    Response<OrderInvoice> singleSelectAndUpdateInvoiceStatus(@RequestBody SelectAndUpdateStatusParam param);

    /**
     * 批量更新开票申请状态
     *
     * @param param 请求参数
     * @return
     */
    @PostMapping("/api/order/invoice/batchUpdateStatus")
    Response batchUpdateInvoiceStatus(@RequestBody SelectAndUpdateStatusParam param);
    //--------------------- 开票管理相关 end -------------------/




    //--------------------- 结算相关 start -------------------/

    /**
     * 结算订单分页查询
     * @return
     */
    @PostMapping("/api/order/settle/settleOrder/pageSearch")
    PageTable<OrderSettleOrder> settleOrderPageSearch(@RequestBody SettleOrderSearchDto searchParam);



    /**
     * 获得结算单信息,仅仅包含结算单信息
     */
    @GetMapping("/api/order/settle/settleOrder/{settleOrderCode}")
    Response<OrderSettleOrder> getSettleOrderBasicInfo(@PathVariable("settleOrderCode") String settleOrderCode);

    /**
     * 第一步：在提交之前，"添加查询到的订单" 时创建结算单
     *
     * @param searchParamDto
     */
    @GetMapping("/api/order/settle/settleOrder/createUpdateBeforeSubmit")
    Response<String> createUpdateCommonSettleOrderBeforeSubmit(@SpringQueryMap SettleAdminSearchOrderItemParamDto searchParamDto);


    /**
     * 分页搜索结算单下的 明细item
     */
    @GetMapping("/api/order/settle/settleOrder/item/search")
    PageTable<SettleSearchOrderItemResultDto> searchSettleItemPage(@SpringQueryMap SearchSettleItemParamDto paramDto);


    /**
     * 删除结算单下的 某个结算单item
     * @param settleOrderCode 结算单编号
     * @param itemId 结算单ItemId
     */
    @GetMapping("/api/order/settle/settleOrder/deleteByItemId")
    Response deleteSettleItemByItemId(@RequestParam("settleOrderCode") String settleOrderCode, @RequestParam("itemId") Integer itemId);


    /**
     * 结算订单操作日志
     * @param settleOrderCode 结算单号
     * @param limit
     * @param page
     * @return
     */
    @GetMapping("/api/order/settle/settleOrder/log/pageSearch")
    PageTable<OrderSettleOrderLog> settleOrderLogPageSearch(@RequestParam(value = "settleOrderCode") String settleOrderCode,
                                                            @RequestParam(value = "limit", defaultValue = "20") int limit,
                                                            @RequestParam(value = "page", defaultValue = "1") int page);


    /**
     * 得到结算单详情,用在结算单详情页面上
     * @param settleOrderCode 结算订单号
     * @return
     */
    @GetMapping("/api/order/settle/settleOrder/fullDetail/{settleOrderCode}")
    Response<OrderSettleOrder> getSettleOrderAndSettleInfoAndAttachment(@PathVariable("settleOrderCode") String settleOrderCode);

    /**
     * 供应商或者运营商结算
     *
     * @param settleInfo 结算信息
     * @return
     */
    @PostMapping("/api/order/settle/settleOrder/supplierOrOperatorSettle")
    Response supplierOrOperatorSettle(@RequestBody OrderSettleInfo settleInfo);

    //--------------------- 结算相关 end -------------------/


}

