package com.rocogz.syy.order.dto.settle.order;

import com.rocogz.syy.order.constant.OrderConstant;
import com.rocogz.syy.order.entity.goods.GoodsOrder;
import com.rocogz.syy.order.entity.orders.SaleOrder;
import com.rocogz.syy.order.util.ArithUtil;
import lombok.Getter;
import lombok.Setter;
import org.springframework.beans.BeanUtils;

import java.math.BigDecimal;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.util.Optional;

/**
 * 结算：搜索订单item结果
 */
@Setter
@Getter
public class SettleSearchOrderItemResultDto {

    /**
     * 结算单itemId
     *
     * @see OrderSettleOrderItem#id
     */
    private Integer settleOrderItemId;

    /**
     * 主订单编号
     */
    private String orderCode;

    /**
     * 子订单编号
     */
    private String orderItemCode;

    /**
     * 主商品编号
     */
    private String goodCode;

    private String skuCode;

    private String originalSkuCode;

    /**
     * 实物商品erp编号
     */
    private String erpCode;

    /**
     * 主商品名称
     */
    private String goodName;

    /**
     * 是否虚拟商品
     */
    private Boolean virtual;

    /**
     * 商品类型
     */
    private String virtualType;

    private String virtualTypeLabel;

    private String payWay;

    private String payWayLabel;

    /**
     * 是否三星专区商品
     */
    private Boolean samsungPrefecture;

    /**
     * 现金支付金额
     */
    private BigDecimal cashAmount;

    /**
     * 积分支付金额
     */
    private BigDecimal centAmount;

    /**
     * 商品实际金额=商品结算价*数量
     */
    private BigDecimal realAmount;

    /**
     * 通道费
     */
    private BigDecimal cashPayFee;

    /**
     * 分期手续费
     */
    private BigDecimal hbFqFee;


    private BigDecimal taxRate;

    //物流费用
    private BigDecimal logisticFee;


    /**
     * 现金支付手续费(精算手续费) = 通道费 + 分期手续费
     */
    private BigDecimal actuaryFee;

    /**
     * 订单现金支付方式
     * @see SaleOrder#cashPayWay
     */
    private String cashPayWay;

    private BigDecimal realSettlementPrice;

    private Integer quantity;


    //支付流水号
    private String payCredential;

    /**
     * 三星会员手机号
     */
    private String samsungMobile;

    /**
     * 三星会员名字
     */
    private String samsungMemName;


    /**
     * 下单时间
     */
    private LocalDateTime createTime;

    /**
     * 完成时间
     */
    private LocalDateTime finishTime;

    /**
     * 充值状态
     */
    private String depositStatus;
    private String depositStatusLabel;

    /**
     * 子订单状态
     */
    private String status;

    private String statusLabel;


    /**
     * 供应商结算状态
     */
    private String supplierSettleStatus;
    private String supplierSettleStatusLabel;

    /**
     * 运营商结算状态
     */
    private String operatorSettleStatus;
    private String operatorSettleStatusLabel;


    //获得下单日期
    public LocalDate getCreateDate() {
        if (createTime == null) {
            return null;
        }

        return createTime.toLocalDate();
    }


    public String getVirtualType() {
        if (Boolean.FALSE.equals(virtual)) {
            return OrderConstant.DictVirtualType.REAL;
        }
        return virtualType;
    }


    /**
     * @return 是否花呗分期
     */
    public boolean isHbfq() {
        if (OrderConstant.DictPayWay.HBPCREDIT.equals(this.cashPayWay)) {
            return true;
        }
        return false;
    }



    /**
     * 实际精算费用 = real_amount-cash_amount
     */
    public BigDecimal getActuaryAmount() {
        return ArithUtil.bigDecimalSubtract(this.realAmount, this.cashAmount);
    }

    /**
     * 现金支付(扣除支付费率和分期手续费)= cash_amount-actuary_fee
     *
     * @return
     */
    public BigDecimal getDeductCashAmount() {
        return ArithUtil.bigDecimalSubtract(this.cashAmount, this.actuaryFee);
    }

    /**
     * 需要精算费用= actuary_amount+actuary_fee+logistic_fee
     */
    public BigDecimal getNeedActuaryAmount() {
        double needActuaryAmt = getActuaryAmount().doubleValue() + actuaryFee.doubleValue() +
                Optional.ofNullable(logisticFee).orElse(BigDecimal.ZERO).doubleValue();
        return ArithUtil.roundToBigDecimal(needActuaryAmt, OrderConstant.SCALE);
    }


    public static SettleSearchOrderItemResultDto valueOf(SaleOrder order) {

        GoodsOrder goodOrder = order.getGoodsOrder();

        SettleSearchOrderItemResultDto resultDto = new SettleSearchOrderItemResultDto();


        BeanUtils.copyProperties(goodOrder, resultDto);

        if (resultDto.getLogisticFee() == null) {
            resultDto.setLogisticFee(BigDecimal.ZERO);
        }

        double realAmt = Optional.ofNullable(goodOrder.getRealSettlementPrice()).orElse(BigDecimal.ZERO).doubleValue() * goodOrder.getQuantity();

        resultDto.setRealAmount(ArithUtil.roundToBigDecimal(realAmt, OrderConstant.SCALE));
        resultDto.setPayWay(order.getPayWay());
        resultDto.setCashPayWay(order.getCashPayWay());
        resultDto.setActuaryFee(order.getTotalCashFee());
        resultDto.setCashAmount(goodOrder.getTotalCash());
        resultDto.setCentAmount(goodOrder.getTotalCent());
        resultDto.setHbFqFee(Optional.ofNullable(order.getHbFqFee()).orElse(BigDecimal.ZERO));
        resultDto.setCashPayFee(order.getCashFee());


        return resultDto;
    }

}
