package com.rocogz.syy.order.entity.orders;


import com.baomidou.mybatisplus.annotation.TableName;
import com.baomidou.mybatisplus.core.toolkit.StringUtils;
import com.rocogz.syy.common.entity.IdEntity;
import com.rocogz.syy.order.constant.OrderConstant;
import com.rocogz.syy.order.entity.goods.GoodsOrder;
import lombok.Getter;
import lombok.Setter;
import lombok.experimental.Accessors;

import java.math.BigDecimal;
import java.time.LocalDate;
import java.time.format.DateTimeFormatter;

/**
 * 每日推送的折扣订单临时数据中间表
 * 当天支付的当天又取消的订单不需要推送
 * 按日batch推送：前一日使用coupon码的数据（新订单+变更订单）
 * - 文件名：: pm_coupon_yymmdd.txt (yymmdd为Day-1,凌晨4点前生成前一日订单，例如 4/14 凌晨生成文件名为 pm_coupon_220413.txt)
 * 推送 折扣兑换码订单信息dto
 * 推送订单格式如下：
 * 订单号|订单状态|订单日期|订单时间|购买产品名称|购买产品ITEM|产品金额|Coupon码|Coupon抵扣金额|积分抵扣金额|积分订单号|实际支付金额|会员账号
 * @author zhangmin
 * */
@TableName("order_push_discount_coupon_order")
@Setter @Getter @Accessors(chain = true)
public class PushDiscountCouponOrder extends IdEntity {

    public static final String CANCELED_STATUS = "CNCL";

    public static final String SOLD_STATUS = "SOLD";

    public static final String SEPERATOR = "|";

    /**
     * 数据日期：
     * 例如 4/14凌晨生成文件名为 pm_coupon_220413.txt,则dataDate = 2022-04-03
     */
    private LocalDate dataDate;

    /**
     * 主订单Id
     */
    private Integer orderId;

    /**
     * 订单号
     */
    private String orderCode;

    /**
     * 三星那边的订单状态：SOLD/CNCL
     */
    private String orderStatus;

    /**
     * 订单日期：日期格式2022-04-08
     */
    private String orderDate;

    /**
     * 订单时间：13:41:29
     */
    private String orderTime;

    /**
     * 产品名称
     */
    private String goodName;

    /**
     * 三星产品itemCode，例如：EF-RG988CBEGCN,折扣兑换码活动推送订单 需要传入此字段
     */
    private String samsungItemCode;


    /**
     * 产品金额： 现金以分为单位传递：例如 5现金 + 1000积分，则 totalMoney = 1500
     * 产品总金额 = GoodOrder#cent + GoodOrder#cash
     */
    private String totalMoney;

    /**
     * 折扣Coupon码
     */
    private String discountRedeemCode;


    /**
     * Coupon抵扣金额,以分为单位
     * cutDownMoney= SaleOrder#cutDownCash + SaleOrder#cutDownCent
     */
    private String cutDownMoney;


    /**
     * 积分抵扣金额 = com.rocogz.syy.order.entity.orders.SaleOrder#totalCent
     */
    private String totalCent;

    /**
     * 积分订单号 = orderCode
     */
    private String centOrderCode;


    /**
     * 实际支付金额，就是实际支付现金，以分为单位
     * payCashMoney =  SaleOrder#totalCash = (totalMoney - cutDownMoney - totalCent)
     */
    private String payCashMoney;


    /**
     * 会员帐户，即三星会员帐户code
     * @see GoodsOrder#samsungMemId
     */
    private String samsungMemCode;


    /**
     * 折扣码兑换码订单 通过文本文件OSS方式 推送到三星状态
     * @see OrderConstant.DictOssPushOrderStatus#PENDING
     */
    private String discountPushStatus;



    public static PushDiscountCouponOrder valueOf(GoodsOrder goodOrder) {

        BigDecimal HUNDRED =  new BigDecimal(100);

        SaleOrder order = goodOrder.getSaleOrder();

        PushDiscountCouponOrder pushOrderDto = new PushDiscountCouponOrder();

        pushOrderDto.setOrderId(order.getId());
        pushOrderDto.setDiscountPushStatus(order.getDiscountPushStatus());
        pushOrderDto.setOrderCode(goodOrder.getOrderCode());

        if(OrderConstant.DictOrderStatus.CANCELED.equalsIgnoreCase(order.getOrderStatus())) {
            //如果是订单先支付完毕，然后再取消订单(例如通过售后取消订单的)
            pushOrderDto.setOrderStatus(CANCELED_STATUS);
        }else {
            pushOrderDto.setOrderStatus(SOLD_STATUS);
        }

        pushOrderDto.setOrderDate(order.getCreateTime().toLocalDate().toString());


        DateTimeFormatter df = DateTimeFormatter.ofPattern("HH:mm:ss");

        pushOrderDto.setOrderTime(df.format(order.getCreateTime().toLocalTime()));

        pushOrderDto.setGoodName(goodOrder.getGoodName());

        String samItemCode = StringUtils.isNotBlank(goodOrder.getSamsungItemCode()) ? goodOrder.getSamsungItemCode() : goodOrder.getGoodCode();

        pushOrderDto.setSamsungItemCode(samItemCode);

        //产品总金额 单位：分
        BigDecimal totalMoney = goodOrder.getCash().multiply(HUNDRED).add(goodOrder.getCent()).multiply(new BigDecimal(goodOrder.getQuantity()));
        //Coupon抵扣金额,以分为单位
        BigDecimal cutdownMoney = order.getCutDownCash().multiply(HUNDRED).add(order.getCutDownCent());

        //实际支付的现金
        BigDecimal payCashMoney = order.getTotalCash().multiply(HUNDRED);

        pushOrderDto.setTotalMoney(String.valueOf(totalMoney));

        pushOrderDto.setDiscountRedeemCode(order.getDiscountRedeemCode());
        pushOrderDto.setCutDownMoney(String.valueOf(cutdownMoney));
        pushOrderDto.setTotalCent(String.valueOf(order.getTotalCent()));
        pushOrderDto.setCentOrderCode(goodOrder.getOrderCode());
        pushOrderDto.setPayCashMoney(String.valueOf(payCashMoney));
        pushOrderDto.setSamsungMemCode(goodOrder.getSamsungMemId());
        return pushOrderDto;
    }

    @Override
    public String toString() {

        StringBuffer orderInfoBuf = new StringBuffer();

        orderInfoBuf.append(orderCode).append(SEPERATOR)
                    .append(orderStatus).append(SEPERATOR)
                    .append(orderDate).append(SEPERATOR)
                    .append(orderTime).append(SEPERATOR)
                    .append(goodName).append(SEPERATOR)
                    .append(samsungItemCode).append(SEPERATOR)
                    .append(totalMoney).append(SEPERATOR)
                    .append(discountRedeemCode).append(SEPERATOR)
                    .append(cutDownMoney).append(SEPERATOR)
                .append(totalCent).append(SEPERATOR)
                .append(centOrderCode).append(SEPERATOR)
                .append(payCashMoney).append(SEPERATOR)
                .append(samsungMemCode);

        return orderInfoBuf.toString();

    }
}