package com.rocogz.syy.user.client;

import com.rocogz.syy.common.PageTable;
import com.rocogz.syy.common.response.Response;
import com.rocogz.syy.user.dto.UserMsgWebDto;
import com.rocogz.syy.user.entity.UserLoginLog;
import com.rocogz.syy.user.entity.UserMsg;
import com.rocogz.syy.user.entity.delivery.address.UserDeliveryAddress;
import com.rocogz.syy.user.entity.info.SamsungMemInfo;
import com.rocogz.syy.user.entity.info.UserBasicInfo;
import org.springframework.web.bind.annotation.*;

import java.util.List;

/**
 * The interface User client service.
 *
 * @author：liangyongtong
 * @since：2020/3/25
 */
public interface IUserClientService {
    // =========  start of  用户信息 by ranxiangfu   ===========
    /**
     * 分页查询用户信息.
     *
     * @param userCode  用户编码
     * @param openid    wx openid
     * @param wxName    微信昵称
     * @param mobile    手机号码
     * @param status    状态
     * @param channel   注册渠道
     * @param startTime 注册起始时间
     * @param endTime   注册结束时间
     * @param page      the page
     * @param limit     the limit
     * @return the page table
     */
    @GetMapping("/user/user-basic-info/pageQuery")
    PageTable<UserBasicInfo> pageQueryUserBasicInfo(@RequestParam(value = "userCode", required = false) String userCode,
                                               @RequestParam(value = "openid", required = false) String openid,
                                               @RequestParam(value = "wxName", required = false) String wxName,
                                               @RequestParam(value = "mobile", required = false) String mobile,
                                               @RequestParam(value = "status", required = false) String status,
                                               @RequestParam(value = "channel", required = false) String channel,
                                               @RequestParam(value = "startTime", required = false) String startTime,
                                               @RequestParam(value = "endTime", required = false) String endTime,
                                               @RequestParam(value = "page", defaultValue = "1") Integer page,
                                               @RequestParam(value = "limit", defaultValue = "20") Integer limit);

    /**
     * Save or update user basic info response.
     *
     * @param userBasicInfo the user basic info
     * @return the response
     */
    @PostMapping("/user/user-basic-info")
    Response<UserBasicInfo> saveOrUpdateUserBasicInfo(@RequestBody UserBasicInfo userBasicInfo);

    /**
     * Update user basic info by user code response.
     *
     * @param userBasicInfo the user basic info
     * @return the response
     */
    @PostMapping("/user/user-basic-info/updateByUserCode")
    Response updateUserBasicInfoByUserCode(@RequestBody UserBasicInfo userBasicInfo);

    /**
     * Gets user basic info by openid.
     *
     * @param openid the openid
     * @return the user basic info by openid
     */
    @GetMapping("/user/user-basic-info/getByOpenid")
    Response<UserBasicInfo> getUserBasicInfoByOpenid(@RequestParam("openid") String openid);

    /**
     * Gets user basic info by wx unionid.
     *
     * @param unionid the unionid
     * @return the user basic info by wx unionid
     */
    @GetMapping("/user/user-basic-info/getByWxUnionid")
    Response<UserBasicInfo> getUserBasicInfoByWxUnionid(@RequestParam("unionid") String unionid);

    /**
     * Gets user basic info by user code.
     *
     * @param userCode the user code
     * @return the user basic info by user code
     */
    @GetMapping("/user/user-basic-info/getByUserCode")
    Response<UserBasicInfo> getUserBasicInfoByUserCode(@RequestParam("userCode") String userCode);

    /**
     * Gets user basic info by samsung mem id.
     *
     * @param samsungMemId the samsung mem id
     * @return the user basic info by samsung mem id
     */
    @GetMapping("/user/user-basic-info/getBySamsungMemId")
    Response<UserBasicInfo> getUserBasicInfoBySamsungMemId(@RequestParam("samsungMemId") String samsungMemId);

    /**
     * Gets user basic info by samsung mem id or unionid.
     *
     * @param samsungMemId the samsung mem id
     * @param unionid      the unionid
     * @return the user basic info by samsung mem id or unionid
     */
    @GetMapping("/user/user-basic-info/getBySamsungMemIdOrUnionid")
    Response<UserBasicInfo> getUserBasicInfoBySamsungMemIdOrUnionid(@RequestParam(value = "samsungMemId", required = false) String samsungMemId,
                                                                    @RequestParam(value = "unionid", required = false) String unionid);
    // =========  end of  用户信息 by ranxiangfu   ===========

    // =========  start of  三星账户信息 by ranxiangfu   ===========
    /**
     * Save or update samsung mem info response.
     *
     * @param samsungMemInfo the samsung mem info
     * @return the response
     */
    @PostMapping("/user/samsung-mem-info")
    Response<SamsungMemInfo> saveOrUpdateSamsungMemInfo(@RequestBody SamsungMemInfo samsungMemInfo);

    /**
     * Update samsung mem info by user code response.
     *
     * @param samsungMemInfo the samsung mem info
     * @return the response
     */
    @PostMapping("/user/samsung-mem-info/updateByMemId")
    Response updateSamsungMemInfoByMemId(@RequestBody SamsungMemInfo samsungMemInfo);

    /**
     * Gets samsung mem info by wx unionid.
     *
     * @param unionid the unionid
     * @return the samsung mem info by wx unionid
     */
    @GetMapping("/user/samsung-mem-info/getByWxUnionid")
    Response<SamsungMemInfo> getSamsungMemInfoByWxUnionid(@RequestParam("unionid") String unionid);

    /**
     * Gets samsung mem info by mem id.
     *
     * @param memId the mem id
     * @return the samsung mem info by mem id
     */
    @GetMapping("/user/samsung-mem-info/getByMemId")
    Response<SamsungMemInfo> getSamsungMemInfoByMemId(@RequestParam("memId") String memId);
    // =========  end of  三星账户信息 by ranxiangfu   ===========

    // =========  start of  用户登录日志 by ranxiangfu   ===========

    /**
     * 保存用户登录日志.
     *
     * @param userLoginLog the user login log
     * @return the response
     */
    @PostMapping("/api/user/user-login-log")
    Response saveUserLoginLog(@RequestBody UserLoginLog userLoginLog);

    /**
     * 分页查询用户登录日志.
     *
     * @param userCode  the user code
     * @param openid    the openid
     * @param startTime the start time
     * @param endTime   the end time
     * @param page      the page
     * @param limit     the limit
     * @return the page table
     */
    @GetMapping("/api/user/user-login-log/pageQuery")
    PageTable<UserLoginLog> pageQueryUserLoginLog(@RequestParam(value = "userCode", required = false) String userCode,
                                                  @RequestParam(value = "openid", required = false) String openid,
                                                  @RequestParam(value = "startTime", required = false) String startTime,
                                                  @RequestParam(value = "endTime", required = false) String endTime,
                                                  @RequestParam(value = "page", defaultValue = "1") Integer page,
                                                  @RequestParam(value = "limit", defaultValue = "20") Integer limit);

    // =========  end of  用户登录日志 by ranxiangfu   ===========

    // =========  start of 用户模板消息   ===========

    /**
     * 保存模板消息
     *
     * @param userMsg the user msg
     * @return Response response
     */
    @PostMapping("/api/user/msg/saveUserMsg")
    Response saveUserMsg(@RequestBody UserMsg userMsg);

    /**
     * 获取用户消息列表，不分页
     *
     * @param userCode the user code
     * @return response
     */
    @GetMapping("/api/user/msg/listUserMsg/{userCode}")
    Response<List<UserMsg>> findUserMsgList(@PathVariable(value = "userCode") String userCode);

    /**
     * 获取用户消息列表，分页
     *
     * @param userCode the user code
     * @param limit    the limit
     * @param page     the page
     * @return page table
     */
    @GetMapping("/api/user/msg/pageUserMsgList/{userCode}")
    PageTable<UserMsg> pageUserMsgList(@PathVariable(value = "userCode") String userCode,
                                       @RequestParam(value = "limit", defaultValue = "20") int limit,
                                       @RequestParam(value = "page", defaultValue = "1") int page);

    /**
     * 阅读消息 通过code
     *
     * @param code the code
     * @return response
     */
    @GetMapping("/api/user/msg/readUserMsgByCode/{code}")
    Response<UserMsg> readUserMsgByCode(@PathVariable(value = "code") String code);

    /**
     * 清空消息（不传消息类型，消息全部清空；传消息类型，清空所有的消息类型）
     * 清空消息的同时，需要更新，相关的已读未读清空
     *
     * @param userCode 用户Code
     * @param msgType  消息类型（非必填）
     * @return response
     */
    @GetMapping("/api/user/msg/emptyUserMsgByUserCode/{userCode}")
    Response emptyUserMsgByUserCode(@PathVariable("userCode") String userCode,
                                    @RequestParam(value = "msgType", required = false) String msgType);

    /**
     * web端, 用户消息列表
     *
     * @param userCode   the user code
     * @param code       the code
     * @param senderCode the sender code
     * @param bindMobile the bind mobile
     * @param msgType    the msg type
     * @param hasRead    the has read
     * @param delStatus  the del status
     * @param startTime  the start time
     * @param endTime    the end time
     * @param page       the page
     * @param limit      the limit
     * @return page table
     */
    @GetMapping("/api/user/msg/pageWebQueryPageUserMsg")
    PageTable<UserMsgWebDto> pageWebQueryPageUserMsg(@RequestParam(value = "userCode", required = false) String userCode,
                                                     @RequestParam(value = "code", required = false) String code,
                                                     @RequestParam(value = "senderCode", required = false) String senderCode,
                                                     @RequestParam(value = "bindMobile", required = false) String bindMobile,
                                                     @RequestParam(value = "msgType", required = false) String msgType,
                                                     @RequestParam(value = "hasRead", required = false) Integer hasRead,
                                                     @RequestParam(value = "delStatus", required = false) Integer delStatus,
                                                     @RequestParam(value = "startTime", required = false) String startTime,
                                                     @RequestParam(value = "endTime", required = false) String endTime,
                                                     @RequestParam(value = "page", defaultValue = "1") Integer page,
                                                     @RequestParam(value = "limit", defaultValue = "20") Integer limit);
    // =========  end of  用户模板消息   ===========

    // =========  start of  用户收货地址   ===========

    /**
     * List user delivery address response.
     *
     * @param userCode the user code
     * @param deleted  the deleted
     * @return the response
     */
    @GetMapping("/api/user/user-delivery-address/list")
    Response<List<UserDeliveryAddress>> listUserDeliveryAddress(@RequestParam("userCode") String userCode,
                                                                @RequestParam(value = "deleted", required = false) Boolean deleted);

    /**
     * Save or update user delivery address response.
     *
     * @param userDeliveryAddress the user delivery address
     * @return the response
     */
    @PostMapping("/api/user/user-delivery-address")
    Response<Integer> saveOrUpdateUserDeliveryAddress(@RequestBody UserDeliveryAddress userDeliveryAddress);

    /**
     * Gets user delivery address by id.
     *
     * @param id the id
     * @return the user delivery address by id
     */
    @GetMapping("/api/user/user-delivery-address")
    Response<UserDeliveryAddress> getUserDeliveryAddressById(@RequestParam("id") Integer id);

    /**
     * Gets one user delivery address.
     *
     * @param userCode the user code
     * @return the one user delivery address
     */
    @GetMapping("/api/user/user-delivery-address/getOne")
    Response<UserDeliveryAddress> getOneUserDeliveryAddress(@RequestParam("userCode") String userCode);

    // =========  end of  用户收货地址   ===========
}
