package com.rocogz.syy.common.response;

import com.fasterxml.jackson.annotation.JsonIgnoreProperties;

import java.io.Serializable;

/**
 * 统一响应处理
 * @author：liangyongtong
 * @date：2020/1/7
 */
@JsonIgnoreProperties(ignoreUnknown = true)
public class Response<T> implements Serializable {
    /**
     * 状态码
     */
    protected String code;

    /**
     * 信息
     */
    protected String message;

    /**
     * 携带数据
     */
    protected T data;

    public Response() {
    }

    /**
     * 响应对象
     * @param code 业务编码
     */
    public Response(String code) {
        this(code, "", null);
    }

    /**
     * 响应对象
     * @param code 业务编码
     * @param message 响应消息内容
     */
    public Response(String code, String message) {
        this(code, message, null);
    }

    /**
     * 响应对象
     * @param code 业务编码
     * @param message 响应消息内容
     * @param data 返回对象
     */
    public Response(String code, String message, T data) {
        this.code = code;
        this.message = message;
        this.data = data;
    }

    /**
     * 成功响应 无消息 无数据
     * @return
     */
    public static Response succeed() {
        return new Response(ResponseCode.SUCCEED);
    }

    /***
     * 成功响应 自定义消息内容
     * @param message 消息内容
     * @return
     */
    public static Response succeed(String message) {
        return new Response(ResponseCode.SUCCEED, message);
    }

    /**
     * 成功响应 带数据
     * @param data 数据
     * @return
     */
    public static <E> Response<E> succeed(E data) {
        return new Response(ResponseCode.SUCCEED, "SUCCEED", data);
    }

    /**
     * 成功响应 自定义响应码 带数据
     * @param code 响应码
     * @param data 数据
     * @return
     */
    public static <E> Response<E> succeed(String code, E data) {
        return new Response(code, "SUCCEED", data);
    }

    /**
     * 失败响应
     * @return
     */
    public static Response failure() {
        return new Response(ResponseCode.FAILURE);
    }

    /**
     * 失败响应
     * @param message 消息
     * @return
     */
    public static Response failure(String message) {
        return new Response(ResponseCode.FAILURE, message);
    }


    /**
     * 失败响应
     * @param code 失败响应码
     * @param message 消息
     * @return
     */
    public static Response failure(String code, String message) {
        return new Response(code, message);
    }

    /**
     * 失败响应
     * @param message 错误消息
     * @param data 失败携带的数据
     * @return
     */
    public static <E> Response failureWithData(String message,E data) {
        return failure(message).setData(data);
    }

    /**
     * 判断编码是否一致
     * @param targetCode 目标编码
     * @return
     */
    public boolean codeEqual(String targetCode) {
        return targetCode.equals(this.code);
    }

    /**
     * 判断结果是否成功
     * @return true 成功 false 失败
     */
    public boolean isSucceed() {
        return ResponseCode.SUCCEED.equals(this.code);
    }

    public String getCode() {
        return code;
    }

    public Response setCode(String code) {
        this.code = code;
        return this;
    }

    public String getMessage() {
        return message;
    }

    public Response setMessage(String message) {
        this.message = message;
        return this;
    }

    public T getData() {
        return data;
    }

    public Response setData(T data) {
        this.data = data;
        return this;
    }
}
