package com.rocogz.syy.common.aliyunoss.service;

import com.aliyun.oss.OSSClient;
import com.aliyun.oss.model.CannedAccessControlList;
import com.aliyun.oss.model.CreateBucketRequest;
import com.aliyun.oss.model.OSSObject;
import com.aliyun.oss.model.StorageClass;
import com.rocogz.syy.common.aliyunoss.config.AliyunOssProperties;
import com.rocogz.syy.common.response.Response;
import net.coobird.thumbnailator.Thumbnails;
import net.coobird.thumbnailator.geometry.Positions;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.http.util.Asserts;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.scheduling.concurrent.ThreadPoolTaskExecutor;

import javax.imageio.ImageIO;
import java.awt.*;
import java.awt.image.BufferedImage;
import java.io.*;
import java.math.BigDecimal;
import java.nio.file.Files;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.*;
import java.util.List;
import java.util.concurrent.Callable;
import java.util.concurrent.Future;

/**
 * 上传
 * @author liangyongtong
 */
public class UploadService {

    private Logger logger = LoggerFactory.getLogger(UploadService.class);

    private DateTimeFormatter formatter = DateTimeFormatter.ofPattern("yyyyMMdd");

    @Autowired
    private ThreadPoolTaskExecutor threadPoolTaskExecutor;
    
    private AliyunOssProperties ssoProperties;

    public UploadService() {}

    /**
     * 初始化对象需要构造函数
     * @param ssoProperties
     */
    public UploadService(AliyunOssProperties ssoProperties) {
        this.ssoProperties = ssoProperties;
        init();
    }

    /**
     * 初始化OSSClient和bucket
     */
    public void init() {
        // 检查阿里云上传配置是否填写
        checkAliyunProperties();

        // 配置使用aliyun的默认配置，参考 https://help.aliyun.com/document_detail/32010.html
        initBucket();
    }

    /**
     * 检查配置
     */
    private void checkAliyunProperties() {
        Asserts.check(Objects.nonNull(this.ssoProperties), "阿里云所需要配置缺失");
        Asserts.check(StringUtils.isNotEmpty(this.ssoProperties.getAccessKeyId()), "阿里云权限认证：AccessKeyId 缺失");
        Asserts.check(StringUtils.isNotEmpty(this.ssoProperties.getAccessKeySecret()), "阿里云权限认证：accessKeySecret 缺失");
        Asserts.check(StringUtils.isNotEmpty(this.ssoProperties.getBucket()), "阿里云权限认证：bucket 缺失");
        Asserts.check(StringUtils.isNotEmpty(this.ssoProperties.getEndpoint()), "阿里云权限认证：endpoint 缺失");
    }

    public OSSClient getOSSClient() {
        return new OSSClient(ssoProperties.getEndpoint(), ssoProperties.getAccessKeyId(), ssoProperties.getAccessKeySecret());
    }

    /**
     * 初始化bucket，如果buket存在，则不创建。
     */
    private void initBucket() {
        OSSClient ossClient = getOSSClient();
        boolean exists = ossClient.doesBucketExist(ssoProperties.getBucket());
        if (!exists) {
            // 创建bucket
            CreateBucketRequest cbr = new CreateBucketRequest(ssoProperties.getBucket());
            // 设置bucket权限为公共读，默认是私有读写
            cbr.setCannedACL(CannedAccessControlList.PublicRead);
            // 设置bucket存储类型为标准类型
            cbr.setStorageClass(StorageClass.Standard);
            ossClient.createBucket(cbr);
        }
    }

    public String fileUpload(InputStream inputStream, String originName, String type, String[] thumbImgSizeList) {
        //获取上传至oss文件的key
        String uniqueId = getUniqueId(type);
        StringBuilder mainFileName = new StringBuilder();
        String fileSubfix = getFileSubfix(originName);
        mainFileName.append(uniqueId).append(fileSubfix);
        byte[] data = streamToByteArray(inputStream);
        String mainUrl = fileUploadByString(new ByteArrayInputStream(data), originName, type, mainFileName.toString());
        // 生成缩略图
        if (ArrayUtils.isNotEmpty(thumbImgSizeList)) {
            for (String imageStr : thumbImgSizeList) {
                String[] img = imageStr.split("_");
                StringBuilder thumb = new StringBuilder();
                thumb.append(uniqueId).append(imageStr).append(fileSubfix);
                threadPoolTaskExecutor.submit(new ZipImage(data, Integer.valueOf(img[0]), Integer.valueOf(img[1]), originName, type, thumb.toString()));
            }
        }
        return mainUrl;
    }

    /**
     * 上传文件到阿里云
     * @param inputStream 文件流
     * @param fileName 文件名称，带扩展名；如：abc.jpg
     * @param type 上传类型
     * @return 返回文件路径 如： http://picc-car-dev.oss-cn-beijing.aliyuncs.com/{type}/{yyyyMMdd}/{fileName}
     */
    public String fileUpload(InputStream inputStream, String fileName, String type) {
        StringBuilder key = new StringBuilder(type);
        key.append("/");
        key.append(getDateString());
        key.append("/");
        key.append(fileName);
        byte[] data = streamToByteArray(inputStream);
        String mainUrl = fileUploadByString(new ByteArrayInputStream(data), fileName, type, key.toString());
        return mainUrl;
    }

    /**
     * 文件保存到永久目录
     * @param inputStream 来源数据流
     * @param originName 文件名 （最终文件名 {HHmmss}-originName)
     * @return 绝对文件路径
     */
    public String fileSaveToLocal(InputStream inputStream, String originName) {
        return fileSave(inputStream, this.ssoProperties.getLocalDirectory(), originName);
    }

    /**
     * 文件保存到临时目录
     * @param inputStream 来源数据流
     * @param originName 文件名 （最终文件名 {HHmmss}-originName)
     * @return 绝对文件路径
     */
    public String fileSaveToTmpLocal(InputStream inputStream, String originName) {
        return fileSave(inputStream, this.ssoProperties.getLocalTmpDirectory(), originName);
    }

    /**
     * 保存文件
     * @param inputStream 来源数据流
     * @param directory 配置的目录
     * @param originName 目标文件名
     * @return
     */
    private String fileSave(InputStream inputStream, String directory, String originName) {
        // 目录文件生成
        String path = directory
                + File.separator
                + LocalDate.now().format(DateTimeFormatter.ofPattern("yyyyMMdd"))
                + File.separator
                + LocalDateTime.now().format(DateTimeFormatter.ofPattern("HHmmss"))
                + "-"
                + originName;

        try {
            File target = new File(path);

            // 目录不存在，需要生成
            if (!target.getParentFile().exists()) {
                target.getParentFile().mkdirs();
            }
            logger.info("开始保存文件[{}]", originName);
            Files.copy(inputStream, target.toPath());
            logger.info("文件[{}]已保存到[{}]", originName, directory);
        } catch (IOException e) {
            logger.error("保存[{}]文件失败 -> ", originName, e);
        } finally {
            try {
                inputStream.close();
            } catch (IOException e) {
                e.printStackTrace();
            }
        }

        return path;
    }

    public Response generateThumb(InputStream inputStream, String type, String originName, String[] thumbImgSizeList, long fileSize){
        String uniqueId = getUniqueId(type);
        StringBuilder mainFileName = new StringBuilder();
        String fileSubfix = getFileSubfix(originName);
        mainFileName.append(uniqueId).append(fileSubfix);
        byte[] data = streamToByteArray(inputStream);
        try {
            if (ArrayUtils.isNotEmpty(thumbImgSizeList)) {
                ByteArrayInputStream originInputStream = new ByteArrayInputStream(data);    //将b作为输入流；
                BufferedImage originImage = ImageIO.read(originInputStream);     //将in作为输入流，读取图片存入image中，而
                for (String imageStr : thumbImgSizeList) {
                    String[] img = imageStr.split("_");
                    //原图尺寸不能小于压缩后的尺寸
                    if(originImage.getWidth()<Integer.valueOf(img[0])||originImage.getHeight()<Integer.valueOf(img[1])){
                        return Response.failure("原图尺寸小于压缩后的尺寸");
                    }
                }
            }
        } catch (IOException e) {
            logger.error("图片尺寸检查失败{}",e);
            return Response.failure("图片尺寸检查失败");
        }

        String mainUrl = fileUploadByString(new ByteArrayInputStream(data), originName, type, mainFileName.toString());//上传
        // 生成缩略图
        Map<String,Object> thumbPathsMap = new HashMap<>();
        if (ArrayUtils.isNotEmpty(thumbImgSizeList)) {
            for (String imageStr : thumbImgSizeList) {
                String[] img = imageStr.split("_");
                StringBuilder thumb = new StringBuilder();
                thumb.append(uniqueId).append(imageStr);
                Future<List<String>> mapFuture = threadPoolTaskExecutor.submit(new ThumbImage(data, Integer.valueOf(img[0]), Integer.valueOf(img[1]), originName, type, thumb.toString()));
                try {
                    List<String> paths = mapFuture.get();
                    thumbPathsMap.put(imageStr,paths);
                } catch (Exception e) {
                    logger.error("创建缩略图失败{}",e);
                    return Response.failure("创建缩略图失败");
                }
            }
        }

        Map<String, Object> result = new HashMap<>();
        result.put("fileSize", fileSize);
        result.put("path", mainUrl);
        result.put("fileName", originName);
        result.put("type", type);
        thumbPathsMap.put("mainImage", result);
        return Response.succeed(thumbPathsMap);
    }

    public String fileUploadByString(InputStream inputStream, String originName, String type) {
        return fileUploadByString(inputStream, originName, type, getUniqueKey(originName, type));
    }
    public void selectTempThumb(String [] fileUrls){
        for (String fileUrl : fileUrls) {
            String tempThumbKey = fileUrl.substring(fileUrl.lastIndexOf("/")+1);
            String formatName = tempThumbKey.substring(tempThumbKey.lastIndexOf(".")+1);
            String thumbKey = tempThumbKey.substring(0,tempThumbKey.lastIndexOf("_"))+"."+formatName;
            OSSClient ossClient = getOSSClient();
            OSSObject ossObject = ossClient.getObject(ssoProperties.getBucket(), tempThumbKey);
            ossClient.putObject(ssoProperties.getBucket(), thumbKey, ossObject.getObjectContent());
            ossClient.deleteObject(ssoProperties.getBucket(), tempThumbKey);
        }

    }

    /**
     * 将文件上传到阿里云，并返回下载地址
     *
     * @param inputStream
     * @param type
     * @return
     */
    public String fileUploadByString(InputStream inputStream, String originName, String type, String key) {
        Long startTime = System.currentTimeMillis();
        logger.info("文件上传到oss开始");
        OSSClient ossClient = getOSSClient();
        try {
            ossClient.putObject(ssoProperties.getBucket(), key, inputStream);
        } finally {
            ossClient.shutdown();
        }

        // 构造返回路径
        StringBuilder buffer = constructionPath(key);

        logger.info("文件上传到oss完成,耗时{}ms", System.currentTimeMillis() - startTime);
        return buffer.toString();
    }

    /**
     * 构造返回文件路径
     * @param key
     * @return
     */
    private StringBuilder constructionPath(String key) {

        // 如果配置了公网访问路径，则使用公网的访问路径构造

        String point = StringUtils.isNoneBlank(ssoProperties.getVisitEndPoint()) ? ssoProperties.getVisitEndPoint() : ssoProperties.getEndpoint();

        String[] paths = point.split("//");
        StringBuilder buffer = new StringBuilder();
        buffer.append(paths[0])
                .append("//")
                .append(ssoProperties.getBucket())
                .append(".")
                .append(paths[1])
                .append("/")
                .append(key);
        return buffer;
    }

    public byte[] streamToByteArray(InputStream inputStream) {
        byte[] tmp = new byte[1024];
        ByteArrayOutputStream byteArrayOutputStream = new ByteArrayOutputStream();
        int i = 0;
        try {
            while ((i = inputStream.read(tmp)) != -1) {
                byteArrayOutputStream.write(tmp, 0, i);
            }
        } catch (IOException e) {
            e.printStackTrace();
        } finally {
            try {
                byteArrayOutputStream.flush();
                byteArrayOutputStream.close();
            } catch (IOException e) {
                e.printStackTrace();
            }
        }
        return byteArrayOutputStream.toByteArray();
    }

    /**
     * 删除附件
     *
     * @param key
     */
    public void deleteObject(String key) {
        OSSClient ossClient = getOSSClient();
        ossClient.deleteObject(ssoProperties.getBucket(), key);
    }

    /**
     * 获取唯一文件名 contract/20161230/9b9b8ceb2d614554a429e240fd09f53d.docx
     *
     * @return
     */
    public String getUniqueKey(String originName, String type) {
        StringBuilder key = new StringBuilder();
        key.append(getUniqueId(type));
        key.append(getFileSubfix(originName));
        return key.toString();
    }

    public String getFileSubfix(String originName) {
        if (originName.indexOf(".") != -1) {
            return originName.substring(originName.lastIndexOf("."));
        }
        return "";
    }

    public String getUniqueId(String type) {
        StringBuilder key = new StringBuilder(type);
        key.append("/")
        .append(getDateString())
        .append("_")
        .append(getUUID());
        return key.toString();
    }

    /**
     * 获取当前时间的 年月日 的字符串（如： 20161230）
     *
     * @return
     */
    private String getDateString() {
        return LocalDate.now().format(formatter);
    }

    /**
     * 获取32位uuid
     *
     * @return
     */
    private String getUUID() {
        String uuid = UUID.randomUUID().toString();
        //去掉“-”符号
        return uuid.replace("-","");
    }


    class ZipImage implements Callable<String> {
        private int width;
        private int height;
        private String fileName;
        private String type;
        private String key;
        private byte[] data;

        public ZipImage(byte[] data, int width, int height, String fileName, String type, String key) {
            this.width = width;
            this.height = height;
            this.fileName = fileName;
            this.type = type;
            this.key = key;
            this.data = data;
        }

        public int getWidth() {
            return width;
        }

        public void setWidth(int width) {
            this.width = width;
        }

        public int getHeight() {
            return height;
        }

        public void setHeight(int height) {
            this.height = height;
        }

        public String getFileName() {
            return fileName;
        }

        public void setFileName(String fileName) {
            this.fileName = fileName;
        }

        public String getType() {
            return type;
        }

        public void setType(String type) {
            this.type = type;
        }

        public String getKey() {
            return key;
        }

        public void setKey(String key) {
            this.key = key;
        }

        public byte[] getData() {
            return data;
        }

        public void setData(byte[] data) {
            this.data = data;
        }

        @Override
        public String call() throws Exception {
            String formatName = fileName.substring(fileName.lastIndexOf(".") + 1);
            byte[] data = resize(this.getData(), formatName, this.getWidth(), this.getHeight());
            return fileUploadByString(new ByteArrayInputStream(data), this.getFileName(), this.getType(), this.getKey());
        }

    }

    class ThumbImage implements Callable<List<String>> {
        private int width;
        private int height;
        private String fileName;
        private String type;
        private String key;
        private byte[] data;

        public ThumbImage(byte[] data, int width, int height, String fileName, String type, String key) {
            this.width = width;
            this.height = height;
            this.fileName = fileName;
            this.type = type;
            this.key = key;
            this.data = data;
        }

        public int getWidth() {
            return width;
        }

        public void setWidth(int width) {
            this.width = width;
        }

        public int getHeight() {
            return height;
        }

        public void setHeight(int height) {
            this.height = height;
        }

        public String getFileName() {
            return fileName;
        }

        public void setFileName(String fileName) {
            this.fileName = fileName;
        }

        public String getType() {
            return type;
        }

        public void setType(String type) {
            this.type = type;
        }

        public String getKey() {
            return key;
        }

        public void setKey(String key) {
            this.key = key;
        }

        public byte[] getData() {
            return data;
        }

        public void setData(byte[] data) {
            this.data = data;
        }

        @Override
        public List<String> call() throws Exception {
            String formatName = fileName.substring(fileName.lastIndexOf(".") + 1);
            ByteArrayInputStream originInputStream = new ByteArrayInputStream(this.getData());    //将b作为输入流；
            BufferedImage originImage = ImageIO.read(originInputStream);     //将in作为输入流，读取图片存入image中，而
            BigDecimal bigOriginWidth = new BigDecimal(originImage.getWidth());
            BigDecimal bigOriginHeight = new BigDecimal(originImage.getHeight());
            BigDecimal bigTargetWidth = new BigDecimal(this.getWidth());
            BigDecimal bigTargetHeight = new BigDecimal(this.getHeight());
            List<String> paths = new ArrayList<>();
            //目标长宽小于等于原图长宽
            if(this.getWidth()<=originImage.getWidth()&&this.getHeight()<=originImage.getHeight()){
                //判断是否等比缩放
                if(this.getWidth()/originImage.getWidth()==this.getHeight()/originImage.getHeight()&&this.getWidth()%originImage.getWidth()==this.getHeight()%originImage.getHeight()){
                    byte[] thumbData = resize(this.getData(), formatName, this.getWidth(), this.getHeight());
                    String imagePath =  fileUploadByString(new ByteArrayInputStream(thumbData), this.getFileName(), this.getType(), this.getKey()+"."+formatName);
                    paths.add(imagePath);
                }else{
                    double widthRatio = bigTargetWidth.divide(bigOriginWidth,10,BigDecimal.ROUND_HALF_UP).doubleValue();
                    double heightRatio = bigTargetHeight.divide(bigOriginHeight,10,BigDecimal.ROUND_HALF_UP).doubleValue();
                    //原图按长比计算压缩后宽大于目标宽按上中下裁切
                    if(bigTargetWidth.divide(bigOriginWidth,10,BigDecimal.ROUND_HALF_UP).multiply(bigOriginHeight).doubleValue()>bigTargetHeight.doubleValue()){
                        //上中下
                        byte[] thumbData = resize(this.getData(), formatName, widthRatio);
                        ByteArrayOutputStream byteArrayOutputStreamTopCenter = resize(thumbData, Positions.TOP_CENTER, this.getWidth(), this.getHeight(), formatName);
                        String topCenterImagePath = fileUploadByString(new ByteArrayInputStream(byteArrayOutputStreamTopCenter.toByteArray()), this.getFileName(), this.getType(), this.getKey()+"."+formatName);
                        paths.add(topCenterImagePath);
                    }else if (bigTargetHeight.divide(bigOriginHeight,10,BigDecimal.ROUND_HALF_UP).multiply(bigOriginWidth).doubleValue()>bigTargetWidth.doubleValue()){
                        //原图按宽比计算压缩后长大于目标长按左中右裁切
                        byte[] thumbData = resize(this.getData(), formatName, heightRatio);
                        ByteArrayOutputStream byteArrayOutputStreamCenter = resize(thumbData, Positions.CENTER, this.getWidth(), this.getHeight(), formatName);
                        String centerImagePath = fileUploadByString(new ByteArrayInputStream(byteArrayOutputStreamCenter.toByteArray()), this.getFileName(), this.getType(), this.getKey()+"."+formatName);
                        paths.add(centerImagePath);
                    }
                }
            }
            return paths;
        }
    }

    /**
     * 根据长款对图片进行剪裁，保持图片比例不变，如果达不到指定大小，则进行补白
     *
     * @param data       源图片字节
     * @param formatName 文件格式
     * @param width      宽
     * @param height     高
     * @throws IOException size(width,height) 若图片横比200小，高比300小，不变
     */
    private byte[] resize(byte[] data, String formatName, int width, int height) throws IOException {
        BufferedImage src = Thumbnails.of(new ByteArrayInputStream(data)).outputQuality(0.8f).size(width, height).asBufferedImage();
        return paddingWhite(src, formatName, width, height).toByteArray();
    }

    private byte[] resize(byte[] data, String formatName, double scale) throws IOException {
        BufferedImage src = Thumbnails.of(new ByteArrayInputStream(data)).outputQuality(0.8f).scale(scale).asBufferedImage();
        ByteArrayOutputStream out = new ByteArrayOutputStream();
        try {
            ImageIO.write(src, formatName, out);
        } catch (IOException e) {
            e.printStackTrace();
        }
        return out.toByteArray();
    }

    private byte[] resize(InputStream is, String formatName, int width, int height) throws IOException {
        BufferedImage src = Thumbnails.of(is).outputQuality(0.8f).size(width, height).asBufferedImage();
        return paddingWhite(src, formatName, width, height).toByteArray();
    }

    private ByteArrayOutputStream resize(byte[] data,Positions positions,int width,int height,String formatName){
        ByteArrayOutputStream byteArrayOutputStream  = new ByteArrayOutputStream();
        try {
            BufferedImage bufferedImage = Thumbnails.of(new ByteArrayInputStream(data)).sourceRegion(positions,width,height).outputQuality(0.8f).size(width, height).asBufferedImage();
            ImageIO.write(bufferedImage,formatName,byteArrayOutputStream);
        } catch (IOException e) {
            e.printStackTrace();
        }
        return byteArrayOutputStream;
    }

    private ByteArrayOutputStream paddingWhite(BufferedImage src, String formatName, int targetWidth, int targetHeight) {
        ByteArrayOutputStream out = new ByteArrayOutputStream();
        // 获取当前图片的宽、高
        int width = src.getWidth();
        int height = src.getHeight();

        if (width == targetWidth && height == targetHeight) {
            try {
                ImageIO.write(src, formatName, out);
            } catch (IOException e) {
                e.printStackTrace();
            }
            return out;
        }

        BufferedImage target = new BufferedImage(targetWidth, targetHeight, src.getType());
        // 画一张指定大小的白色图
        Graphics2D graphics = target.createGraphics();
        graphics.setColor(Color.WHITE);
        graphics.fillRect(0, 0, targetWidth, targetHeight);
        // 计算图片的插入坐标
        int x = 0;
        int y = 0;
        if (targetWidth > width) {
            x = (targetWidth - width) / 2;
        }
        if (targetHeight > height) {
            y = (targetHeight - height) / 2;
        }
        graphics.drawImage(src, null, x, y);
        graphics.dispose();

        try {
            ImageIO.write(target, formatName, out);
        } catch (IOException e) {
            e.printStackTrace();
        }
        return out;
    }
}
