package com.rocogz.syy.common.web;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.PropertyNamingStrategy;
import com.fasterxml.jackson.databind.deser.std.DateDeserializers;
import com.fasterxml.jackson.datatype.jdk8.Jdk8Module;
import com.fasterxml.jackson.datatype.jsr310.JavaTimeModule;
import com.fasterxml.jackson.datatype.jsr310.deser.LocalDateTimeDeserializer;
import com.fasterxml.jackson.datatype.jsr310.ser.LocalDateSerializer;
import com.fasterxml.jackson.datatype.jsr310.ser.LocalDateTimeSerializer;
import com.fasterxml.jackson.datatype.jsr310.ser.LocalTimeSerializer;
import com.fasterxml.jackson.module.paramnames.ParameterNamesModule;
import com.rocogz.syy.common.web.converter.*;
import org.springframework.boot.autoconfigure.AutoConfigureAfter;
import org.springframework.boot.autoconfigure.condition.ConditionalOnWebApplication;
import org.springframework.boot.autoconfigure.jackson.Jackson2ObjectMapperBuilderCustomizer;
import org.springframework.boot.autoconfigure.jackson.JacksonAutoConfiguration;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.format.FormatterRegistry;
import org.springframework.http.converter.json.Jackson2ObjectMapperBuilder;
import org.springframework.http.converter.json.MappingJackson2HttpMessageConverter;
import org.springframework.web.servlet.config.annotation.WebMvcConfigurer;

import java.text.SimpleDateFormat;
import java.time.format.DateTimeFormatter;
import java.util.TimeZone;

/**
  @author zhangmin
  @date 2020/4/12
 */
@Configuration
@ConditionalOnWebApplication
public class JacksonCustomizerConverterAutoConfiguration {

    private static final String dateFormat = "yyyy-MM-dd";
    private static final String dateTimeFormat = "yyyy-MM-dd HH:mm:ss";
    private static final String TIME_FORMAT = "HH:mm:ss";

    //默认的objectMapper
    public static ObjectMapper OBJECT_MAPPER;

    /**
     * json属性名：是下划线分隔形式
     */
    public static ObjectMapper SNAKE_CASE_OBJECT_MAPPER;

     /**
     * 定制化json输出格式:
     * 解决如下问题：
     * （1）java8中日期与时间 json序列号与反序列化问题
     * （2）实体类中：日期输出 不用再写 @JsonFormat(pattern = YYYY_MM_DD_HH_MM_SS, timezone = TIME_ZONE)
     *  (3)json反序列化是，全局处理 "忽略json字符串中不识别的属性"
     */
      @Bean
      public Jackson2ObjectMapperBuilderCustomizer jsonCustomizer() {
          return builder -> {
              builder.simpleDateFormat(dateTimeFormat);

              DateTimeFormatter dateTimeFormatter = DateTimeFormatter.ofPattern(dateTimeFormat);
              builder.modulesToInstall(new Jdk8Module(),new JavaTimeModule(),new ParameterNamesModule());
              builder.serializers(new LocalDateSerializer(DateTimeFormatter.ofPattern(dateFormat)));
              builder.serializers(new LocalDateTimeSerializer(dateTimeFormatter));

              builder.serializers(new LocalTimeSerializer(DateTimeFormatter.ofPattern(TIME_FORMAT)));

              //java.sql.Date的 反序列化
              DateDeserializers.SqlDateDeserializer sqlDateDeserializer =
                      new DateDeserializers.SqlDateDeserializer(new DateDeserializers.SqlDateDeserializer(),new SimpleDateFormat(dateFormat), dateFormat);
              builder.deserializers(sqlDateDeserializer);


              //java8的 LocalDateTime的 json反序列化的 默认时间格式是："2019-12-12T12:43:12" 即： LocalDateTimeDeserializer
              builder.deserializers(new LocalDateTimeDeserializer(dateTimeFormatter));

              // 忽略json字符串中不识别的属性
              builder.failOnUnknownProperties(false);

              // 忽略 空Bean转json报错， “No serializer found for class com.xxx.xxx”
              builder.failOnEmptyBeans(false);

              //指定时区
              builder.timeZone(TimeZone.getTimeZone("GMT+8:00"));

              // NULL不参与序列化
              // builder.serializationInclusion(JsonInclude.Include.NON_NULL);
          };
        }

    /**
     * （1）对于 @RequestParam 接收的日期数据，注册了 日期格式转换器
     *  (2)对于 x-www-form-urlencoded 表单数据,进行日期格式转换
     * @param registry
     */
      @Bean
      public WebMvcConfigurer registDateConverterConfigurer(MappingJackson2HttpMessageConverter messageConverter,
                                                            Jackson2ObjectMapperBuilder objectMapperBuilder) {
          OBJECT_MAPPER = messageConverter.getObjectMapper();

          SNAKE_CASE_OBJECT_MAPPER = objectMapperBuilder.build().setPropertyNamingStrategy(PropertyNamingStrategy.SNAKE_CASE);

          return new WebMvcConfigurer() {
              @Override
              public void addFormatters(FormatterRegistry registry) {
                  registry.addConverter(new LocalDateConverter());
                  registry.addConverter(new YearMonthConverter());
                  registry.addConverter(new LocalTimeConverter());
                  registry.addConverter(new LocalDateTimeConverter());
              }
          };

      }

}
