package com.rocogz.syy.common.web;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.JavaType;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.type.MapType;
import com.rocogz.syy.common.response.Response;
import org.apache.commons.lang3.StringUtils;

import java.io.IOException;
import java.util.Collections;
import java.util.List;
import java.util.Map;

/**
 * 解决了 java8中日期转换问题的 json服务
 *
 * @author zhangmin
 * @date 2020/5/28
 */
public final class JsonJava8Util {

    private JsonJava8Util() {
    }


    public static ObjectMapper getObjectMapper() {
        return JacksonCustomizerConverterAutoConfiguration.OBJECT_MAPPER;
    }


    /**
     * @param pojo 或者 pojo列表,或者 map
     */
    public static String toJson(Object pojo) {
        if (pojo == null) {
            return null;
        }
        String toJson = StringUtils.EMPTY;
        try {
            toJson = getObjectMapper().writeValueAsString(pojo);
        } catch (JsonProcessingException ex) {
            ex.printStackTrace();
        }
        return toJson;
    }

    /**
     * pojo 转成 格式美化后的json,注意：此方法使用于开发人员调试,不适合生产环境
     * @param pojo
     */
    public static <T> String toJsonPretty(T pojo) {
        if (pojo == null) {
            return null;
        }

        if(pojo instanceof String) {
            return (String) pojo;
        }

        String toJson = StringUtils.EMPTY;
        try {
            toJson = getObjectMapper().writerWithDefaultPrettyPrinter().writeValueAsString(pojo);
        } catch (JsonProcessingException ex) {
            ex.printStackTrace();
        }
        return toJson;

    }


    /**
     * 将json转换成 pojo或map
     *
     * @param json
     * @param pojoClass 或 Map.class
     */
    public static <T> T fromJson(String json, Class<T> pojoClass) {
        if (StringUtils.isBlank(json)) {
            return null;
        } else {
            try {
                return getObjectMapper().readValue(json, pojoClass);
            } catch (IOException ex) {
                return null;
            }
        }
    }


    /**
     * @param typeRef 范型T可以是对象类型，也可以是对象列表
     *                创建 TypeReference<T> typeRef = new TypeReference<>(){};
     */
    public static <T> T fromJson(String json, TypeReference<T> typeRef) {
        if (StringUtils.isBlank(json)) {
            return null;
        }

        try {
            return getObjectMapper().readValue(json, typeRef);
        } catch (IOException e) {
            return null;
        }
    }


    /**
     * 例如：TypeReference<Response<List<User>>> typeRef =  new TypeReference<Response<List<User>>>()
     */
    public static <T> Response<T> fromJsonToResp(String responseJson, TypeReference<Response<T>> typeRef) {
        if (StringUtils.isBlank(responseJson)) {
            return null;
        }

        try {
            return getObjectMapper().readValue(responseJson, typeRef);
        } catch (IOException e) {
            return null;
        }
    }


    /**
     * Json串  转成POJO对象 List,带泛型
     *
     * @param pojoListJson 对象列表json串
     * @param pojoClass    实体类class
     */
    public static <T> List<T> fromJsonAsList(String pojoListJson, Class<T> pojoClass) {
        if (StringUtils.isBlank(pojoListJson)) {
            return Collections.emptyList();
        }

        JavaType type = getObjectMapper().getTypeFactory().constructCollectionType(List.class, pojoClass);
        try {
            return getObjectMapper().readValue(pojoListJson, type);
        } catch (IOException e) {
            return Collections.emptyList();
        }
    }


    /**
     * map json串，转换成map对象
     *
     * @param mapJson
     * @param keyClass
     * @param valueClass 值类型可以是POJO
     */
    public static <K, V> Map<K, V> fromJsonAsMap(String mapJson, Class<K> keyClass, Class<V> valueClass) {
        if (StringUtils.isBlank(mapJson)) {
            return Collections.emptyMap();
        }

        MapType type = getObjectMapper().getTypeFactory().constructMapType(Map.class, keyClass, valueClass);
        try {
            return getObjectMapper().readValue(mapJson, type);
        } catch (IOException e) {
            return Collections.emptyMap();
        }
    }

}
