package com.rocogz.syy.equity.client;

import com.rocogz.syy.common.PageTable;
import com.rocogz.syy.common.response.Response;
import com.rocogz.syy.equity.dto.bestow.BestowRecordQueryDto;
import com.rocogz.syy.equity.dto.equity.OilCashAndOilCardCouponDto;
import com.rocogz.syy.equity.dto.equity.batchDistributionApply.*;
import com.rocogz.syy.equity.dto.equity.oilCardPointAccount.OilCardPointGrantParamDto;
import com.rocogz.syy.equity.dto.equity.oilCardUserCoupon.OilCardUserCouponBatchExchangeDto;
import com.rocogz.syy.equity.dto.equity.statistics.*;
import com.rocogz.syy.equity.dto.equity.userCoupon.*;
import com.rocogz.syy.equity.dto.issuingBody.EquityIssuingBodyAssociatedDto;
import com.rocogz.syy.equity.dto.issuingBody.IssuingBodyDto;
import com.rocogz.syy.equity.dto.tree.IssuingBodyTreeDto;
import com.rocogz.syy.equity.entity.batchDistributionCouponApply.*;
import com.rocogz.syy.equity.entity.bestow.EquityBestowRecord;
import com.rocogz.syy.equity.entity.bestow.EquityBestowRecordItem;
import com.rocogz.syy.equity.entity.coupon.EquityCashCouponAmount;
import com.rocogz.syy.equity.entity.coupon.EquityCouponGiftRel;
import com.rocogz.syy.equity.entity.coupon.EquityCouponInfo;
import com.rocogz.syy.equity.entity.coupon.EquityCouponLaunchInfo;
import com.rocogz.syy.equity.entity.couponType.EquityCouponType;
import com.rocogz.syy.equity.entity.driver.DesignatedDriverRecord;
import com.rocogz.syy.equity.entity.issuingBody.IssuingBody;
import com.rocogz.syy.equity.entity.oilCardPointAccount.EquityOilCardPointAccount;
import com.rocogz.syy.equity.entity.oilCardPointAccount.EquityOilCardPointFlow;
import com.rocogz.syy.equity.entity.oilCardUserCoupon.*;
import com.rocogz.syy.equity.entity.userCoupon.EquityUserCouponConsume;
import com.rocogz.syy.equity.entity.userCoupon.EquityUserCouponGiftInfo;
import com.rocogz.syy.equity.entity.userCoupon.EquityUserCouponGiftRecord;
import com.rocogz.syy.equity.entity.userCoupon.EquityUserCouponInfo;
import org.springframework.cloud.openfeign.SpringQueryMap;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import java.math.BigDecimal;
import java.util.List;
import java.util.Map;

/**
 * <dl>
 * <dd>Description: 权益服务对外接口 service</dd>
 * <dd>Company: 广州大诚若谷信息科技有限公司</dd>
 * <dd>@date：2020-04-05 20:08:42</dd>
 * <dd>@author：Andy</dd>
 * </dl>
 */
public interface IEquityClientService {

    /***********************************************  Andy 优惠券相关 Start   *****************************************************************/
    /**
     * 优惠券信息保存
     *
     * @param equityCouponInfo 优惠券信息
     * @return
     */
    @PostMapping("/api/equity/coupon/saveOrUpdateCoupon")
    void saveOrUpdateCoupon(@RequestBody EquityCouponInfo equityCouponInfo);

    /**
     * 优惠券列表查询
     *
     * @param code      编码
     * @param name      名称
     * @param status    状态
     * @param typeCode  类型
     * @param excludeTypeCode   排除的类型
     * @param startTime 创建开始时间
     * @param endTime   创建结束时间
     * @param limit
     * @param page
     * @return
     */
    @GetMapping("/api/equity/coupon/searchCoupon")
    PageTable<EquityCouponInfo> searchCoupon(@RequestParam(value = "code", required = false) String code,
                                             @RequestParam(value = "name", required = false) String name,
                                             @RequestParam(value = "status", required = false) String status,
                                             @RequestParam(value = "typeCode", required = false) String typeCode,
                                             @RequestParam(value = "excludeTypeCode", required = false) String excludeTypeCode,
                                             @RequestParam(value = "sourceCode", required = false) String sourceCode,
                                             @RequestParam(value = "startTime", required = false) String startTime,
                                             @RequestParam(value = "endTime", required = false) String endTime,
                                             @RequestParam(value = "limit", defaultValue = "20") int limit,
                                             @RequestParam(value = "page", defaultValue = "1") int page);

    /**
     * 提供创建活动选择优惠券
     *
     * @param areaCodeList 专区编号
     * @param status       状态（默认为上架）
     * @param name         名称
     * @param code         编码
     * @param limit        每页记录
     * @param page         页码
     * @return
     */
    @GetMapping("/api/equity/coupon/searchCouponForActivity")
    PageTable<EquityCouponInfo> searchCouponForActivity(@RequestParam(value = "areaCodeList") List<String> areaCodeList,
                                                        @RequestParam(value = "status", defaultValue = "TO_LIST") String status,
                                                        @RequestParam(value = "name", required = false) String name,
                                                        @RequestParam(value = "code", required = false) String code,
                                                        @RequestParam(value = "limit", defaultValue = "20") int limit,
                                                        @RequestParam(value = "page", defaultValue = "1") int page);

    /**
     * 根据优惠券编号list查询优惠券信息
     *
     * @param codeList 编号list
     * @return
     */
    @GetMapping("/api/equity/coupon/findCouponInfoByCodeList")
    Response<List<EquityCouponInfo>> findCouponInfoByCodeList(@RequestParam(value = "codeList") List<String> codeList);

    /**
     * 根据优惠券编号list和关键字查询优惠券信息
     * @param codeList
     * @param keyword
     * @return
     */
    @GetMapping("/api/equity/coupon/findCouponInfoByCodeListAndName")
    Response<List<EquityCouponInfo>> findCouponInfoByCodeListAndName(@RequestParam(value = "codeList") List<String> codeList,@RequestParam(value = "keyword") String keyword);

    /**
     * 根据优惠券编号删除
     *
     * @param code 优惠券编号
     * @return
     */
    @GetMapping("/api/equity/coupon/deleteCouponByCode")
    Response deleteCouponByCode(@RequestParam(value = "code") String code);

    /**
     * 根据优惠券编号更新状态
     *
     * @param code   优惠券编号
     * @param status 状态
     * @return
     */
    @GetMapping("/api/equity/coupon/updateCouponStatus")
    Response updateCouponStatus(@RequestParam(value = "code") String code, @RequestParam(value = "status") String status);

    /**
     * 根据优惠券编号查询
     *
     * @param code 优惠券编号
     * @return
     */
    @GetMapping("/api/equity/coupon/getCouponByCode")
    Response<EquityCouponInfo> getCouponByCode(@RequestParam(value = "code") String code);

    /**
     * 根据优惠券编号列表查询所有优惠券可用张数
     * @param couponCodeList    优惠券编号列表
     * @return
     */
    @GetMapping("/api/equity/couponLaunch/findAvailableQuantityByCouponCode")
    Response<Map<String, Integer>> findAvailableQuantityByCouponCode(@RequestParam(value = "couponCodeList") List<String> couponCodeList);

    /**
     * 追加投放数量
     *
     * @param couponCode 优惠券编号
     * @param quantity   数量
     * @return
     */
    @GetMapping("/api/equity/couponLaunch/addCouponTotalQuantity")
    Response addCouponTotalQuantity(@RequestParam(value = "couponCode") String couponCode,
                                    @RequestParam(value = "quantity") Integer quantity);

    /**
     * 减少投放数量
     *
     * @param couponCode 优惠券编号
     * @param quantity   数量
     * @return
     */
    @GetMapping("/api/equity/couponLaunch/subtractCouponTotalQuantity")
    Response subtractCouponTotalQuantity(@RequestParam(value = "couponCode") String couponCode,
                                         @RequestParam(value = "quantity") Integer quantity);

    /**
     * 更新优惠券领取数量
     * @param couponCode  优惠券编号
     * @param grantNumber   领取数量
     * @return
     */
    @GetMapping("/api/equity/couponLaunch/updateGrantNumber")
    Response updateGrantNumber(@RequestParam(value = "couponCode") String couponCode,
                               @RequestParam(value = "grantNumber") Integer grantNumber);

    /**
     * 根据状态查询优惠券类型
     *
     * @param status 类型状态：OPEN-启用，CLOSE-停用
     * @return
     */
    @GetMapping("/api/equity/couponType/findCouponTypeByStatus")
    Response<List<EquityCouponType>> findCouponTypeByStatus(@RequestParam(value = "status", required = false) String status);

    /**
     * 获取优惠券投放信息
     *
     * @param couponCodeList 优惠券编号
     * @return
     */
    @GetMapping("/api/equity/couponLaunch/getCouponLaunchInfoMap")
    Response<Map<String, EquityCouponLaunchInfo>> getCouponLaunchInfoMap(@RequestParam(value = "couponCodeList") List<String> couponCodeList);

    /*********************************************   优惠券相关 end   *************************************************************************/

    /*********************************************   优惠券发放申请单 start  *******************************************************************/

    /**
     * 新增
     *
     * @param equityBatchDistributionCouponApply 申请单主信息
     * @return
     */
    @PostMapping("/api/equity/batchDistributionCouponApply/saveBatchDistributionCouponApply")
    Response saveBatchDistributionCouponApply(@RequestBody EquityBatchDistributionCouponApply equityBatchDistributionCouponApply);

    /**
     * 申请单列表查询
     *
     * @param applyNo    申请单编号
     * @param sourceCode 投放主体
     * @param status     状态
     * @param statusList 状态列表
     * @param startDate  开始日期
     * @param endDate    结束日期
     * @param nature     发券性质c
     * @param limit
     * @param page
     * @return
     */
    @GetMapping("/api/equity/batchDistributionCouponApply/searchBatchDistributionCouponApply")
    PageTable<EquityBatchDistributionCouponApply> searchBatchDistributionCouponApply(@RequestParam(value = "applyNo", required = false) String applyNo,
                                                                                     @RequestParam(value = "sourceCode", required = false) String sourceCode,
                                                                                     @RequestParam(value = "miniAppid", required = false) String miniAppid,
                                                                                     @RequestParam(value = "status", required = false) String status,
                                                                                     @RequestParam(value = "statusList", required = false) List<String> statusList,
                                                                                     @RequestParam(value = "startDate", required = false) String startDate,
                                                                                     @RequestParam(value = "endDate", required = false) String endDate,
                                                                                     @RequestParam(value = "nature", required = false) String nature,
                                                                                     @RequestParam(value = "limit", defaultValue = "20") int limit,
                                                                                     @RequestParam(value = "page", defaultValue = "1") int page);

    /**
     * 批量发券申请单-通过
     *
     * @param applyNo     申请单编号
     * @param operateUser 操作用户
     * @return
     */
    @GetMapping("/api/equity/batchDistributionCouponApply/batchDistributionApplyPass")
    Response batchDistributionApplyPass(@RequestParam(value = "applyNo") String applyNo, @RequestParam(value = "operateUser") String operateUser);

    /**
     * 申请不通过
     *
     * @param applyNo 申请单编号
     * @return
     */
    @GetMapping("/api/equity/batchDistributionCouponApply/batchDistributionApplyDisagree")
    Response batchDistributionApplyDisagree(@RequestParam(value = "applyNo") String applyNo, @RequestParam(value = "operateUser") String operateUser);

    /**
     * 批量发券申请单-再次处理
     *
     * @param applyNo     申请单编号
     * @param operateUser 操作用户
     * @return
     */
    @GetMapping("/api/equity/batchDistributionCouponApply/batchDistributionApplyAgain")
    Response batchDistributionApplyAgain(@RequestParam(value = "applyNo") String applyNo, @RequestParam(value = "operateUser") String operateUser);

    /**
     * 批量发券申请单-作废
     *
     * @param applyNo     申请单编号
     * @param operateUser 操作用户
     * @return
     */
    @GetMapping("/api/equity/batchDistributionCouponApply/batchDistributionApplyCancel")
    Response batchDistributionApplyCancel(@RequestParam(value = "applyNo") String applyNo, @RequestParam(value = "operateUser") String operateUser);

    /**
     * 批量发券申请单-待处理
     * @param applyNo       申请单编号
     * @param sourceCode    发放主体
     * @param operateUser   操作用户
     * @return
     */
    @GetMapping("/api/equity/batchDistributionCouponApply/batchDistributionApplyPending")
    Response batchDistributionApplyPending(@RequestParam(value = "applyNo") String applyNo, @RequestParam(value = "sourceCode") String sourceCode, @RequestParam(value = "operateUser") String operateUser);

    /**
     * 调整发券性质
     * @param applyNo
     * @param nature
     * @param rmk
     * @return
     */
    @RequestMapping("/api/equity/batchDistributionCouponApply/adjustmentNature")
    Response adjustmentNature(@RequestParam(value = "applyNo") String applyNo,
                                     @RequestParam(value = "nature") String nature,
                                     @RequestParam(value = "rmk", required = false) String rmk,
                                     @RequestParam(value = "operateUser") String operateUser);

    /**
     * 一键发送短信
     * @param applyNo       申请单编号
     * @param totalPerson   申请单总人数
     * @param type          类型
     * @param executeTime   发放时间
     * @return
     */
    @RequestMapping("/api/equity/batchDistributionCouponApply/sendMessage")
    Response sendMessage(@RequestParam(value = "applyNo") String applyNo,
                         @RequestParam(value = "totalPerson") Integer totalPerson,
                         @RequestParam(value = "type") String type,
                         @RequestParam(value = "executeTime", required = false) String executeTime,
                         @RequestParam(value = "operateUser") String operateUser);

    /**
     * 审核
     * @param detailCode        清单编号
     * @param mobile            手机号
     * @param approveResult     审核结果
     * @param description       审核备注
     * @param operateUser       操作人
     * @return
     */
    @GetMapping("/api/equity/batchDistributionCouponApply/approve")
    Response approve(@RequestParam(value = "detailCode") String detailCode,
                     @RequestParam(value = "mobile") String mobile,
                    @RequestParam(value = "approveResult") String approveResult,
                    @RequestParam(value = "description", required = false) String description,
                    @RequestParam(value = "operateUser") String operateUser);

    /**
     * 根据申请单编号查询发放申请单
     *
     * @param applyNo 申请单编号
     * @return
     */
    @GetMapping("/api/equity/batchDistributionCouponApply/getBatchDistributionApplyByApplyNo")
    Response<EquityBatchDistributionCouponApply> getBatchDistributionApplyByApplyNo(@RequestParam(value = "applyNo") String applyNo);

    /**
     * 根据申请单编号查询批量发券优惠券结果记录
     *
     * @param applyNo 申请单编号
     * @return
     */
    @GetMapping("/api/equity/batchDistributionResultRecord/findBatchDistributionCouponResultRecordByApplyNo")
    Response<List<EquityBatchDistributionCouponResultRecord>> findBatchDistributionCouponResultRecordByApplyNo(@RequestParam(value = "applyNo") String applyNo);

    /**
     * 根据申请单编号查询批量发券清单列表查询
     *
     * @param applyNo       申请单编号
     * @param mobile        手机号
     * @param couponCode    优惠券编号
     * @param limit
     * @param page
     * @return
     */
    @GetMapping("/api/equity/batchDistributionCouponApplyDetail/searchBatchDistributionApplyDetail")
    PageTable<EquityBatchDistributionCouponApplyDetail> searchBatchDistributionApplyDetail(@RequestParam(value = "applyNo") String applyNo,
                                                                                           @RequestParam(value = "mobile", required = false) String mobile,
                                                                                           @RequestParam(value = "couponCode", required = false) String couponCode,
                                                                                           @RequestParam(value = "limit", defaultValue = "20") int limit,
                                                                                           @RequestParam(value = "page", defaultValue = "1") int page);

    /**
     * 查询审批清单
     * @param miniAppid
     * @param sourceCode
     * @param couponCode
     * @param couponName
     * @param typeCode
     * @param mobile
     * @param applyNo
     * @param status
     * @param applyStartDate
     * @param applyEndDate
     * @param limit
     * @param page
     * @return
     */
    @GetMapping("/api/equity/batchDistributionCouponApplyDetail/searchDetailForApprove")
    PageTable<DetailForApproveDto> searchDetailForApprove(@RequestParam(value = "miniAppid", required = false) String miniAppid,
                                                                 @RequestParam(value = "sourceCode", required = false) String sourceCode,
                                                                 @RequestParam(value = "couponCode", required = false) String couponCode,
                                                                 @RequestParam(value = "couponName", required = false) String couponName,
                                                                 @RequestParam(value = "typeCode", required = false) String typeCode,
                                                                 @RequestParam(value = "mobile", required = false) String mobile,
                                                                 @RequestParam(value = "applyNo", required = false) String applyNo,
                                                                 @RequestParam(value = "status", required = false) String status,
                                                                 @RequestParam(value = "applyStartDate", required = false) String applyStartDate,
                                                                 @RequestParam(value = "applyEndDate", required = false) String applyEndDate,
                                                                 @RequestParam(value = "grantStartDate", required = false) String grantStartDate,
                                                                 @RequestParam(value = "grantEndDate", required = false) String grantEndDate,
                                                                 @RequestParam(value = "receiveStartDate", required = false) String receiveStartDate,
                                                                 @RequestParam(value = "receiveEndDate", required = false) String receiveEndDate,
                                                                 @RequestParam(value = "receiveStatus", required = false) String receiveStatus,
                                                                 @RequestParam(value = "nature", required = false) String nature,
                                                                 @RequestParam(value = "approveFlag", required = false) String approveFlag,
                                                                 @RequestParam(value = "applyDetailCode", required = false) String applyDetailCode,
                                                                 @RequestParam(value = "limit", defaultValue = "20") int limit,
                                                                 @RequestParam(value = "page", defaultValue = "1") int page);

    /**
     * 根据发放主体、手机号统计发放次数
     * @param applyNo       申请单号
     * @param sourceCode    发放主体
     * @param mobile        手机号
     * @return
     */
    @GetMapping("/api/equity/batchDistributionCouponApplyDetail/countTotalQuantityByMobileAndSourceCode")
    Response<Integer> countTotalQuantityByMobileAndSourceCode(@RequestParam(value = "applyNo") String applyNo,
                                                              @RequestParam(value = "sourceCode") String sourceCode,
                                                              @RequestParam(value = "mobile") String mobile);
    /**
     * 导出未领取的清单
     * @param applyNoList   申请单编号列表
     * @return
     */
    @RequestMapping("/api/equity/batchDistributionCouponApplyDetail/exportUnReceiveDetail")
    Response<List<UnReceiveDetailDto>> exportUnReceiveDetail(@RequestParam(value = "applyNoList") List<String> applyNoList);

    /**
     * 批量发券申请单操作日志列表查询
     *
     * @param applyNo 申请单编号
     * @param limit
     * @param page
     * @return
     */
    @GetMapping("/api/equity/batchDistributionApplyOperateLog/searchBatchDistributionApplyDetail")
    PageTable<EquityBatchDistributionApplyOperateLog> searchBatchDistributionApplyOperateLog(@RequestParam(value = "applyNo") String applyNo,
                                                                                             @RequestParam(value = "limit", defaultValue = "20") int limit,
                                                                                             @RequestParam(value = "page", defaultValue = "1") int page);

    /*********************************************   优惠券发放申请单 end   ********************************************************************/

    /*********************************************   用户券相关 start   ********************************************************************/

    /**
     * 发放用户券
     *
     * @param userCouponDto 用户券发放参数dto
     * @return
     * @throws Exception
     */
    @PostMapping("/api/equity/userCoupon/grantUserCoupon")
    Response<EquityUserCouponInfo> grantUserCoupon(@RequestBody UserCouponDto userCouponDto);

    /**
     * 用户券列表查询-后台管理
     *
     * @param code           用户券编号
     * @param userCode       会员编号
     * @param mobile         手机号
     * @param couponCode     优惠券编号
     * @param couponTypeCode 优惠券类型
     * @param activityCode   活动编号
     * @param status         状态
     * @param sourceCode     发放主体
     * @param applyNo        申请单编号
     * @param nature         发券性质
     * @param limit
     * @param page
     * @return
     */
    @GetMapping("/api/equity/userCoupon/searchUserInfo")
    PageTable<EquityUserCouponInfo> searchUserInfo(@RequestParam(value = "code", required = false) String code,
                                                   @RequestParam(value = "userCode", required = false) String userCode,
                                                   @RequestParam(value = "mobile", required = false) String mobile,
                                                   @RequestParam(value = "couponCode", required = false) String couponCode,
                                                   @RequestParam(value = "couponTypeCode", required = false) String couponTypeCode,
                                                   @RequestParam(value = "activityCode", required = false) String activityCode,
                                                   @RequestParam(value = "status", required = false) String status,
                                                   @RequestParam(value = "sourceCode", required = false) String sourceCode,
                                                   @RequestParam(value = "miniAppids", required = false) String miniAppids,
                                                   @RequestParam(value = "receiveFlag", required = false) String receiveFlag,
                                                   @RequestParam(value = "clickStatus", required = false) String clickStatus,
                                                   @RequestParam(value = "receiveStartDate", required = false) String receiveStartDate,
                                                   @RequestParam(value = "receiveEndDate", required = false) String receiveEndDate,
                                                   @RequestParam(value = "applyNo", required = false) String applyNo,
                                                   @RequestParam(value = "nature", required = false) String nature,
                                                   @RequestParam(value = "applyDetailCode", required = false) String applyDetailCode,
                                                   @RequestParam(value = "limit", defaultValue = "20") int limit,
                                                   @RequestParam(value = "page", defaultValue = "1") int page);

    /**
     * 导出数据
     * @return
     */
    @GetMapping("/api/equity/userCoupon/exportTotalRow")
    Map<String, Object> exportTotalRow(@RequestParam(value = "code",required = false) String code,
                                              @RequestParam(value = "userCode", required = false) String userCode,
                                              @RequestParam(value = "mobile", required = false) String mobile,
                                              @RequestParam(value = "couponCode", required = false) String couponCode,
                                              @RequestParam(value = "couponTypeCode", required = false) String couponTypeCode,
                                              @RequestParam(value = "activityCode", required = false) String activityCode,
                                              @RequestParam(value = "status", required = false) String status,
                                              @RequestParam(value = "sourceCode", required = false) String sourceCode,
                                              @RequestParam(value = "miniAppids", required = false) String miniAppids,
                                              @RequestParam(value = "receiveFlag", required = false) String receiveFlag,
                                              @RequestParam(value = "clickStatus", required = false) String clickStatus,
                                              @RequestParam(value = "receiveStartDate", required = false) String receiveStartDate,
                                              @RequestParam(value = "receiveEndDate", required = false) String receiveEndDate,
                                              @RequestParam(value = "applyNo", required = false) String applyNo,
                                              @RequestParam(value = "nature", required = false) String nature,
                                              @RequestParam(value = "applyDetailCode", required = false) String applyDetailCode);

    /**
     * 导出数据
     * @return
     */
    @GetMapping("/api/equity/userCoupon/exportData")
    List<UserCouponExportDto> exportData(@RequestParam(value = "code",required = false) String code,
                                                     @RequestParam(value = "userCode", required = false) String userCode,
                                                     @RequestParam(value = "mobile", required = false) String mobile,
                                                     @RequestParam(value = "couponCode", required = false) String couponCode,
                                                     @RequestParam(value = "couponTypeCode", required = false) String couponTypeCode,
                                                     @RequestParam(value = "activityCode", required = false) String activityCode,
                                                     @RequestParam(value = "status", required = false) String status,
                                                     @RequestParam(value = "sourceCode", required = false) String sourceCode,
                                                     @RequestParam(value = "miniAppids", required = false) String miniAppids,
                                                     @RequestParam(value = "receiveFlag", required = false) String receiveFlag,
                                                     @RequestParam(value = "clickStatus", required = false) String clickStatus,
                                                     @RequestParam(value = "receiveStartDate", required = false) String receiveStartDate,
                                                     @RequestParam(value = "receiveEndDate", required = false) String receiveEndDate,
                                                     @RequestParam(value = "applyNo", required = false) String applyNo,
                                                     @RequestParam(value = "nature", required = false) String nature,
                                                     @RequestParam(value = "applyDetailCode", required = false) String applyDetailCode,
                                                     @RequestParam(value = "limit", defaultValue = "500") int limit,
                                                     @RequestParam(value = "page", defaultValue = "1") int page);

    /**
     * 用户券列表查询-车主端
     * @param type   类型
     * @param status 状态
     * @param openId openid
     * @param limit
     * @param page
     * @return
     */
    @GetMapping("/api/equity/userCoupon/searchUserInfoForCar")
    PageTable<EquityUserCouponInfo> searchUserInfoForCar(@RequestParam(value = "type") String type,
                                                         @RequestParam(value = "status", required = false) String status,
                                                         @RequestParam(value = "amountLimit", required = false) String amountLimit,
                                                         @RequestParam(value = "openId") String openId,
                                                         @RequestParam(value = "limit", defaultValue = "20") int limit,
                                                         @RequestParam(value = "page", defaultValue = "1") int page);

    /**
     * 领券中心查询用户券信息
     *
     * @param couponCodeList   优惠券编号List
     * @param activityCodeList 活动编号List
     * @param openId           openId
     * @param status           状态
     * @return
     */
    @GetMapping("/api/equity/userCoupon/findUserCouponForCouponCenter")
    Response<List<EquityUserCouponInfo>> findUserCouponForCouponCenter(@RequestParam(value = "couponCodeList") List<String> couponCodeList,
                                                                       @RequestParam(value = "activityCodeList") List<String> activityCodeList,
                                                                       @RequestParam(value = "openId") String openId,
                                                                       @RequestParam(value = "status", required = false) String status);

    /**
     * 根据用户券编号查询用户券信息-车主端
     *
     * @param code   用户券编号
     * @param openId
     * @return
     */
    @GetMapping("/api/equity/userCoupon/getUserCouponInfoByCodeAndOpenId")
    Response<EquityUserCouponInfo> getUserCouponInfoByCodeAndOpenId(@RequestParam(value = "code") String code,
                                                                    @RequestParam(value = "openId") String openId);

    /**
     * 根据用户券编号查询用户券信息
     *
     * @param code
     * @return
     */
    @GetMapping("/api/equity/userCoupon/getUserCouponInfoByCode")
    Response<EquityUserCouponInfo> getUserCouponInfoByCode(@RequestParam(value = "code") String code);

    /**
     * 根据用户券生效、失效日期更新券状态为已生效或者已失效
     *
     * @return
     */
    @GetMapping("/api/equity/userCoupon/updateUserCouponStatusByEffectiveDateOrInvalidDate")
    void updateUserCouponStatusByEffectiveDateOrInvalidDate();

    /**
     * 批量兑换代金券
     * @param platformCode      使用平台
     * @param redeemCodeList    兑换码列表
     * @param orderCode         订单编号
     * @param openId
     * @return
     */
    @PostMapping("/api/equity/userCoupon/batchConsumeCashByRedeemCode")
    Response batchConsumeCashByRedeemCode(@RequestParam(value = "platformCode") String platformCode,
                                                 @RequestParam(value = "redeemCodeList") List<String> redeemCodeList,
                                                 @RequestParam(value = "orderCode") String orderCode,
                                                 @RequestParam(value = "orderAmount") BigDecimal orderAmount,
                                                 @RequestParam(value = "openId") String openId);


    /**
     * 根据兑换码列表、openId查询优惠券信息
     * @param openId
     * @param redeemCodeList    兑换码列表
     */
    @GetMapping("/api/equity/userCoupon/findCashCouponByRedeemCode")
    Response<List<EquityCashCouponAmount>> findCashCouponByRedeemCode(@RequestParam(value = "openId") String openId,
                                                                      @RequestParam(value = "redeemCodeList") List<String> redeemCodeList);

    /**
     * 根据手机号更新用户券会员信息
     *
     * @param mobile         手机号
     * @param miniAppid      小程序ID
     * @param idType         证件类型
     * @param idCardExplicit 证件号码
     * @param openId         openid
     * @return
     */
    @GetMapping("/api/equity/userCouponUserInfo/updateUserCouponUserInfoByMobile")
    Response updateUserCouponUserInfoByMobile(@RequestParam(value = "mobile") String mobile,
                                              @RequestParam(value = "miniAppid") String miniAppid,
                                              @RequestParam(value = "userCode") String userCode,
                                              @RequestParam(value = "idType", required = false) String idType,
                                              @RequestParam(value = "idCardExplicit", required = false) String idCardExplicit,
                                              @RequestParam(value = "openId") String openId);

    /**
     * 统计用户券各类型待使用的张数
     * @param openId
     */
    @GetMapping("/api/equity/userCouponUserInfo/countUserCouponNumberByOpenId")
    Response<Map<String, String>> countUserCouponNumberByOpenId(@RequestParam(value = "openId") String openId);

    /**
     * 更新用户券状态updateUserCouponStatus
     *
     * @param code        用户券编号
     * @param couponCode  优惠券编号
     * @param status      状态
     * @param operateUser 操作用户
     * @return
     */
    @GetMapping("/api/equity/userCoupon/updateUserCouponStatus")
    Response updateUserCouponStatus(@RequestParam(value = "code") String code,
                                    @RequestParam(value = "couponCode") String couponCode,
                                    @RequestParam(value = "status") String status,
                                    @RequestParam(value = "operateUser") String operateUser);

    /**
     * 更新用户券有效期
     *
     * @param userCouponCode 用户券编号
     * @param couponCode     优惠券信息
     * @param effectiveDate  生效日期
     * @param invalidDate    失效日期
     * @return
     */
    @GetMapping("/api/equity/userCouponUseInfo/updateUserCouponDate")
    Response updateUserCouponDate(@RequestParam(value = "userCouponCode") String userCouponCode,
                                  @RequestParam(value = "couponCode") String couponCode,
                                  @RequestParam(value = "effectiveDate") String effectiveDate,
                                  @RequestParam(value = "invalidDate") String invalidDate,
                                  @RequestParam(value = "operateUser") String operateUser);

    /**
     * 根据核销码查询优惠券信息
     *
     * @param redeemCode 核销码
     * @return
     */
    @GetMapping("/api/equity/userCoupon/getUserCouponByRedeemCode")
    Response<EquityUserCouponInfo> getUserCouponByRedeemCode(@RequestParam(value = "redeemCode") String redeemCode);

    /**
     * 根据核销码列表查询优惠券信息
     * @param redeemCodeList   核销码列表
     * @return
     */
    @GetMapping("/api/equity/userCoupon/getUserCouponByRedeemCodeList")
    Response<List<EquityUserCouponInfo>> getUserCouponByRedeemCodeList(@RequestParam(value = "redeemCodeList") List<String> redeemCodeList);

    /**
     * 根据核销码、第三方订单查询优惠券信息
     * @param redeemCode   核销码
     * @param thirdCode    第三方编号
     * @return
     */
    @GetMapping("/api/equity/userCoupon/getUserCouponByRedeemCodeThirdCode")
    Response<EquityUserCouponInfo> getUserCouponByRedeemCodeThirdCode(@RequestParam(value = "redeemCode", required = false) String redeemCode,
                                                                             @RequestParam(value = "thirdCode", required = false) String thirdCode);

    /**
     * 用户券消耗使用
     *
     * @param platformCode 使用平台
     * @param redeemCode   券码
     * @param orderCode    订单编号
     * @param useWay       使用方式：MANUAL-手工，SYSTEM-系统
     * @return
     */
    @PostMapping("/api/equity/userCoupon/consumeByRedeemCode")
    Response<String> consumeByRedeemCode(@RequestParam(value = "platformCode") String platformCode,
                                 @RequestParam(value = "redeemCode") String redeemCode,
                                 @RequestParam(value = "orderCode") String orderCode,
                                 @RequestParam(value = "useWay") String useWay);

    /**
     * 用户券消耗使用-根据第三方权益编号
     * @param platformCode  使用平台
     * @param thirdCode     第三方权益编号
     * @param useWay        使用方式：MANUAL-手工，SYSTEM-系统（默认传入）
     * @return
     */
    @PostMapping("/api/equity/userCoupon/consumeByThirdCode")
    Response<String> consumeByThirdCode(@RequestParam(value = "platformCode") String platformCode,
                                       @RequestParam(value = "thirdCode") String thirdCode,
                                       @RequestParam(value = "useWay") String useWay);

    /**
     * 用户券退还
     * @param platformCode      平台码
     * @param codeList          用户券编号
     * @param orderCode         订单编号
     * @return
     */
    @PostMapping("/api/equity/userCoupon/userCouponReturn")
    Response userCouponReturn(@RequestParam(value = "platformCode") String platformCode,
                                     @RequestParam(value = "codeList") List<String> codeList,
                                     @RequestParam(value = "orderCode") String orderCode);


    /**
     * 获取未领取的用户券
     * @param openId
     * @return
     */
    @GetMapping("/api/equity/userCoupon/findUnrReceiveUserCoupon")
    Response<Map<String, Object>> findUnrReceiveUserCoupon(@RequestParam(value = "openId") String openId);


    /**
     * 更新用户券、油卡券领取状态
     * @param openId
     * @return
     */
    @GetMapping("/api/equity/userCoupon/updateUserCouponReceiveFlag")
    Response<Map<String, Integer>> updateUserCouponReceiveFlag(@RequestParam(value = "openId") String openId);

    /**
     * 更新用户券、油卡券领取状态(根据车牌）
     * @param openId
     * @param licenseNoList 车牌列表
     * @return
     */
    @GetMapping("/api/equity/userCoupon/updateUserCouponReceiveFlagByLicenseNo")
    Response<Map<String, Integer>> updateUserCouponReceiveFlagByLicenseNo(@RequestParam(value = "openId") String openId, @RequestParam(value = "licenseNoList") List<String> licenseNoList);

    /**
     * 用户券消耗记录列表
     *
     * @param userCouponCode 用户券编号
     * @param limit
     * @param page
     * @return
     */
    @GetMapping("/api/equity/userCouponConsume/searchUserCouponConsume")
    PageTable<EquityUserCouponConsume> searchUserCouponConsume(@RequestParam(value = "userCouponCode") String userCouponCode,
                                                               @RequestParam(value = "limit", defaultValue = "20") int limit,
                                                               @RequestParam(value = "page", defaultValue = "1") int page);

    /**
     * 根据用户券编号查询订单编号
     *
     * @param userCouponCode 用户券编号
     * @return
     */
    @GetMapping("/api/equity/userCouponConsume/getOrderCodeByUserCouponCode")
    Response<List<String>> getOrderCodeByUserCouponCode(@RequestParam(value = "userCouponCode") String userCouponCode);

    /**
     * 根据请求码查询
     * @param serialNo
     * @return
     */
    @GetMapping("/api/equity/userCoupon/userCouponGiftInfo/findGiftInfoBySerialNo")
    Response<List<EquityUserCouponGiftInfo>> findGiftInfoBySerialNo(@RequestParam("serialNo") String serialNo);

    /**
     * 根据请求码查询
     * @param serialNoList
     * @return
     */
    @GetMapping("/api/equity/userCoupon/userCouponGiftInfo/findGiftInfoBySerialNoList")
    Response<List<EquityUserCouponGiftInfo>> findGiftInfoBySerialNoList(@RequestParam(value="serialNoList") List<String> serialNoList);

    /**
     * 查询失败的调用记录同时更新为处理中
     * @return
     */
    @GetMapping("/api/equity/userCoupon/userCouponGiftInfo/findFailGiftInfoToInProcess")
    Response<List<EquityUserCouponGiftInfo>> findFailGiftInfoToInProcess();

    /**
     * 根据请求码更新状态
     * @param serialNo  请求码
     * @param status    状态
     * @param rmk       备注
     * @return
     */
    @GetMapping("/api/equity/userCoupon/userCouponGiftInfo/updateBySerialNo")
    Response updateBySerialNo(@RequestParam(value = "serialNo") String serialNo, @RequestParam(value = "status") String status, @RequestParam(value = "rmk", required = false) String rmk);

    /**
     * 根据优惠券编码查询第三方权益编号
     * @param couponCode
     * @return
     */
    @GetMapping("/api/equity/couponThirdEquity/getThirdEquityCodeByCouponCode")
    Response<String> getThirdEquityCodeByCouponCode(@RequestParam(value="couponCode") String couponCode);

    /**
     * 根据请求码查询调用第三方清单
     * @param serialNo
     * @return
     */
    @GetMapping("/api/equity/userCoupon/userCouponGiftRecord/findGiftRecordBySerialNo")
    Response<List<EquityUserCouponGiftRecord>> findGiftRecordBySerialNo(@RequestParam(value="serialNo") String serialNo);

    /**
     * 查询发放限制配置
     * @param sourceCode    发放主体
     * @param status        状态
     * @param limit
     * @param page
     * @return
     */
    @GetMapping("/api/equity/batchDistributionApplyLimitConfig/searchLimitConfig")
    PageTable<EquityBatchDistributionApplyLimitConfig> searchLimitConfig(@RequestParam(value = "sourceCode",required = false) String sourceCode,
                                                                        @RequestParam(value = "status", required = false) String status,
                                                                        @RequestParam(value = "limit", defaultValue = "20") int limit,
                                                                        @RequestParam(value = "page", defaultValue = "1") int page);

    /**
     * 根据发放主体查询
     * @param sourceCode    发放主体
     * @return
     */
    @GetMapping("/api/equity/batchDistributionApplyLimitConfig/getLimitConfigBySourceCode")
    Response<EquityBatchDistributionApplyLimitConfig> getLimitConfigBySourceCode(@RequestParam(value = "sourceCode") String sourceCode);

    /**
     * 根据限制数量编号查询
     * @param code 编号
     * @return
     */
    @GetMapping("/api/equity/batchDistributionApplyLimitConfig/getLimitConfigByCode")
    Response<EquityBatchDistributionApplyLimitConfig> getLimitConfigByCode(@RequestParam(value = "code") String code);

    /**
     * 新增、编辑
     *
     * @param entity
     * @return
     */
    @PostMapping("/api/equity/batchDistributionApplyLimitConfig/saveOrUpdateLimitConfig")
    Response saveOrUpdateLimitConfig(@RequestBody EquityBatchDistributionApplyLimitConfig entity);

    /**
     * 更新状态
     * @param code          编号
     * @param status        状态
     * @param operateUser   操作用户
     * @return
     */
    @PostMapping("/api/equity/batchDistributionApplyLimitConfig/updateLimitConfigStatus")
    Response updateLimitConfigStatus(@RequestParam(value = "code") String code,
                                 @RequestParam(value = "status") String status,
                                 @RequestParam(value = "operateUser") String operateUser);

    /*********************************************   用户券相关 end   ********************************************************************/

    /*********************************************  尹世豪 发放主体相关 Start   ****************************************************************/
    /**
     * 左侧发放主体树列表
     *
     * @return
     */
    @GetMapping("/api/equity/issuingBody/findIssuingBodyTree/{type}")
    Response<List<IssuingBodyTreeDto>> findIssuingBodyTree(@PathVariable("type") String type,
                                                           @RequestParam(value = "status", required = false) String status,
                                                           @RequestParam(value = "dataPermissions", required = false) List<String> dataPermissions);

    /**
     * 根据主体id获取发放主体信息
     *
     * @param id 主体id
     * @return 发放主体dto对象
     */
    @GetMapping("/api/equity/issuingBody/getIssuingBodyById/{id}")
    Response<IssuingBodyDto> getIssuingBodyById(@PathVariable("id") Integer id);

    /**
     * 根据主体编码获取发放主体信息
     *
     * @param code 主体编码
     * @return 发放主体dto对象
     */
    @GetMapping("/api/equity/issuingBody/getIssuingBodyByCode/{code}")
    Response<IssuingBodyDto> getIssuingBodyByCode(@PathVariable("code") String code);

    /**
     * 根据上级编码查询对应的下级主体列表
     *
     * @param pCode
     * @param limit
     * @param page
     * @return
     */
    @GetMapping("/api/equity/issuingBody/getIssuingBodyListByPCode/{pCode}")
    PageTable<IssuingBodyDto> getIssuingBodyListByPCode(@PathVariable("pCode") String pCode,
                                                        @RequestParam(value = "limit", defaultValue = "20") int limit,
                                                        @RequestParam(value = "page", defaultValue = "1") int page);

    /**
     * 根据主体id删除主体
     *
     * @param id
     * @return
     */
    @GetMapping("/api/equity/issuingBody/deleteIssuingBodyById/{id}")
    Response deleteIssuingBodyById(@PathVariable("id") Integer id);

    /**
     * 根据主体id修改状态
     *
     * @param issuingBody
     * @return
     */
    @PostMapping("/api/equity/issuingBody/changeIssuingBodyStatusById")
    Response changeIssuingBodyStatusById(@RequestBody IssuingBody issuingBody);

    /**
     * 创建或更新主体
     *
     * @param issuingBody
     * @return
     */
    @PostMapping("/api/equity/issuingBody/saveOrUpdateIssuingBody")
    Response<IssuingBody> saveOrUpdateIssuingBody(@RequestBody IssuingBody issuingBody);

    /**
     * 获取发放主体列表(分页),不分类型
     *
     * @param code  主体编码 (精确匹配)
     * @param name  主体名称(模糊匹配)
     * @param limit
     * @param page
     * @return
     */
    @GetMapping("/api/equity/issuingBody/findIssuingBodyList")
    PageTable<IssuingBody> findIssuingBodyList(@RequestParam(value = "code", required = false) String code,
                                               @RequestParam(value = "name", required = false) String name,
                                               @RequestParam(value = "dataPermissions", required = false) List<String> dataPermissions,
                                               @RequestParam(value = "type", required = false) String type,
                                               @RequestParam(value = "status", required = false) String status,
                                               @RequestParam(value = "limit", defaultValue = "20") int limit,
                                               @RequestParam(value = "page", defaultValue = "1") int page);

    /**
     * 根据发放主体编码列表查询 发放主体列表
     *
     * @param codes
     * @return
     */
    @PostMapping("/api/equity/issuingBody/getIssuingBodyListByCodes")
    Response<List<IssuingBody>> getIssuingBodyListByCodes(@RequestBody List<String> codes);

    /**
     * 根据机构代码 查询 发放主体编号
     * @param linkedBodyCode
     * @return
     */
    @GetMapping("/api/equity/issuingBody/getCodeByLinkedBodyCode")
    Response<IssuingBody> getCodeByLinkedBodyCode(@RequestParam("linkedBodyCode") String linkedBodyCode);

    /**
     * 为平台商户自动创建一个发放主体
     * @param issuingBody
     * @return
     */
    @PostMapping("/api/equity/issuingBody/createBusinessIssuingBody")
    Response createBusinessIssuingBody(@RequestBody IssuingBody issuingBody);

    /*********************************************   发放主体相关 End   *******************************************************************/

    /*********************************************   Andy 电子加油券相关 start   *******************************************************************/
    /**
     * 验证订单请求的操作是否已存在
     *
     * @param orderCode 订单编号
     * @param type      操作类型：ADD-发券，CANCEL-作废，BACK-退回
     * @return
     */
    @GetMapping("/api/equity/oilCardRequestRecord/checkOrderIsExistByOrderCode")
    Response<EquityOilCardUserCouponRequestRecord> checkOrderIsExistByOrderCode(@RequestParam(value = "orderCode") String orderCode, @RequestParam(value = "type") String type);

    /**
     * 新增油卡请求参数记录
     *
     * @param entity
     * @return
     * @throws Exception
     */
    @PostMapping("/api/equity/oilCardRequestRecord/saveOilCardRequestRecord")
    Response saveOilCardRequestRecord(@RequestBody EquityOilCardUserCouponRequestRecord entity);

    /**
     * 根据订单编号查询油卡请求信息记录
     *
     * @param oilOrderCode 订单编号
     * @return
     */
    @GetMapping("/api/equity/oilCardRequestRecord/getOilCardUserCouponRequestRecordByOilOrderCode")
    Response<EquityOilCardUserCouponRequestRecord> getOilCardUserCouponRequestRecordByOilOrderCode(@RequestParam(value = "oilOrderCode") String oilOrderCode);

    /**
     * 更新请求记录发放状态
     * @param orderCode    订单编号
     * @param oldStatus    更新前状态（INIT-初始）
     * @param newStatus    更新发放状态（SUCCESS-成功，FAIL-失败）
     * @param rmk          备注ad
     */
    @GetMapping("/api/equity/oilCardRequestRecord/updateOilCardRequestRecordStatus")
    Response updateOilCardRequestRecordStatus(@RequestParam(value = "orderCode") String orderCode,
                                                     @RequestParam(value = "oldStatus") String oldStatus,
                                                     @RequestParam(value = "newStatus") String newStatus,
                                                     @RequestParam(value = "rmk") String rmk);

    /**
     * 查询油卡券发放失败的数据
     * @return
     */
    @GetMapping("/api/equity/oilCardRequestRecord/findOilCardUserCouponGranFailAndToInProcess")
    Response<List<EquityOilCardUserCouponRequestRecord>> findOilCardUserCouponGranFailAndToInProcess();

    /**
     * 新增操作流水
     *
     * @param orderCode 订单编号
     * @param type      操作类型
     * @return
     */
    @PostMapping("/api/equity/oilCardRequestOperateSerial/saveOperateSerial")
    void saveOperateSerial(@RequestParam(value = "orderCode") String orderCode,
                           @RequestParam(value = "type") String type);

    /**
     * 更新操作流水状态
     *
     * @param orderCode    订单编号
     * @param type         操作类型（ADD-发券、CANCEL-作废、BACK-退回）
     * @param resultStatus 结果（PROCESS-已通知，FAIL-通知失败，ABNORMAL-通知异常）
     * @param rmk          备注
     */
    @GetMapping("/api/equity/oilCardRequestOperateSerial/updateOperateSerial")
    Response updateOperateSerial(@RequestParam(value = "orderCode") String orderCode,
                                 @RequestParam(value = "type") String type,
                                 @RequestParam(value = "resultStatus") String resultStatus,
                                 @RequestParam(value = "rmk", required = false) String rmk);

    /**
     * 查询处理失败的结果并且更新为处理中
     *
     * @return
     */
    @GetMapping("/api/equity/oilCardRequestOperateSerial/findFailAndToInProcess")
    Response<List<EquityOilCardUserCoupon>> findFailAndToInProcess();

    /**
     * 根据油卡订单编号查询对应的油卡请求操作流水 (分页)
     *
     * @param oilOrderCode 油卡订单编号
     * @param limit        每页条数
     * @param page         当前页
     * @return
     */
    @GetMapping("/api/equity/oilCardRequestOperateSerial/findOilCardUserCouponRequestOperateSerialList")
    PageTable<EquityOilCardUserCouponRequestOperateSerial> findOilCardUserCouponRequestOperateSerialList(@RequestParam(value = "oilOrderCode") String oilOrderCode,
                                                                                                         @RequestParam(defaultValue = "20", value = "limit") int limit,
                                                                                                         @RequestParam(defaultValue = "1", value = "page") int page);

    /**
     * 保存油卡用户券
     * @param entity      电子兑换券对象
     * @return
     */
    @PostMapping("/api/equity/oilCardUserCoupon/saveOilCardUserCoupon")
    Response<EquityOilCardUserCoupon> saveOilCardUserCoupon(@RequestBody EquityOilCardUserCoupon entity);


    /**
     * 发放积分(下单返利)
     * @return
     */
    @PostMapping("/api/equity/oilCardUserCoupon/saveOilCardUserCouponForRebate")
    Response<EquityOilCardUserCoupon> saveOilCardUserCouponForRebate(@RequestBody OilCardPointGrantParamDto entity);

    /**
     * 油卡系统用户券列表
     *
     * @param status 状态
     * @param openId openid
     * @param limit
     * @param page
     * @return
     */
    @GetMapping("/api/equity/oilCardUserCoupon/searchOilUserCoupon")
    PageTable<EquityOilCardUserCoupon> searchOilUserCoupon(@RequestParam(value = "status", required = false) String status,
                                                           @RequestParam(value = "openId") String openId,
                                                           @RequestParam(value = "limit", defaultValue = "20") int limit,
                                                           @RequestParam(value = "page", defaultValue = "1") int page);

    /**
     * 后台查询电子券(专用)列表
     *
     * @param code         电子券编码
     * @param oilOrderCode 油卡订单编号
     * @param name         电子券名称
     * @param userCode     会员编码
     * @param openId       微信openid
     * @param mobile       手机号
     * @param licenseNo    车牌号
     * @param policyNo     保单号
     * @param status       状态
     * @param limit        每页条数
     * @param page         当前页
     * @return
     */
    @GetMapping("/api/equity/oilCardUserCoupon/findOilUserCouponList")
    PageTable<EquityOilCardUserCoupon> findOilUserCouponList(@RequestParam(required = false, value = "code") String code,
                                                             @RequestParam(required = false, value = "oilOrderCode") String oilOrderCode,
                                                             @RequestParam(required = false, value = "name") String name,
                                                             @RequestParam(required = false, value = "userCode") String userCode,
                                                             @RequestParam(required = false, value = "openId") String openId,
                                                             @RequestParam(required = false, value = "mobile") String mobile,
                                                             @RequestParam(required = false, value = "licenseNo") String licenseNo,
                                                             @RequestParam(required = false, value = "policyNo") String policyNo,
                                                             @RequestParam(required = false, value = "status") String status,
                                                             @RequestParam(required = false, value = "receiveFlag") String receiveFlag,
                                                             @RequestParam(required = false, value = "receiveStartTime") String receiveStartTime,
                                                             @RequestParam(required = false, value = "receiveEndTime") String receiveEndTime,
                                                             @RequestParam(required = false, value = "sourceCode") String sourceCode,
                                                             @RequestParam(required = false, value = "miniAppids") String miniAppids,
                                                             @RequestParam(defaultValue = "20", value = "limit") int limit,
                                                             @RequestParam(defaultValue = "1", value = "page") int page);

    /**
     * 后台查询电子券(专用) 导出
     * @param code         油卡用户券编码
     * @param oilOrderCode 油卡订单编号
     * @param name         电子券名称
     * @param userCode     会员编码
     * @param openId       微信openid
     * @param mobile       手机号
     * @param licenseNo    车牌号
     * @param policyNo     保单号
     * @param status       状态
     * @param sourceCode   发放主体
     * @param miniAppids   小程序ID
     * @param limit        每页条数
     * @param page         当前页
     * @return
     */
    @GetMapping("/api/equity/oilCardUserCoupon/exportList")
    List<EquityOilCardUserCoupon> exportList(@RequestParam(required = false, value = "code") String code,
                                             @RequestParam(required = false, value = "oilOrderCode") String oilOrderCode,
                                             @RequestParam(required = false, value = "name") String name,
                                             @RequestParam(required = false, value = "userCode") String userCode,
                                             @RequestParam(required = false, value = "openId") String openId,
                                             @RequestParam(required = false, value = "mobile") String mobile,
                                             @RequestParam(required = false, value = "licenseNo") String licenseNo,
                                             @RequestParam(required = false, value = "policyNo") String policyNo,
                                             @RequestParam(required = false, value = "status") String status,
                                             @RequestParam(required = false, value = "receiveFlag") String receiveFlag,
                                             @RequestParam(required = false, value = "receiveStartTime") String receiveStartTime,
                                             @RequestParam(required = false, value = "receiveEndTime") String receiveEndTime,
                                             @RequestParam(required = false, value = "sourceCode") String sourceCode,
                                             @RequestParam(required = false, value = "miniAppids") String miniAppids,
                                             @RequestParam(defaultValue = "20", value = "limit") int limit,
                                             @RequestParam(defaultValue = "1", value = "page") int page);

    /**
     * 根据油卡用户券编码查询油卡用户券信息
     *
     * @param code 油卡用户券编码
     * @return
     */
    @GetMapping("/api/equity/oilCardUserCoupon/getOilCardUserCouponByCode")
    Response<EquityOilCardUserCoupon> getOilCardUserCouponByCode(@RequestParam(value = "code") String code);

    /**
     * 根据用户统计待使用的油卡总额
     *
     * @param openId
     * @return
     */
    @GetMapping("/api/equity/oilCardUserCoupon/getOilCardTotalAmountByOpenId")
    Response<BigDecimal> getOilCardTotalAmountByOpenId(@RequestParam(value = "openId") String openId);

    /**
     * 根据用户券兑换码查询油卡用户券
     *
     * @param redeemCode 用户券兑换码
     * @param openId     openid
     * @return
     */
    @GetMapping("/api/equity/oilCardUserCoupon/getOilCardUserCouponByRedeemCode")
    Response<EquityOilCardUserCoupon> getOilCardUserCouponByRedeemCode(@RequestParam(value = "redeemCode") String redeemCode,
                                                                       @RequestParam(value = "openId") String openId);

    /**
     * 根据用户券编号查询油卡用户券详情
     *
     * @param redeemCode 用户券兑换码
     * @param orderCode  订单编号
     * @return
     */
    @GetMapping("/api/equity/oilCardUserCoupon/getOilCardUserCouponDetail")
    Response<EquityOilCardUserCoupon> getOilCardUserCouponDetail(@RequestParam(value = "redeemCode", required = false) String redeemCode,
                                                                 @RequestParam(value = "orderCode") String orderCode);

    /**
     * 更新油卡用户券状态
     *
     * @param redeemCode 用户券兑换码
     * @param oldStatus  更新前状态
     * @param status     更新状态
     * @return
     */
    @GetMapping("/api/equity/oilCardUserCoupon/updateOilCardUserCoupon")
    Response updateOilCardUserCoupon(@RequestParam(value = "redeemCode") String redeemCode,
                                     @RequestParam(value = "oldStatus") String oldStatus,
                                     @RequestParam(value = "status") String status);

    /**
     * 油卡充值
     *
     * @param redeemCode  用户券兑换码
     * @param orderCode   订单编号
     * @param orderAmount 订单金额
     * @param openId      openId
     * @param oldStatus   更新前状态
     * @param status      状态
     * @return
     */
    @GetMapping("/api/equity/oilCardUserCoupon/oilCardUserCouponExchange")
    Response oilCardUserCouponExchange(@RequestParam(value = "redeemCode") String redeemCode,
                                       @RequestParam(value = "orderCode") String orderCode,
                                       @RequestParam(value = "orderAmount") BigDecimal orderAmount,
                                       @RequestParam(value = "openId") String openId,
                                       @RequestParam(value = "oldStatus") String oldStatus,
                                       @RequestParam(value = "status") String status);

    /**
     * 油卡批量充值
     *
     * @param exchangeDto 兑换信息
     * @return
     */
    @PostMapping("/api/equity/oilCardUserCoupon/oilCardUserCouponBatchExchange")
    Response oilCardUserCouponBatchExchange(@RequestBody OilCardUserCouponBatchExchangeDto exchangeDto);

    /**
     * 更新油卡用户券为已过期
     */
    @GetMapping("/api/equity/oilCardUserCoupon/updateOilCardUserCouponInvalidByInvalidDate")
    void updateOilCardUserCouponInvalidByInvalidDate();

    /**
     * 电子兑换券提现接口
     * @param openId
     * @param orderCode 提现订单号
     * @param amount    提现金额
     * @param pointType 积分类型
     * @return
     */
    @GetMapping("/api/equity/oilCardUserCoupon/oilCardUserCouponCashExtract")
    Response<List<EquityOilCardUserCoupon>> oilCardUserCouponCashExtract(@RequestParam(value = "openId") String openId,
                                                 @RequestParam(value = "orderCode") String orderCode,
                                                 @RequestParam(value = "amount") BigDecimal amount,
                                                 @RequestParam(value = "pointType") String pointType);

    /**
     * 积分退回
     * @param orderCode 订单编号
     * @param amount    订单金额
     * @return
     */
    @GetMapping("/api/equity/oilCardUserCoupon/oilCardUserCouponReturn")
    Response oilCardUserCouponReturn(@RequestParam(value = "orderCode") String orderCode,
                                            @RequestParam(value = "amount") BigDecimal amount);

    /**
     * 积分撤销（返利积分发放后撤销发放）
     * @param businessSerialNo  业务流水号
     * @param businessCode      业务单号
     * @param redeemCode        兑换码
     * @return
     */
    @GetMapping("/api/equity/oilCardUserCoupon/oilCardUserCouponCancelForRebate")
    Response oilCardUserCouponCancelForRebate(@RequestParam(value = "businessSerialNo") String businessSerialNo,
                                                     @RequestParam(value = "businessCode") String businessCode,
                                                     @RequestParam(value = "redeemCode") String redeemCode);

    /**
     * 根据用户券编号查询对应的电子加油券消耗记录 (分页)
     *
     * @param userCouponCode 用户券编号
     * @param limit          每页条数
     * @param page           当前页
     * @return
     */
    @GetMapping("/api/equity/oilCardUserCouponConsume/findOilCardUserCouponConsumeList")
    PageTable<EquityOilCardUserCouponConsume> findOilCardUserCouponConsumeList(@RequestParam(value = "userCouponCode") String userCouponCode,
                                                                               @RequestParam(defaultValue = "20", value = "limit") int limit,
                                                                               @RequestParam(defaultValue = "1", value = "page") int page);
    /*********************************************   电子加油券相关 end   *******************************************************************/

    /**
     * 根据充值额度统计可用优惠券张数
     * @param openId
     * @param amountLimit   额度
     */
    @GetMapping("/api/equity/userCouponOilCash/countCashCouponNumberByAmountLimit")
    Response countCashCouponNumberByAmountLimit(@RequestParam(value = "openId") String openId,
                                                       @RequestParam(value = "amountLimit") String amountLimit);

    /**
     * 查询代金券、电子兑换券
     * @param openId
     * @param limit
     * @param page
     * @return
     */
    @GetMapping("/api/equity/oilCashAndOilCardCoupon/findCashAndOilCardCoupon")
    PageTable<OilCashAndOilCardCouponDto> findCashAndOilCardCoupon(@RequestParam(value = "openId") String openId,
                                                                     @RequestParam(value = "limit", defaultValue = "20") int limit,
                                                                     @RequestParam(value = "page", defaultValue = "1") int page);

    /**
     * 提供订单功能根据用户券编号或兑换码查询用户券信息(服务券、代金券、电子兑换券)
     * @param userCouponCode          券编号
     * @param redeemCode    兑换码
     * @return
     */
    @GetMapping("/api/equity/oilCashAndOilCardCoupon/getUserCouponInfoForOrderByUserCouponCode")
    Response<PushOilUserCouponUseDetailDto> getUserCouponInfoForOrderByUserCouponCode(@RequestParam(value = "userCouponCode", required = false)String userCouponCode,
                                                                                             @RequestParam(value = "redeemCode", required = false)String redeemCode);



    /**
     * 得到所有的发放主体 huhl 数据权限管理时用
     * @return
     */
    @GetMapping("/api/equity/issuingBody/getIssuingBodyList")
    Response<List<IssuingBody>> getIssuingBodyList();


    /**
     * 查询代金券
     * @param status 状态
     * @param type   基准券类型
     * @param name 名称
     * @param code 编码
     * @param excludedCode 排除已选择的优惠券
     * @param limit
     * @param page
     * @return
     * @description yifenrui 2020-09-01 10:03:03
     * @description andy 2020-12-29 17:39:19 提供后台聚合礼包功能查询代金券增加type字段
     */
    @GetMapping("/api/equity/coupon/searchOilCashCoupon")
    PageTable<EquityCouponInfo> searchOilCashCoupon(@RequestParam(value = "status", defaultValue = "TO_LIST") String status,
                                                    @RequestParam(value = "type") String type,
                                                    @RequestParam(value = "name", required = false) String name,
                                                    @RequestParam(value = "code", required = false) String code,
                                                    @RequestParam(value = "excludedCode", required = false) String[] excludedCode,
                                                    @RequestParam(value = "limit", defaultValue = "20") int limit,
                                                    @RequestParam(value = "page", defaultValue = "1") int page);

    /**
     * 用户券转赠
     * @param openId           openId
     * @param redeemCodeList   兑换码列表
     * @param transferCode     转赠编号
     * @return
     */
    @GetMapping("/api/equity/userCoupon/userCouponTransfer")
    Response userCouponTransfer(@RequestParam(value = "openId") String openId,
                                @RequestParam(value = "redeemCodeList") List<String> redeemCodeList,
                                @RequestParam(value = "transferCode") String transferCode);

    /**
     * 用户券转赠撤销
     * @param openId            openId
     * @param redeemCodeList    兑换码列表
     * @param transferCode      转赠编码
     * @return
     */
    @GetMapping("/api/equity/userCoupon/userCouponTransferCancel")
    Response userCouponTransferCancel(@RequestParam(value = "openId") String openId,
                                      @RequestParam(value = "redeemCodeList") List<String> redeemCodeList,
                                      @RequestParam(value = "transferCode") String transferCode);

    /**
     * 用户券转赠领取
     * @param entity
     * @return
     */
    @PostMapping("/api/equity/userCoupon/userCouponTransferReceive")
    Response userCouponTransferReceive(@Validated @RequestBody UserCouponTransferReceiveDto entity);

    /**
     * 自动发券
     * @param entity
     * @return
     */
    @PostMapping("/api/equity/userCoupon/autoGrantUserCoupon")
    Response autoGrantUserCoupon(@Validated @RequestBody UserCouponAutoGrantDto entity);

    /**
     * 更新用户券第三方编号
     * @param userCouponDingjuDto   响应参数
     * @return
     */
    @PostMapping("/api/equity/userCoupon/updateUserCouponThirdCode")
    Response updateUserCouponThirdCode(@RequestBody UserCouponDingjuDto userCouponDingjuDto);

    /**
     * 根据用户券编号更新第三方编码
     * @param serialNo          请求码
     * @param thirdCode         第三方编号
     * @param thirdAmount       第三方金额
     * @return
     */
    @PostMapping("/api/equity/userCoupon/updateUserCouponThirdCodeByCode")
    Response updateUserCouponThirdCodeByCode(@RequestParam(value = "serialNo") String serialNo,
                                                    @RequestParam(value = "thirdCode") String thirdCode,
                                                    @RequestParam(value = "thirdAmount") String thirdAmount);

    /**
     * 积分领取
     * @param entity    对象
     * @return
     */
    @PostMapping("/api/equity/equityOilCardTransferReceiveRecord/saveTransferRecord")
    Response saveTransferRecord(@RequestBody EquityOilCardTransferReceiveRecord entity);

    /**
     * 根据手机号或openId、账户类型查询积分账户
     * @param mobile    手机号
     * @param openId    openId
     * @param type      账户类型
     * @return
     */
    @GetMapping("/api/equity/oilCardPointAccount/getPointAccountByOpenIdAndType")
    Response<EquityOilCardPointAccount> getPointAccountByMobileAndType(@RequestParam(value = "mobile", required = false) String mobile,
                                                                              @RequestParam(value = "openId", required = false) String openId,
                                                                              @RequestParam(value = "type") String type);


    /**
     * 根据openId、账户类型查询积分账户
     * @param openId    openId
     * @param type      账户类型
     * @return
     */
    @GetMapping("/api/equity/oilCardPointAccount/getUserEquity")
    Response<Map<String, BigDecimal>> getUserEquity(@RequestParam(value = "openId") String openId, @RequestParam(value = "type") String type);

    /**
     * 积分发放流水查询
     * @param type          发放类型
     * @param startDate     开始日期
     * @param endDate       结束日期
     * @param pointType     积分类型
     * @param openId        openId
     * @param limit         每页记录数
     * @param page          页码
     * @return
     */
    @GetMapping("/api/equity/oilCardPointFlow/searchPointFlow")
    PageTable<EquityOilCardPointFlow> searchPointFlow(@RequestParam(value = "type") String type,
                                                      @RequestParam(value = "startDate", required = false) String startDate,
                                                      @RequestParam(value = "endDate", required = false) String endDate,
                                                      @RequestParam(value = "pointType") String pointType,
                                                      @RequestParam(value = "openId") String openId,
                                                      @RequestParam(value = "limit", defaultValue = "20") int limit,
                                                      @RequestParam(value = "page", defaultValue = "1") int page);

    /**
     * 积分发放流水统计
     * @param type      发放类型
     * @param startDate 开始日期
     * @param endDate   结束日期
     * @param pointType 积分类型
     * @param openId openid
     * @return
     */
    @GetMapping("/api/equity/oilCardPointFlow/countPointFlow")
    Response<Map<String, BigDecimal>> countPointFlow(@RequestParam(value = "type") String type,
                                                            @RequestParam(value = "startDate", required = false) String startDate,
                                                            @RequestParam(value = "endDate", required = false) String endDate,
                                                            @RequestParam(value = "pointType") String pointType,
                                                            @RequestParam(value = "openId") String openId);

    /**
     * 统计邀请返利积分金额
     * @param pointType     积分类型
     * @param openId        openId
     * @return
     */
    @GetMapping("/api/equity/oilCardPointFlow/countInvitationRebate")
    Response<BigDecimal> countInvitationRebate(@RequestParam(value = "pointType") String pointType, @RequestParam(value = "openId") String openId);

    /*************************************************** 转赠-ranxiangfu start ************************************************************/
    /**
     * Page query equity bestow record page table.
     *
     * @param bestowRecordQueryDto bestow record query dto
     * @return the page table
     */
    @GetMapping("/api/equity/equity-bestow-record/pageQuery")
    PageTable<EquityBestowRecord> pageQueryEquityBestowRecord(@SpringQueryMap BestowRecordQueryDto bestowRecordQueryDto);

    /**
     * 导出权益转赠记录.
     * @param bestowRecordQueryDto bestow record query dto
     *
     * @return response
     */
    @GetMapping("/api/equity/equity-bestow-record/exportEquityBestowRecord")
    Response<List<EquityBestowRecord>> exportEquityBestowRecord(@SpringQueryMap BestowRecordQueryDto bestowRecordQueryDto);

    /**
     * Gets equity bestow record by id.
     *
     * @param id the id
     * @return the equity bestow record by id
     */
    @GetMapping("/api/equity/equity-bestow-record")
    Response<EquityBestowRecord> getEquityBestowRecordById(@RequestParam("id") Integer id);

    /**
     * Gets equity bestow record by code.
     *
     * @param code the code
     * @return the equity bestow record by code
     */
    @GetMapping("/api/equity/equity-bestow-record/getByCode")
    Response<EquityBestowRecord> getEquityBestowRecordByCode(@RequestParam("code") String code);

    /**
     * Gets equity bestow record with item by code.
     *
     * @param code the code
     * @return the equity bestow record with item by code
     */
    @GetMapping("/api/equity/equity-bestow-record/getWithItemByCode")
    Response<EquityBestowRecord> getEquityBestowRecordWithItemByCode(@RequestParam("code") String code);

    /**
     * Save equity bestow record response.
     *
     * @param equityBestowRecord the equity bestow record
     * @return the response
     */
    @PostMapping("/api/equity/equity-bestow-record")
    Response<String> saveEquityBestowRecord(@RequestBody EquityBestowRecord equityBestowRecord);

    /**
     * Change equity bestow record status response.
     *
     * @param code               the code
     * @param beforeStatus       the before status
     * @param afterStatus        the after status
     * @param updateUser         the update user
     * @param receiverUserMobile the receiver user mobile
     * @return the response
     */
    @GetMapping("/api/equity/equity-bestow-record/changeStatus")
    Response<EquityBestowRecord> changeEquityBestowRecordStatus(@RequestParam("code") String code,
                                                                @RequestParam("beforeStatus") String beforeStatus,
                                                                @RequestParam("afterStatus") String afterStatus,
                                                                @RequestParam("updateUser") String updateUser,
                                                                @RequestParam("receiverUserMobile") String receiverUserMobile);

    /**
     * List equity bestow record item by code response.
     *
     * @param code the code
     * @return the response
     */
    @GetMapping("/api/equity/equity-bestow-record-item/listByCode")
    Response<List<EquityBestowRecordItem>> listEquityBestowRecordItemByCode(@RequestParam("code") String code);


    /**
     * 接收人的分页查询.
     *
     * @param bestowRecordQueryDto the bestow record query dto
     * @return the page table
     */
    @GetMapping("/api/equity/equity-bestow-record/receiver/pageQuery")
    PageTable<EquityBestowRecord> receiverPageQueryEquityBestowRecordWithItem(@SpringQueryMap BestowRecordQueryDto bestowRecordQueryDto);

    /**
     * Count receive record item response.
     *
     * @param receiverMobile the receiver mobile
     * @param status         the status
     * @param miniAppid      the mini appid
     * @return the response
     */
    @GetMapping("/api/equity/equity-bestow-record/countItem")
    Response<Integer> countReceiveRecordItem(@RequestParam(value = "receiverMobile") String receiverMobile,
                                             @RequestParam(value = "status", required = false) String status,
                                             @RequestParam(value = "miniAppid") String miniAppid);

    /**
     * List un receive record with item response.
     *
     * @param receiverUserMobile the receiver user mobile
     * @param miniAppid          the mini appid
     * @return the response
     */
    @GetMapping("/api/equity/equity-bestow-record/listUnReceiveRecordWithItem")
    Response<List<EquityBestowRecord>> listUnReceiveRecordWithItem(@RequestParam(value = "receiverUserMobile") String receiverUserMobile,
                                                                   @RequestParam(value = "miniAppid") String miniAppid);
    /*************************************************** 转赠-ranxiangfu end ************************************************************/

    /**
     * 根据优惠券编号查询套餐
     * @param code 编号
     * @return
     */
    @GetMapping("/api/equity/couponGift/list")
    Response<List<EquityCouponGiftRel>> list(@RequestParam(value = "code") String code);

    /**
     * 统计用户券
     * @aram searchEntity   查询条件dto
     * @return
     */
    @RequestMapping("/api/equity/userCouponStatistics/searchUserCouponStatistics")
    PageTable<EquityStatisticsDto> searchUserCouponStatistics(@RequestBody EquityStatisticsParamDto searchEntity);

    /**
     * 代驾券使用统计
     * @aram searchEntity   查询条件dto
     * @return
     */
    @RequestMapping("/api/equity/userCouponStatistics/drivingServiceStatistics")
    PageTable<EquityDrivingServiceStatisticsDto> drivingServiceStatistics(@RequestBody EquityDrivingServiceStatisticsParamDto searchEntity);

    /**
     * 查询券列表
     * @param searchEntity
     * @return
     */
    @RequestMapping("/api/equity/userCouponStatistics/findDrivingServiceList")
    PageTable<EquityDrivingServicesListDto> findDrivingServiceList(@RequestBody EquityDrivingServiceStatisticsParamDto searchEntity);

    /**
     * 代驾券详情
     * @param code  券编码
     * @return
     */
    @RequestMapping("/api/equity/userCouponStatistics/drivingServiceDetail")
    Response<EquityDrivingServicesStatisticsDetailDto> drivingServiceDetail(@RequestParam(value = "code") String code);

    /**
     * 保存代驾券使用记录
     * @param designatedDriverRecord
     * @return
     */
    @RequestMapping("/api/equity/designatedDriver/saverUpdate")
    Response saveOrUpdateDesignatedDriverRecord(@RequestBody DesignatedDriverRecord designatedDriverRecord);


    /**
     * 新增主体获取平台专区关联信息
     * @param associatedDto
     * @return
     */
    @PostMapping("/api/equity/issuingBodyAssociated/saveAssociated")
    Response saveAssociated(@RequestBody EquityIssuingBodyAssociatedDto associatedDto);

    /**
     * 根据发放主体获取平台专区关联信息
     * @param bodyCode
     * @return
     */
    @GetMapping("/api/equity/issuingBodyAssociated/getByBodyCode/{bodyCode}")
    Response<List<EquityIssuingBodyAssociatedDto>> getEquityIssuingBodyAssociatedByBodyCode(@PathVariable("bodyCode") String bodyCode);

    /**
     * 查询
     * @return
     */
    @RequestMapping("/api/equity/sendMessageInfo/sendMessageStatisticSearch")
    PageTable<UserCouponSendMessageStatisticDto> sendMessageStatisticSearch(@RequestBody UserCouponSendMessageStatisticParamDto paramDto);

    /**
     * 查询详情
     * @return
     */
    @RequestMapping("/api/equity/sendMessageInfo/sendMessageStatisticDetail")
    Response<UserCouponSendMessageStatisticDto> sendMessageStatisticDetail(@RequestParam(value = "batchNo") String batchNo);


    /**
     * 发放记录清单记录查询
     * @param mobile        手机号
     * @param idCard        证件号码
     * @param licenseNo     车牌号
     * @param policyNo      保单号
     * @return
     */
    @RequestMapping("/api/equity/sendMessageRecord/sendMessageRecordSearch")
    PageTable<UserCouponSendMessageRecordDto> sendMessageRecordSearch(@RequestParam(value = "batchNo") String batchNo,
                                                                             @RequestParam(value = "mobile", required = false) String mobile,
                                                                             @RequestParam(value = "idCard", required = false) String idCard,
                                                                             @RequestParam(value = "licenseNo", required = false) String licenseNo,
                                                                             @RequestParam(value = "policyNo", required = false) String policyNo,
                                                                             @RequestParam(value = "page", defaultValue = "1") int page,
                                                                             @RequestParam(value = "limit", defaultValue = "20") int limit,
                                                                             @RequestParam(value = "sortField", defaultValue = "receiveTime") String sortField,
                                                                             @RequestParam(value = "sortType", defaultValue = "desc") String sortType);

    /**
     * 清单详情信息
     * @param code   清单编号
     * @return
     */
    @RequestMapping("/api/equity/batchDistributionCouponApplyDetail/detailInfo")
    Response<EquityBatchDistributionCouponApplyDetail> detailInfo(@RequestParam(value = "code") String code);

    /**
     * 查询清单用户短信发放记录
     * @param code   清单编号
     * @return
     */
    @RequestMapping("/api/equity/batchDistributionCouponApplyDetail/sendMessageRecordSearch")
    Response<UserCouponSendMessageRecordDetailDto> getSendMessageRecord(@RequestParam(value = "code") String code);

    /**
     * 清单短信发送
     * @param code   清单编号
     * @param sendType  发送类型：BATCH-批量通知，NORMAL-正常通知, OUT-外呼通知
     * @param operateUser 操作用户
     * @return
     */
    @RequestMapping("/api/equity/batchDistributionCouponApplyDetail/sendMessageByCode")
    Response sendMessageByCode(@RequestParam(value = "code") String code,
                                      @RequestParam(value = "sendType") String sendType,
                                      @RequestParam(value = "operateUser") String operateUser);
}