package com.rocogz.merchant.client;

import com.rocogz.merchant.constant.Constant;
import com.rocogz.merchant.dto.EvaluateDto;
import com.rocogz.merchant.dto.FilterAreaCountDto;
import com.rocogz.merchant.dto.TreeDto;
import com.rocogz.merchant.entity.areaservice.MerchantAreaService;
import com.rocogz.merchant.entity.areaservice.MerchantAreaServiceSettlement;
import com.rocogz.merchant.entity.brand.MerchantBrand;
import com.rocogz.merchant.entity.evaluate.MerchantEvaluateKpi;
import com.rocogz.merchant.entity.goods.MerchantGoods;
import com.rocogz.merchant.entity.goods.MerchantGoodsClassification;
import com.rocogz.merchant.entity.goods.MerchantGoodsLog;
import com.rocogz.merchant.entity.impressionLabel.MerchantAreaServiceImpressionLabel;
import com.rocogz.merchant.entity.industry.MerchantIndustry;
import com.rocogz.merchant.entity.label.MerchantLabel;
import com.rocogz.merchant.entity.service.ServiceClassification;
import com.rocogz.merchant.entity.service.ServiceClassificationIndustry;
import com.rocogz.merchant.entity.store.*;
import com.rocogz.merchant.entity.storeservice.*;
import com.rocogz.merchant.entity.supplier.MerchantSupplier;
import com.rocogz.merchant.entity.supplier.MerchantSupplierUlog;
import com.rocogz.merchant.entity.type.MerchantType;
import com.rocogz.syy.common.PageTable;
import com.rocogz.syy.common.response.Response;
import org.springframework.web.bind.annotation.*;

import java.util.List;

/**
 * 商户接口
 *
 * @author：liangyongtong
 * @since：2020/3/25
 */
public interface IMerchantClientService {

    /*********************************************  liangyongtong 商户相关 Start   ****************************************************************/
    /**
     * 更新保存商户
     *
     * @param store
     * @return
     */
    @PostMapping("/api/merchant/store")
    Response<String> saveMerchantStore(@RequestBody MerchantStore store);

    /**
     * 上架审批回调接口
     *
     * @param pass
     * @param code
     * @param updateUser
     * @return
     */
    @GetMapping("/api/merchant/store/storeApproveCallBlack")
    Response storeApproveCallBlack(@RequestParam(value = "pass") String pass, @RequestParam(value = "code") String code, @RequestParam(value = "updateUser") String updateUser);

    /**
     * 商户列表数据查询
     *
     * @param keyword        名称 模糊匹配
     * @param code           编码 全匹配
     * @param areaCode       专区编码 全匹配
     * @param industryCode   行业编码 全匹配
     * @param settlementCode 结算编码 全匹配
     * @param type           商户类型
     * @param status         商户状态
     * @param limit          每页条数
     * @param page           页码
     * @return
     */
    @GetMapping("/api/merchant/store/queryPage")
    PageTable<MerchantStore> merchantStorePageTable(@RequestParam(value = "keyword", required = false) String keyword,
                                                    @RequestParam(value = "code", required = false) String code,
                                                    @RequestParam(value = "areaCode", required = false) String areaCode,
                                                    @RequestParam(value = "industryCode", required = false) String industryCode,
                                                    @RequestParam(value = "settlementCode", required = false) String settlementCode,
                                                    @RequestParam(value = "type", required = false) String type,
                                                    @RequestParam(value = "status", required = false) String status,
                                                    @RequestParam(value = "limit", defaultValue = "20") int limit,
                                                    @RequestParam(value = "page", defaultValue = "1") int page);

    /**
     * 商户列表数据查询(主表)
     *
     * @param keyword        名称 模糊匹配
     * @param code           编码 全匹配
     * @param areaCode       专区编码 全匹配
     * @param industryCode   行业编码 全匹配
     * @param settlementCode 结算编码 全匹配
     * @param type           商户类型
     * @param status         商户状态
     * @param limit          每页条数
     * @param page           页码
     * @return
     */
    @GetMapping("/api/merchant/store/basicQueryPage")
    PageTable<MerchantStore> merchantStoreBasicPageTable(@RequestParam(value = "keyword", required = false) String keyword,
                                                         @RequestParam(value = "code", required = false) String code,
                                                         @RequestParam(value = "areaCode", required = false) String areaCode,
                                                         @RequestParam(value = "industryCode", required = false) String industryCode,
                                                         @RequestParam(value = "settlementCode", required = false) String settlementCode,
                                                         @RequestParam(value = "type", required = false) String type,
                                                         @RequestParam(value = "status", required = false) String[] status,
                                                         @RequestParam(value = "limit", defaultValue = "20") int limit,
                                                         @RequestParam(value = "page", defaultValue = "1") int page);

    /**
     * 操作商户状态
     *
     * @param code       商户编码
     * @param status     目标商户状态
     * @param updateUser 更新用户
     * @return
     */
    @GetMapping("/api/merchant/store/operateStatus")
    Response merchantStoreStatusOperate(@RequestParam(value = "code") String code, @RequestParam(value = "status") String status, @RequestParam(value = "updateUser") String updateUser);

    /**
     * 根据商户编码删除商户
     *
     * @param code 商户编码
     * @return
     */
    @GetMapping("/api/merchant/store/deleteStore")
    Response merchantStoreDeleteByCode(@RequestParam(value = "code") String code);

    /**
     * 根据商户编码获取商户详情
     *
     * @param code 商户编码
     * @return
     */
    @GetMapping("/api/merchant/store/storeDetail")
    Response<MerchantStore> merchantStoreDetailByCode(@RequestParam(value = "code") String code);

    /**
     * 更新保存账号信息
     *
     * @param account 账号对象
     * @return
     */
    @PostMapping("/api/merchant/store/account")
    Response merchantStoreAccountSave(@RequestBody MerchantStoreAccount account);

    /**
     * 商户账号列表查询
     *
     * @param storeCode 商户编码
     * @return
     */
    @GetMapping("/api/merchant/store/account/queryPage")
    PageTable<MerchantStoreAccount> merchantStoreAccountPageTable(@RequestParam(value = "storeCode") String storeCode,
                                                                  @RequestParam(value = "limit", defaultValue = "20") int limit,
                                                                  @RequestParam(value = "page", defaultValue = "1") int page);

    /**
     * 根据商户编码查询对应的账号列表
     *
     * @param storeCode 商户编码
     * @param status    账号状态
     * @return
     */
    @GetMapping("/api/merchant/store/account/queryList")
    Response<List<MerchantStoreAccount>> merchantStoreAccountListByStoreCode(@RequestParam(value = "storeCode") String storeCode,
                                                                             @RequestParam(value = "status", required = false) String status);

    /**
     * 更改商户账号状态
     *
     * @param code   商户账号编码
     * @param status 目标状态
     * @return
     */
    @GetMapping("/api/merchant/store/account/operateStatus")
    Response merchantStoreAccountStatusOperate(@RequestParam(value = "code") String code, @RequestParam(value = "status") String status, @RequestParam(value = "operateUser") String operateUser);

    /**
     * 重置商户账号密码
     *
     * @param code 待重置密码的商户账号编码
     * @return
     */
    @GetMapping("/api/merchant/store/account/resetpassword")
    Response merchantStoreAccountResetPassword(@RequestParam(value = "code") String code, @RequestParam(value = "newPass", required = false) String newPass, @RequestParam(value = "operateUser") String operateUser);

    /**
     * 根据手机号返回账号列表
     *
     * @param mobile 手机号
     * @return
     */
    @GetMapping("/api/merchant/store/account/listbymobile")
    Response<List<MerchantStoreAccount>> merchantStoreAccountListByMobile(@RequestParam(value = "mobile") String mobile);

    /**
     * 商户操作日志记录
     *
     * @param storeCode 商户编码
     * @param limit     每页条数
     * @param page      页码
     * @return
     */
    @GetMapping("/api/merchant/store/ulog/queryPage")
    PageTable<MerchantStoreUlog> merchantStoreUlogPageTable(@RequestParam(value = "storeCode") String storeCode,
                                                            @RequestParam(value = "limit", defaultValue = "20") int limit,
                                                            @RequestParam(value = "page", defaultValue = "1") int page);

    /**
     * 根据商户编码获取商户基本信息 (merchant_store 表数据)
     *
     * @param code 商户编码
     * @return
     */
    @GetMapping("/api/merchant/store/storeBaseDetail")
    Response<MerchantStore> merchantStoreBaseInfoByCode(@RequestParam(value = "code") String code);

    /**
     * 更新商户基本信息表数据
     *
     * @param store 商户信息
     * @return
     */
    @PostMapping("/api/merchant/store/baseinfo")
    Response merchantStoreBaseSave(@RequestBody MerchantStore store);

    /**
     * 查询绑定过该商户账号的微信列表
     *
     * @param accountCode 商户账号
     * @return
     */
    @GetMapping("/api/merchant/store/account/wxlist")
    Response<List<MerchantStoreAccountLoginStatus>> merchantStoreAccountWxLoginList(@RequestParam(value = "accountCode") String accountCode);

    /**
     * 删除某个商户账号下关联的微信登录信息
     *
     * @param accountCode 商户账号
     * @param wxOpenid    关联的微信openid
     * @param operateUser 操作用户
     * @return
     */
    @GetMapping("/api/merchant/store/account/deletewx")
    Response merchantStoreAccountDeleteWx(@RequestParam(value = "accountCode") String accountCode, @RequestParam(value = "wxOpenid") String wxOpenid, @RequestParam(value = "operateUser") String operateUser);

    /**
     * 通过商户账号编码获取账号详情
     *
     * @param code 账号编码
     * @return
     */
    @GetMapping("/api/merchant/store/account/detail")
    Response<MerchantStoreAccount> merchantStoreAccountDetailByCode(@RequestParam(value = "code") String code);

    /**
     * 按关联时间倒序返回登录关联列表
     *
     * @param openid openid
     * @return
     */
    @GetMapping("/api/merchant/store/account/statuslist")
    Response<List<MerchantStoreAccountLoginStatus>> merchantStoreAccountLoginStatusListByOpenid(@RequestParam(value = "openid") String openid);

    /**
     * 保存或者更新账号登录状态
     *
     * @param loginStatus 账号登录状态记录
     * @return
     */
    @PostMapping("/api/merchant/store/account/loginstatus/save")
    Response merchantStoreAccountLoginStatusSaveOrUpdate(@RequestBody MerchantStoreAccountLoginStatus loginStatus);

    /**
     * 返回限制规则列表
     *
     * @param areaCode        专区编码
     * @param areaServiceCode 专区服务编码
     * @return 限制列表
     */
    @GetMapping("/api/merchant/store/limitRules")
    Response<List<MerchantLimitRules>> merchantLimitRulesByAreaCodeAndAreaServiceCode(@RequestParam(value = "areaCode") String areaCode,
                                                                                      @RequestParam(value = "areaServiceCode") String areaServiceCode);

    /**
     * 运营管理-首页推荐商户-根据商户codelist查询商户列表---yifenrui
     *
     * @param listCode
     * @return
     */
    @GetMapping("/api/merchant/store/merchantStoreListByListCode")
    Response<List<MerchantStore>> merchantStoreListByListCode(@RequestParam(value = "code") List<String> listCode);

    /**
     * 添加商户和商户服务的订单数量
     *
     * @param storeCode        商户编码
     * @param storeServiceCode 商户服务编码
     * @param count            新添加的订单数量
     * @return 添加结果
     */
    @GetMapping("/api/merchant/store/appendOrderCount")
    Response appendStoreOrderCount(@RequestParam(value = "storeCode") String storeCode, @RequestParam(value = "storeServiceCode") String storeServiceCode, @RequestParam(value = "count") int count);


    /**
     * 获取加油站商户列表
     * @param lat 纬度
     * @param lng 经度
     * @param types 商户类型
     * @param brands 商户品牌
     * @return 商户列表
     */
    @GetMapping("/api/merchant/wx/store/gaslist")
    Response<List<MerchantStore>> getGasStationList(@RequestParam(value = "lat", required = false) double lat,
                                                   @RequestParam(value = "lng", required = false) double lng,
                                                   @RequestParam(value = "types", required = false) String[] types,
                                                   @RequestParam(value = "brands", required = false) String[] brands);


    /**
     * 重新刷新对应的编码进redis缓存
     * @return
     */
    @GetMapping("/api/merchant/wx/store/gaslist/refresh")
    Response refreshGasListToCache();

    /**
     * 测试事务
     *
     * @return
     */
    @GetMapping("/api/merchant/store/testtx")
    Response testTx();
    /*********************************************   liangyongtong 商户相关 end   ********************************************************************/

    /*********************************************  liangyongtong 商户类型相关 Start   ****************************************************************/
    /**
     * 分页查询商户类型.
     *
     * @param name   商户类型名称(模糊匹配)
     * @param code   商户类型编码(模糊匹配)
     * @param status 商户类型状态
     * @param page   the page
     * @param limit  the limit
     * @return the prefecture tree
     */
    @GetMapping("/api/merchant/type/pageQuery")
    PageTable<MerchantType> merchantTypePageQuery(@RequestParam(value = "name", required = false) String name,
                                                  @RequestParam(value = "code", required = false) String code,
                                                  @RequestParam(value = "status", required = false) String status,
                                                  @RequestParam(value = "page", defaultValue = "1") Integer page,
                                                  @RequestParam(value = "limit", defaultValue = "20") Integer limit);

    /**
     * 查询商户类型.
     *
     * @param name   名称(精准匹配)
     * @param code   编码(精准匹配)
     * @param industryCode   行业编码(精准匹配)
     * @param status the status
     * @return the response
     */
    @GetMapping("/api/merchant/type/list")
    Response<List<MerchantType>> listMerchantType(@RequestParam(value = "name", required = false) String name,
                                                  @RequestParam(value = "code", required = false) String code,
                                                  @RequestParam(value = "industryCode", required = false) String industryCode,
                                                  @RequestParam(value = "status", required = false, defaultValue = "OPEN") String status);

    /**
     * 根据id查询商户类型.
     *
     * @param id the id
     * @return the by id
     */
    @GetMapping("/api/merchant/type/{id}")
    Response<MerchantType> getMerchantTypeById(@PathVariable("id") Integer id);

    /**
     * 新增或者更新商户类型信息(对象的id属性为空时代表新增).
     *
     * @param merchantType 商户类型对象 {@link MerchantType}
     * @return the response
     */
    @PostMapping("/api/merchant/type")
    Response saveOrUpdateMerchantType(@RequestBody MerchantType merchantType);

    /*********************************************   liangyongtong商户类型相关 end   ********************************************************************/

    /*********************************************  yinshihao 商户品牌相关 Start   ****************************************************************/
    /**
     * 查询商户品牌列表(分页)
     *
     * @param code    品牌名
     * @param keyword 所属行业编码
     * @param status  品牌状态
     * @param industryCodeList 行业编码
     * @param page    页码
     * @param limit   每页条数
     * @return
     */
    @GetMapping("/api/merchant/brand")
    PageTable<MerchantBrand> findMerchantBrandList(@RequestParam(value = "code", required = false) String code,
                                                   @RequestParam(value = "keyword", required = false) String keyword,
                                                   @RequestParam(value = "status", required = false) String status,
                                                   @RequestParam(value = "industryCodeList",required = false) List<String> industryCodeList,
                                                   @RequestParam(value = "limit", defaultValue = "20") int limit,
                                                   @RequestParam(value = "page", defaultValue = "1") int page);

    /**
     * 新增或编辑商户品牌
     *
     * @param merchantBrand 商户品牌
     * @return
     */
    @PostMapping("/api/merchant/brand/saveOrUpdateMerchantBrand")
    Response saveOrUpdateMerchantBrand(@RequestBody MerchantBrand merchantBrand);

    /**
     * 添加商户品牌对应的行业
     *
     * @param merchantBrand 商户品牌
     * @return
     */
    @PostMapping("/api/merchant/brand/addBrandIndustry")
    Response addBrandIndustry(@RequestBody MerchantBrand merchantBrand);

    /**
     * 改变商户品牌状态
     *
     * @param merchantBrand 商户品牌
     * @return
     */
    @PostMapping("/api/merchant/brand/status")
    Response switchMerchantBrandStatus(@RequestBody MerchantBrand merchantBrand);

    /**
     * 根据id查询商户品牌信息
     *
     * @param id 商户品牌id
     * @return
     */
    @GetMapping("/api/merchant/brand/{id}")
    Response<MerchantBrand> getMerchantBrandById(@PathVariable("id") Integer id);

    /**
     * 根据id查询商户品牌全部信息(包括对应行业信息)
     *
     * @param id 品牌id
     * @return
     */
    @GetMapping("/api/merchant/brand/getAllInfoById/{id}")
    Response<MerchantBrand> getMerchantBrandAllInfoById(@PathVariable("id") Integer id);

    /**
     * 根据商户品牌编号删除专区信息
     *
     * @param name             品牌名称
     * @param brandCode        品牌编码
     * @param industryCodeList 专区编码
     * @return
     */
    @GetMapping("/api/merchant/brand/deleteIndustryByCode")
    Response deleteIndustryByCode(@RequestParam("name") String name, @RequestParam("brandCode") String brandCode, @RequestParam("industryCodeList") List<String> industryCodeList);

    /**
     * 获取启用的品牌列表
     * @param industryCode 行业编码
     * @return
     */
    @GetMapping("/api/merchant/brand/listwithopen")
    Response<List<MerchantBrand>> getMerchantBrandListWithOpen(@RequestParam(value = "industryCode", required = false) String industryCode);

    /*********************************************   yinshihao 商户品牌 end   ********************************************************************/

    /*********************************************  yinshihao 商户行业相关 Start   ****************************************************************/
    /**
     * 商户行业列表(分页)
     *
     * @param keyword 行业名或简称
     * @param code    行业编码
     * @param page    页码
     * @param limit   每页条数
     * @return
     */
    @GetMapping("/api/merchant/industry")
    PageTable<MerchantIndustry> findMerchantIndustryList(@RequestParam(value = "keyword", required = false) String keyword,
                                                         @RequestParam(value = "code", required = false) String code,
                                                         @RequestParam(value = "limit", defaultValue = "20") int limit,
                                                         @RequestParam(value = "page", defaultValue = "1") int page);

    /**
     * 新增或编辑商户行业
     *
     * @param merchantIndustry 商户行业
     * @return
     */
    @PostMapping("/api/merchant/industry/saveOrUpdateMerchantIndustry")
    Response saveOrUpdateMerchantIndustry(@RequestBody MerchantIndustry merchantIndustry);

    /**
     * 根据id查询商户行业信息
     *
     * @param id 商户行业id
     * @return
     */
    @GetMapping("/api/merchant/industry/{id}")
    Response<MerchantIndustry> getMerchantIndustryById(@PathVariable("id") Integer id);

    /**
     * 根据code查询行业信息
     *
     * @param code 行业code
     * @return
     */
    @GetMapping("/api/merchant/industry/getByCode/{code}")
    Response<MerchantIndustry> getMerchantIndustryByCode(@PathVariable("code") String code);

    /**
     * 根据商户品牌编号查询对应专区编码
     *
     * @param brandCode    品牌编码
     * @param industryCode 行业编码
     * @param keyword      行业名称或简称
     * @return
     */
    @GetMapping("/api/merchant/industry/getIndustryByBrandCode")
    Response<List<MerchantIndustry>> getIndustryByBrandCode(@RequestParam("brandCode") String brandCode,
                                                            @RequestParam(value = "industryCode", required = false) String industryCode,
                                                            @RequestParam(value = "keyword", required = false) String keyword);

    /**
     * 根据服务分类编码查询其所有所属行业.
     *
     * @param serviceClassificationCode 服务分类编码
     * @return the response
     */
    @GetMapping("/api/merchant/industry/listByServiceClassificationCode")
    Response<List<MerchantIndustry>> listByServiceClassificationCode(@RequestParam("serviceClassificationCode") String serviceClassificationCode);

    /**
     * 商户行业列表(服务分类页面使用，用于查询某服务分类未关联的行业)
     *
     * @param serviceClassificationCode 服务分类编码
     * @param industryCodeList          行业编码列表(not in)
     * @param keyword                   行业名或简称
     * @param code                      行业编码
     * @param page                      页码
     * @param limit                     每页条数
     * @return
     */
    @GetMapping("/api/merchant/industry/classificationSearchScrollPage")
    PageTable<MerchantIndustry> classificationSearchScrollPage(@RequestParam(value = "serviceClassificationCode", required = false) String serviceClassificationCode,
                                                               @RequestParam(value = "industryCodeList[]", required = false) List<String> industryCodeList,
                                                               @RequestParam(value = "keyword", required = false) String keyword,
                                                               @RequestParam(value = "code", required = false) String code,
                                                               @RequestParam(value = "limit", defaultValue = "20") int limit,
                                                               @RequestParam(value = "page", defaultValue = "1") int page);
    /*********************************************   yinshihao 商户行业相关 end   ********************************************************************/

    /*********************************************  yinshihao 商户标签相关 Start   ****************************************************************/

    /**
     * 商户标签列表
     *
     * @param name           标签名
     * @param prefectureCode 所属行业编码
     * @param status         标签状态
     * @param page           页码
     * @param limit          每页条数
     * @return
     */
    @GetMapping("/api/merchant/label")
    PageTable<MerchantLabel> findMerchantLabelList(@RequestParam(value = "name", required = false) String name,
                                                   @RequestParam(value = "prefectureCode", required = false) String prefectureCode,
                                                   @RequestParam(value = "status", required = false) String status,
                                                   @RequestParam(value = "limit", defaultValue = "20") int limit,
                                                   @RequestParam(value = "page", defaultValue = "1") int page);

    /**
     * 新增或编辑商户标签
     *
     * @param merchantLabel
     * @return
     */
    @PostMapping("/api/merchant/label/saveOrUpdateMerchantLabel")
    Response saveOrUpdateMerchantLabel(@RequestBody MerchantLabel merchantLabel);

    /**
     * 添加标签对应专区
     *
     * @param merchantLabel
     * @return
     */
    @PostMapping("/api/merchant/label/addLabelPrefecture")
    Response addLabelPrefecture(@RequestBody MerchantLabel merchantLabel);

    /**
     * 商户标签状态改变
     *
     * @param merchantLabel 商户标签
     * @return
     */
    @PostMapping("/api/merchant/label/status")
    Response switchMerchantLabelStatus(@RequestBody MerchantLabel merchantLabel);

    /**
     * 根据id查询商户标签信息
     *
     * @param id 商户标签id
     * @return
     */
    @GetMapping("/api/merchant/label/{id}")
    Response<MerchantLabel> getMerchantLabelById(@PathVariable("id") Integer id);

    /**
     * 根据商户标签编号查询对应专区编码
     *
     * @param labelCode      标签编码
     * @param prefectureCode 专区编码
     * @param prefectureName 专区名称
     * @return
     */
    @GetMapping("/api/merchant/label/getPrefectureByLabelCode")
    Response<List<String>> getPrefectureByLabelCode(@RequestParam("labelCode") String labelCode,
                                                    @RequestParam(value = "prefectureCode", required = false) String prefectureCode,
                                                    @RequestParam(value = "prefectureName", required = false) String prefectureName);

    /**
     * 根据code查询标签信息
     *
     * @param code 标签code
     * @return
     */
    @GetMapping("/api/merchant/label/getMerchantLabelByCode/{code}")
    Response<MerchantLabel> getMerchantLabelByCode(@PathVariable("code") String code);

    /**
     * 根据商户标签编号删除专区信息
     *
     * @param labelCode          标签编码
     * @param prefectureCodeList 专区编码
     * @return
     */
    @GetMapping("/api/merchant/label/deletePrefectureByCode")
    Response deletePrefectureByCode(@RequestParam("labelCode") String labelCode, @RequestParam("prefectureCodeList") List<String> prefectureCodeList);
    /*********************************************   yinshihao 商户标签相关 end   ********************************************************************/

    /*********************************************  ranxiangfu 服务分类和行业对应相关 Start   ****************************************************************/

    /**
     * 根据服务分类编码获取所属行业列表.
     *
     * @param serviceClassificationCode 服务分类编码
     * @return the response
     */
    @GetMapping("/api/merchant/service-classification-industry/listByServiceClassificationCode")
    Response<List<ServiceClassificationIndustry>> listServiceClassificationIndustryByCode(@RequestParam("serviceClassificationCode") String serviceClassificationCode);

    /**
     * 新增服务分类所属行业.
     *
     * @param serviceClassificationIndustry 服务分类所属行业对象 {@link com.rocogz.merchant.entity.service.ServiceClassificationIndustry}
     * @return the response
     */
    @PostMapping("/api/merchant/service-classification-industry")
    Response addServiceClassificationIndustry(@RequestBody ServiceClassificationIndustry serviceClassificationIndustry);

    /**
     * 批量新增服务分类所属行业.
     *
     * @param serviceClassificationIndustryList 服务分类所属行业对象列表 {@link com.rocogz.merchant.entity.service.ServiceClassificationIndustry}
     * @return the response
     */
    @PostMapping("/api/merchant/service-classification-industry/saveBatch")
    Response saveBatchServiceClassificationIndustry(@RequestBody List<ServiceClassificationIndustry> serviceClassificationIndustryList);

    /**
     * 根据id删除服务分类所属行业.
     *
     * @param id the id
     * @return the response
     */
    @DeleteMapping("/api/merchant/service-classification-industry")
    Response removeServiceClassificationIndustry(@RequestParam("id") Integer id);

    /**
     * 根据id删除服务分类所属行业.
     *
     * @param serviceClassificationCode 服务分类编码
     * @param industryCode              行业编码
     * @return the response
     */
    @DeleteMapping("/api/merchant/service-classification-industry/removeByCode")
    Response removeServiceClassificationIndustryByCode(@RequestParam("serviceClassificationCode") String serviceClassificationCode,
                                                       @RequestParam("industryCode") String industryCode);
    /*********************************************   ranxiangfu 服务分类和行业相关 end   ********************************************************************/

    /*********************************************  ranxiangfu 服务分类相关 Start   ****************************************************************/

    /**
     * 分页查询服务分类.
     *
     * @param page  the page
     * @param limit the limit
     * @return the prefecture tree
     */
    @GetMapping("/api/merchant/service-classification/pageQuery")
    PageTable<ServiceClassification> pageQueryServiceClassification(@RequestParam(value = "page", defaultValue = "1") Integer page,
                                                                    @RequestParam(value = "limit", defaultValue = "20") Integer limit);

    /**
     * 获取服务分类树.
     *
     * @return the prefecture tree
     */
    @GetMapping("/api/merchant/service-classification/getTree")
    Response<List<TreeDto<ServiceClassification>>> getServiceClassificationTree();

    /**
     * 根据id查询服务分类.
     *
     * @param id the id
     * @return the by id
     */
    @GetMapping("/api/merchant/service-classification/{id}")
    Response<ServiceClassification> getServiceClassificationById(@PathVariable("id") Integer id);

    /**
     * 根据编码查询服务分类.
     *
     * @param code 服务分类编码
     * @return the by id
     */
    @GetMapping("/api/merchant/service-classification/getByCode")
    Response<ServiceClassification> getServiceClassificationByCode(@RequestParam("code") String code);

    /**
     * 新增或者更新服务分类信息(对象的id属性为空时代表新增).
     *
     * @param serviceClassification 专区对象 {@link com.rocogz.merchant.entity.service.ServiceClassification}
     * @return the response
     */
    @PostMapping("/api/merchant/service-classification")
    Response saveOrUpdateServiceClassification(@RequestBody ServiceClassification serviceClassification);
    /*********************************************   ranxiangfu 服务分类相关 end   ********************************************************************/

    /*********************************************  liangyongtong 专区服务相关 Start   ****************************************************************/
    /**
     * 保存更新专区服务项目
     *
     * @param areaService 专区服务项目对象
     * @return
     */
    @PostMapping("/api/merchant/areaservice")
    Response merchantAreaServiceSave(@RequestBody MerchantAreaService areaService);

    /**
     * 专区服务项目列表查询
     *
     * @param keyword           名称
     * @param code              编码
     * @param cataCode          服务分类编码
     * @param areaCodes         专区编码列表
     * @param nature            范围
     * @param status            状态
     * @param noRepeatStoreCode 商户编码，添加该商户的服务项目的时候选择的专区服务项目非套餐的情况下不能重复
     * @param limit             每页条数
     * @param page              页码
     * @return
     */
    @GetMapping("/api/merchant/areaservice/queryPage")
    PageTable<MerchantAreaService> merchantAreaServicePageTable(@RequestParam(required = false, value = "keyword") String keyword,
                                                                @RequestParam(required = false, value = "code") String code,
                                                                @RequestParam(required = false, value = "cataCode") String cataCode,
                                                                @RequestParam(required = false, value = "areaCodes") String[] areaCodes,
                                                                @RequestParam(required = false, value = "nature") String nature,
                                                                @RequestParam(required = false, value = "status") String status,
                                                                @RequestParam(required = false, value = "noRepeatStoreCode") String noRepeatStoreCode,
                                                                @RequestParam(required = false, value = "limit", defaultValue = "20") int limit,
                                                                @RequestParam(required = false, value = "page", defaultValue = "1") int page);

    /**
     * 通过专区服务项目编码获取专区服务项目详情
     *
     * @param code 专区服务编码
     * @return
     */
    @GetMapping("/api/merchant/areaservice/detailbycode")
    Response<MerchantAreaService> merchantAreaServiceDetailByCode(@RequestParam(value = "code") String code);

    /**
     * 更新专区服务项目的状态
     *
     * @param code        专区服务项目编码
     * @param status      目标状态
     * @param operateUser 操作人
     * @return
     */
    @GetMapping("/api/merchant/areaservice/operate/status")
    Response merchantAreaServiceOperateStatus(@RequestParam(value = "code") String code, @RequestParam(value = "status") String status, @RequestParam(value = "operateUser") String operateUser);

    /**
     * 查询专区服务项目基本信息
     *
     * @param codes 专区服务项目编码列表 最多200个
     * @return 专区服务项目列表
     */
    @GetMapping("/api/merchant/areaservice/list/servicesByCodes")
    Response<List<MerchantAreaService>> merchantAreaServiceBaseInfoByCodes(@RequestParam(value = "codes") String[] codes);

    /**
     * 保存专区服务结算信息
     *
     * @param settlement 结算对象
     * @return
     */
    @PostMapping("/api/merchant/areaservice/settle/save")
    Response merchantAreaServiceSettleSave(@RequestBody MerchantAreaServiceSettlement settlement);

    /**
     * 获取结算信息详情
     *
     * @param settlementCode 结算信息编码
     * @return
     */
    @GetMapping("/api/merchant/areaservice/settle/detail")
    Response<MerchantAreaServiceSettlement> merchantAreaServiceSettlementDetailBySettlementCode(@RequestParam(value = "settlementCode") String settlementCode);

    /**
     * 获取专区服务项目对应的结算列表
     *
     * @param code 专区服务项目编码
     * @return
     */
    @GetMapping("/api/merchant/areaservice/settle/list")
    Response<List<MerchantAreaServiceSettlement>> merchantAreaServiceSettlementListByAreaServiceCode(@RequestParam(value = "code") String code);

    /**
     * 删除结算信息对象
     *
     * @param code 结算信息对象编码
     * @return
     */
    @GetMapping("/api/merchant/areaservice/settle/remove")
    Response merchentAreaServiceSettlementRemoveBySettlementCode(@RequestParam(value = "code") String code);

    /**
     * 通过专区项目编码查询可关联的专区和结算商的数据
     *
     * @param code       专区服务项目编码
     * @param areaCode   专区编码
     * @param settleCode 结算商编码
     * @return
     */
    @GetMapping("/api/merchant/areaservice/settle/pricelist")
    Response<MerchantAreaServiceSettlement> merchantAreaServiceSettlementByAreaCodeAndSettleCode(@RequestParam(value = "code") String code, @RequestParam(value = "areaCode") String areaCode, @RequestParam(value = "settleCode") String settleCode);

    /*********************************************   liangyongtong 专区服务相关 end   ********************************************************************/

    /*********************************************  liangyongtong 商户服务相关 Start   ****************************************************************/
    /**
     * 更新保存商户服务项目
     *
     * @param storeService 商户服务项目对象
     * @return
     */
    @PostMapping("/api/merchant/storeservice")
    Response merchantStoreServiceSave(@RequestBody MerchantStoreService storeService);

    /**
     * 商户服务项目列表查询 (关联商户表)
     *
     * @param keyword      名称
     * @param code         编码
     * @param excludedCode 不包含编码
     * @param cataCode     服务分类编码
     * @param areaCodes    专区编码列表
     * @param storeCodes   商户编码列表
     * @param status       状态
     * @param limit        每页条数
     * @param page         页码
     * @return
     */
    @GetMapping("/api/merchant/storeservice/queryPage")
    PageTable<MerchantStoreService> merchantStoreServicePageTable(@RequestParam(value = "keyword", required = false) String keyword,
                                                                  @RequestParam(value = "code", required = false) String code,
                                                                  @RequestParam(value = "excludedCode", required = false) String excludedCode,
                                                                  @RequestParam(value = "cataCode", required = false) String cataCode,
                                                                  @RequestParam(value = "areaCodes", required = false) String[] areaCodes,
                                                                  @RequestParam(value = "storeCodes", required = false) String[] storeCodes,
                                                                  @RequestParam(value = "status", required = false) String status,
                                                                  @RequestParam(value = "limit", required = false, defaultValue = "20") int limit,
                                                                  @RequestParam(value = "page", required = false, defaultValue = "1") int page);

    /**
     * 商户服务项目列表查询 (只查询商户服务项目表)
     *
     * @param keyword      名称
     * @param code         编码
     * @param excludedCode 不包含编码
     * @param cataCode     服务分类编码
     * @param storeCode    商户编码
     * @param status       状态
     * @param limit        每页条数
     * @param page         页码
     * @return
     */
    @GetMapping("/api/merchant/storeservice/base/queryPage")
    PageTable<MerchantStoreService> merchantStoreServiceBaseInfoPageTable(@RequestParam(value = "keyword", required = false) String keyword,
                                                                          @RequestParam(value = "code", required = false) String code,
                                                                          @RequestParam(value = "excludedCode", required = false) String excludedCode,
                                                                          @RequestParam(value = "cataCode", required = false) String cataCode,
                                                                          @RequestParam(value = "storeCode", required = false) String storeCode,
                                                                          @RequestParam(value = "status", required = false, defaultValue = Constant.DictData.MERCHANT_STORE_STATUS_UP_LIST) String status,
                                                                          @RequestParam(value = "limit", required = false, defaultValue = "20") int limit,
                                                                          @RequestParam(value = "page", required = false, defaultValue = "1") int page);

    /**
     * 批量操作状态上架下架
     *
     * @param codes       待操作的商户服务项目编码列表
     * @param status      目标状态
     * @param operateUser 操作人
     * @return
     */
    @GetMapping("/api/merchant/storeservice/batch/status")
    Response merchantStoreServiceBatchOperateStatus(@RequestParam(value = "codes") String[] codes, @RequestParam(value = "status") String status, @RequestParam(value = "operateUser") String operateUser);

    /**
     * 操作商户服务项目的状态
     *
     * @param code        待操作的商户服务项目编码
     * @param status      目标状态
     * @param operateUser 操作人
     * @return
     */
    @GetMapping("/api/merchant/storeservice/operate/status")
    Response merchantStoreServiceOperateStatus(@RequestParam(value = "code") String code, @RequestParam(value = "status") String status, @RequestParam(value = "operateUser") String operateUser);

    /**
     * 删除草稿状态的商户服务项目
     *
     * @param code 待删除的商户服务项目编码
     * @return
     */
    @GetMapping("/api/merchant/storeservice/remove")
    Response merchantStoreServiceRemoveByCode(@RequestParam(value = "code") String code);

    /**
     * 获取商户服务项目详情信息
     *
     * @param code 商户服务项目编码
     * @returnmerchantStoreServiceBaseInfoPageTable
     */
    @GetMapping("/api/merchant/storeservice/detail")
    Response<MerchantStoreService> merchantStoreServiceDetailByCode(@RequestParam(value = "code") String code);

    /**
     * 查询商户下的服务列表（不分页，不含结算信息)
     *
     * @param storeCode 商户编码
     * @param status    状态 （默认为上架，传 "")
     * @return 服务列表
     */
    @GetMapping("/api/merchant/storeservice/nopagelist")
    Response<List<MerchantStoreService>> merchantStoreServiceListByStoreCode(@RequestParam(value = "storeCode") String storeCode, @RequestParam(value = "status", required = false, defaultValue = Constant.DictData.MERCHANT_STORE_STATUS_UP_LIST) String status);

    /**
     * 查询商户下服务项目基本信息
     *
     * @param storeCode 商户编码
     * @param codes     服务项目编码列表
     * @return 服务项目列表
     */
    @GetMapping("/api/merchant/storeservice/store/services")
    Response<List<MerchantStoreService>> merchantStoreServiceBaseInfoByStoreCodeAndCodes(@RequestParam(value = "storeCode") String storeCode,
                                                                                         @RequestParam(value = "codes") String[] codes);

    /**
     * 返回该商户服务项目下对应车型的价格
     * @param storeServiceCode 商户服务项目编码
     * @param carType 车型
     * @return 价格对象
     */
    @GetMapping("/api/merchant/storeservice/store/service/price")
    Response<MerchantStoreServicePrice> merchantStoreServicePriceByStoreServiceCodeAndCarType(@RequestParam(value = "storeServiceCode") String storeServiceCode, @RequestParam(value = "carType") String carType);

    /**
     * 查询服务项目基本信息
     *
     * @param codes 服务项目编码列表 最多200个
     * @return 服务项目列表
     */
    @GetMapping("/api/merchant/storeservice/store/servicesByCodes")
    Response<List<MerchantStoreService>> merchantStoreServiceBaseInfoByCodes(@RequestParam(value = "codes") String[] codes);

    /**
     * 对商户服务进行排序
     *
     * @param code   商户服务项目编码
     * @param newSeq 新的排序号
     * @return
     */
    @GetMapping("/api/merchant/storeservice/seq")
    Response merchantStoreServviceSeq(@RequestParam(value = "code") String code, @RequestParam(value = "newSeq") int newSeq);

    /**
     * 查询服务项目操作日志列表
     *
     * @param storeServiceCode 商户服务项目编码
     * @param limit            每页数量
     * @param page             当前页
     * @return 列表
     */
    @GetMapping("/api/merchant/storeservice/ulog/queryPage")
    PageTable<MerchantStoreServiceUlog> merchantStoreServiceUlogPageTable(@RequestParam(value = "storeServiceCode") String storeServiceCode,
                                                                          @RequestParam(required = false, defaultValue = "20", value = "limit") int limit,
                                                                          @RequestParam(required = false, defaultValue = "1", value = "page") int page);

    /**
     * 新增或保存预约数据
     *
     * @param storeService
     * @return
     */
    @PostMapping("/api/merchant/storeservice/appointment/saveorupdate")
    Response merchantStoreServiceAppointmentSaveOrUpdate(@RequestBody MerchantStoreService storeService);

    /**
     * 通过商户服务项目编码获取对应的预约数据
     *
     * @param storeServiceCode 商户服务项目编码
     * @return
     */
    @GetMapping("/api/merchant/storeservice/appointment/listByStoreServiceCode")
    Response<List<MerchantStoreServiceAppointment>> merchantStoreServiceAppointmentByStoreServiceCode(@RequestParam(value = "storeServiceCode") String storeServiceCode);

    /**
     * 通过商户编码获取该商户下第一个开启了预约设置的商户服务项目对应的预约数据
     *
     * @param storeCode 商户编码
     * @return
     */
    @GetMapping("/api/merchant/storeservice/appointment/listByStoreCode")
    Response<List<MerchantStoreServiceAppointment>> merchantStoreServiceAppointmentByStoreCode(@RequestParam(value = "storeCode") String storeCode);

    /**
     * 通过商户编码获取该商户下第一个开启了预约设置的商户服务项目
     *
     * @param storeCode 商户编码
     * @return
     */
    @GetMapping("/api/merchant/storeservice/appointment/serviceItemByStoreCode")
    Response<MerchantStoreService> merchantStoreServiceAppointmentItemByStoreCode(@RequestParam(value = "storeCode") String storeCode);

    /*********************************************   liangyongtong 商户服务相关 end   ********************************************************************/

    /*********************************************   liangyongtong 商户服务标签 start   ********************************************************************/
    /**
     * 新增或更新商户服务标签
     *
     * @param label
     * @return
     */
    @PostMapping("/api/merchant/storeservice/label/saveOrUpdate")
    Response merchantStoreServiceLabelSaveOrUpdate(@RequestBody MerchantStoreServiceLabel label);

    /**
     * 查询商户服务项目标签列表
     *
     * @param keyword 标签名称关键字
     * @param status  状态
     * @param limit   每页长度
     * @param page    当前页
     * @return
     */
    @GetMapping("/api/merchant/storeservice/label/queryPage")
    PageTable<MerchantStoreServiceLabel> merchantStoreServiceLabelPageTable(@RequestParam(value = "keyword", required = false) String keyword,
                                                                            @RequestParam(value = "status", required = false) String status,
                                                                            @RequestParam(value = "limit", required = false, defaultValue = "20") int limit,
                                                                            @RequestParam(value = "page", required = false, defaultValue = "1") int page);

    /**
     * 变更标签的状态
     *
     * @param id          标签id
     * @param status      目标状态
     * @param operateUser 操作人
     * @return
     */
    @GetMapping("/api/merchant/storeservice/label/operateStatus")
    Response merchantStoreServiceLabelOperateStatus(@RequestParam(value = "id") Integer id, @RequestParam(value = "status") String status, @RequestParam(value = "operateUser") String operateUser);
    /*********************************************   liangyongtong 商户服务标签 end   ********************************************************************/

    /*********************************************   liangyongtong 评价结果保存 start   ********************************************************************/
    /**
     * 更新商户、商户服务项目平均分，记录印象标签统计数量
     *
     * @param evaluate 评价结果
     * @return
     */
    @PostMapping("/api/merchant/evaluate/kpi/all")
    Response evaluateKpiAll(@RequestBody EvaluateDto evaluate);

    /**
     * 更新商户、商户服务项目订单数
     *
     * @param evaluate 记录结果
     * @return
     */
    @PostMapping("/api/merchant/evaluate/kpi/store")
    Response evaluateKpiForStore(@RequestBody EvaluateDto evaluate);

    /**
     * 获取商户服务项目的印象标签
     *
     * @param storeServiceCode 商户服务项目编码
     * @return 针对该商户服务项目的印象标签统计列表
     */
    @GetMapping("/api/merchant/evaluate/kpi/storeservice/impression")
    Response<List<MerchantEvaluateKpi>> merchantEvaluateKpiWithStoreService(@RequestParam(value = "storeServiceCode") String storeServiceCode);

    /**
     * 获取商户的印象标签
     *
     * @param storeCode 商户编码
     * @return 针对该商户的印象标签统计列表
     */
    @GetMapping("/api/merchant/evaluate/kpi/store/impression")
    Response<List<MerchantEvaluateKpi>> merchantEvaluateKpiWithStore(@RequestParam(value = "storeCode") String storeCode);

    /**
     * 设置评价印象标签数量
     *
     * @param storeServiceCode 商户服务项目编码
     * @param type             com.rocogz.merchant.constant.Constant.General#MERCHANTEVALUATEKPI_ADD,
     *                         com.rocogz.merchant.constant.Constant.General#MERCHANTEVALUATEKPI_CANCEL,
     *                         com.rocogz.merchant.constant.Constant.General#MERCHANTEVALUATEKPI_DELETE,
     *                         *             com.rocogz.merchant.constant.Constant.General#MERCHANTEVALUATEKPI_HIDE,
     *                         *             com.rocogz.merchant.constant.Constant.General#MERCHANTEVALUATEKPI_SHOW
     * @param impressinos      印象标签数组
     * @param perfect          是否精选
     * @return
     */
    @GetMapping("/api/merchant/evaluate/kpi/storeservice/kpioperateamount")
    Response merchantEvaluateKpiOperateAmount(@RequestParam(value = "storeServiceCode") String storeServiceCode, @RequestParam(value = "type") String type, @RequestParam(value = "impressinos") String[] impressinos, @RequestParam(value = "perfect", required = false) Boolean perfect);
    /*********************************************   liangyongtong 评价结果保存 end   ********************************************************************/

    /*********************************************  liangyongtong 车主商户端商户相关 Start   ****************************************************************/
    /**
     * 查询商户列表数据（含基本表数据，停业信息，标签）【每次最多1000个】
     *
     * @param codes  商户编码列表
     * @param status 目标查询状态
     * @return
     */
    @GetMapping("/api/merchant/wx/store/list/bycodes")
    Response<List<MerchantStore>> getMerchantStoreListByCodes(@RequestParam(value = "codes") String[] codes, @RequestParam(value = "status", required = false) String status);

    /**
     * 查询商户列表数据（主要针对会员端商户列表）
     *
     * @param areaCode          专区编码
     * @param keyword           关键字
     * @param provinceCode      省编码
     * @param cityCode          市编码
     * @param districtCode      区编码
     * @param streetCode        街道编码
     * @param industryCode      行业编码
     * @param serviceType       服务类型
     * @param storeTypes        商户类型
     * @param orderType         排序类型
     * @param lat               纬度
     * @param lng               经度
     * @param areaServiceCode   专区服务项目编码
     * @param includeStoreCodes 包含的商户编码列表
     * @param excludeStoreCodes 不包含的商户编码列表
     * @param limit             每页数量
     * @param page              页码
     * @return
     */
    @GetMapping("/api/merchant/wx/store/list")
    PageTable<MerchantStore> getMerchantStoreCarList(@RequestParam(value = "areaCode", required = false) String areaCode,
                                                     @RequestParam(value = "keyword", required = false) String keyword,
                                                     @RequestParam(value = "provinceCode", required = false) String provinceCode,
                                                     @RequestParam(value = "cityCode", required = false) String cityCode,
                                                     @RequestParam(value = "districtCode", required = false) String districtCode,
                                                     @RequestParam(value = "streetCode", required = false) String streetCode,
                                                     @RequestParam(value = "industryCode", required = false) String industryCode,
                                                     @RequestParam(value = "serviceType", required = false) String serviceType,
                                                     @RequestParam(value = "storeTypes", required = false) String[] storeTypes,
                                                     @RequestParam(value = "orderType", required = false) String orderType,
                                                     @RequestParam(value = "lat", required = false, defaultValue = "0") double lat,
                                                     @RequestParam(value = "lng", required = false, defaultValue = "0") double lng,
                                                     @RequestParam(value = "areaServiceCode", required = false) String areaServiceCode,
                                                     @RequestParam(value = "includeStoreCodes", required = false) String[] includeStoreCodes,
                                                     @RequestParam(value = "excludeStoreCodes", required = false) String[] excludeStoreCodes,
                                                     @RequestParam(value = "limit", defaultValue = "20") int limit,
                                                     @RequestParam(value = "page", defaultValue = "1") int page
    );

    /**
     * 查询商户列表数据（主要针对会员端地图模式商户列表）
     *
     * @param areaCode          专区编码
     * @param keyword           关键字
     * @param provinceCode      省编码
     * @param cityCode          市编码
     * @param districtCode      区编码
     * @param streetCode        街道编码
     * @param industryCode      行业编码
     * @param serviceType       服务类型
     * @param storeTypes        商户类型
     * @param lat               纬度
     * @param lng               经度
     * @param areaServiceCode   专区服务项目编码
     * @param includeStoreCodes 包含的商户编码列表
     * @param excludeStoreCodes 不包含的商户编码列表
     * @return
     */
    @GetMapping("/api/merchant/wx/store/nearlist")
    Response<List<MerchantStore>> getMerchantNearStoreCarList(@RequestParam(value = "areaCode", required = false) String areaCode,
                                                              @RequestParam(value = "keyword", required = false) String keyword,
                                                              @RequestParam(value = "provinceCode", required = false) String provinceCode,
                                                              @RequestParam(value = "cityCode", required = false) String cityCode,
                                                              @RequestParam(value = "districtCode", required = false) String districtCode,
                                                              @RequestParam(value = "streetCode", required = false) String streetCode,
                                                              @RequestParam(value = "industryCode", required = false) String industryCode,
                                                              @RequestParam(value = "serviceType", required = false) String serviceType,
                                                              @RequestParam(value = "storeTypes", required = false) String[] storeTypes,
                                                              @RequestParam(value = "lat", required = false, defaultValue = "0") double lat,
                                                              @RequestParam(value = "lng", required = false, defaultValue = "0") double lng,
                                                              @RequestParam(value = "areaServiceCode", required = false) String areaServiceCode,
                                                              @RequestParam(value = "includeStoreCodes", required = false) String[] includeStoreCodes,
                                                              @RequestParam(value = "excludeStoreCodes", required = false) String[] excludeStoreCodes
    );

    /**
     * 查询专区下各街道的上架商户数
     *
     * @param areaServiceCode   专区服务项目编码
     * @param includeStoreCodes 包含的商户编码列表
     * @param excludeStoreCodes 不包含的商户编码列表
     * @param areaCode          专区编码
     * @return
     */
    @GetMapping("/api/merchant/wx/store/areaStoreCount")
    Response<List<FilterAreaCountDto>> getAreaStoreCount(@RequestParam(value = "areaServiceCode", required = false) String areaServiceCode,
                                                         @RequestParam(value = "includeStoreCodes", required = false) String[] includeStoreCodes,
                                                         @RequestParam(value = "excludeStoreCodes", required = false) String[] excludeStoreCodes,
                                                         @RequestParam(value = "areaCode") String areaCode);

    /**
     * 查询用户券的可用商户数量
     *
     * @param areaServiceCode   专区服务项目编码
     * @param includeStoreCodes 包含的商户编码列表
     * @param excludeStoreCodes 不包含的商户编码列表
     * @param areaCode          专区编码
     * @return 可用商户数
     */
    @GetMapping("/api/merchant/wx/store/usableStoreCount")
    Response<Integer> getUsableStoreCount(@RequestParam(value = "areaServiceCode") String areaServiceCode,
                                          @RequestParam(value = "includeStoreCodes", required = false) String[] includeStoreCodes,
                                          @RequestParam(value = "excludeStoreCodes", required = false) String[] excludeStoreCodes,
                                          @RequestParam(value = "areaCode") String areaCode);

    /**
     * 查询商户下某专区服务项目列表信息
     *
     * @param storeCode       商户编码
     * @param areaServiceCode 专区服务项目编码
     * @param status          服务状态
     * @return 商户服务项目列表
     */
    @GetMapping("/api/merchant/storeservice/store/servicesbyareaservicecode")
    Response<List<MerchantStoreService>> merchantStoreServiceByStoreCodeAndAreaServiceCode(@RequestParam(value = "storeCode") String storeCode,
                                                                                           @RequestParam(value = "areaServiceCode") String areaServiceCode,
                                                                                           @RequestParam(value = "status", required = false) String status);

    /**
     * 通过专区服务项目编码获取对应的评价指标数据
     *
     * @param areaServiceCode 专区服务项目编码
     * @return 只含两个指标维度数据的专区服务项目对象
     */
    @GetMapping("/api/merchant/wx/areaservice/kpi")
    Response<MerchantAreaService> merchantAreaServiceEvaluativeDimension(@RequestParam(value = "areaServiceCode") String areaServiceCode);

    /*********************************************   yinshihao MerchantAreaServiceImpressionLabel start   ********************************************************************/
    /**
     * 新增或更新商户服务标签
     *
     * @param label
     * @return
     */
    @PostMapping("/api/merchant/service/impression/label/saveOrUpdate")
    Response merchantAreaServiceImpressionLabelSaveOrUpdate(@RequestBody MerchantAreaServiceImpressionLabel label);

    /**
     * 查询商户服务项目标签列表
     *
     * @param keyword 标签名称关键字
     * @param status  状态
     * @param limit   每页长度
     * @param page    当前页
     * @return
     */
    @GetMapping("/api/merchant/service/impression/label/queryPage")
    PageTable<MerchantAreaServiceImpressionLabel> merchantAreaServiceImpressionLabelPageTable(@RequestParam(value = "keyword", required = false) String keyword,
                                                                                              @RequestParam(value = "status", required = false) String status,
                                                                                              @RequestParam(value = "limit", required = false, defaultValue = "20") int limit,
                                                                                              @RequestParam(value = "page", required = false, defaultValue = "1") int page);

    /**
     * 变更标签的状态
     *
     * @param id          标签id
     * @param status      目标状态
     * @param operateUser 操作人
     * @return
     */
    @GetMapping("/api/merchant/service/impression/label/operateStatus")
    Response merchantAreaServiceImpressionLabelOperateStatus(@RequestParam(value = "id") Integer id, @RequestParam(value = "status") String status, @RequestParam(value = "operateUser") String operateUser);

    /**
     * 删除标签
     *
     * @param id 标签id
     * @return
     */
    @GetMapping("/api/merchant/service/impression/label/deleteById")
    Response deleteMerchantAreaServiceImpressionLabeById(@RequestParam("id") Integer id);
    /*********************************************   yinshihao MerchantAreaServiceImpressionLabel end   ********************************************************************/

    /*********************************************   yinshihao MerchantGoods start   ********************************************************************/

    /**
     * 更新保存商品
     *
     * @param goods
     * @return
     */
    @PostMapping("/api/merchant/goods/saveMerchantGoods")
    Response<String> saveMerchantGoods(@RequestBody MerchantGoods goods);

    /**
     * 商品列表数据查询
     *
     * @param keyword                 名称 模糊匹配
     * @param code                    编码 全匹配
     * @param brandCode               品牌编码 全匹配
     * @param supplierCode            供应商编码 全匹配
     * @param goodsClassificationCode 商品分类编码 全匹配
     * @param status                  商品状态
     * @param invented                是否虚拟商品
     * @param limit                   每页条数
     * @param page                    页码
     * @return
     */
    @GetMapping("/api/merchant/goods/queryPage")
    PageTable<MerchantGoods> merchantGoodsPageTable(@RequestParam(value = "keyword", required = false) String keyword,
                                                    @RequestParam(value = "code", required = false) String code,
                                                    @RequestParam(value = "brandCode", required = false) String brandCode,
                                                    @RequestParam(value = "supplierCode", required = false) String supplierCode,
                                                    @RequestParam(value = "goodsClassificationCode", required = false) String goodsClassificationCode,
                                                    @RequestParam(value = "status", required = false) String status,
                                                    @RequestParam(value = "invented", required = false) String invented,
                                                    @RequestParam(value = "limit", defaultValue = "20") int limit,
                                                    @RequestParam(value = "page", defaultValue = "1") int page);

    /**
     * 操作商品状态
     *
     * @param code       商品编码
     * @param status     目标商品状态
     * @param updateUser 更新用户
     * @return
     */
    @GetMapping("/api/merchant/goods/operateStatus")
    Response merchantGoodsStatusOperate(@RequestParam(value = "code") String code, @RequestParam(value = "status") String status, @RequestParam(value = "updateUser") String updateUser);

    /**
     * 审批回调更改状态（上架审批后回调）
     *
     * @param pass
     * @param code
     * @param updateUser
     * @return
     */
    @GetMapping("/api/merchant/goods/goodsApproveCallBlack")
    Response goodsApproveCallBlack(@RequestParam(value = "pass") String pass, @RequestParam(value = "code") String code, @RequestParam(value = "updateUser") String updateUser);

    /**
     * 根据商品编码删除商品
     *
     * @param code 商品编码
     * @return
     */
    @GetMapping("/api/merchant/goods/deleteGoods")
    Response merchantGoodsDeleteByCode(@RequestParam(value = "code") String code);

    /**
     * 根据商品编码获取商品详情
     *
     * @param code 商品编码
     * @return
     */
    @GetMapping("/api/merchant/goods/goodsDetail")
    Response<MerchantGoods> merchantGoodsDetailByCode(@RequestParam(value = "code") String code);

    /**
     * 批量操作状态上架下架
     *
     * @param codes       待操作的商户服务项目编码列表
     * @param status      目标状态
     * @param operateUser 操作人
     * @return
     */
    @GetMapping("/api/merchant/goods/batch/status")
    Response merchantGoodsBatchOperateStatus(@RequestParam(value = "codes") String[] codes, @RequestParam(value = "status") String status, @RequestParam(value = "operateUser") String operateUser);

    /**
     * 根据商品分类编码获取商品列表(基本信息)
     *
     * @param goodsClassificationCode 商品分类编码
     * @return
     */
    @GetMapping("/api/merchant/goods/findGoodsListByClassificationCode")
    Response<List<MerchantGoods>> findGoodsListByClassificationCode(@RequestParam("goodsClassificationCode") String goodsClassificationCode);
    /*********************************************   yinshihao MerchantGoods end   ********************************************************************/

    /*********************************************   yinshihao MerchantGoodsLog start   ********************************************************************/

    /**
     * 商品操作日期记录
     *
     * @param goodsCode 商品编码
     * @param limit     每页条数
     * @param page      页码
     * @return
     */
    @GetMapping("/api/merchant/goodsLog/queryPage")
    PageTable<MerchantGoodsLog> merchantGoodsLogPageTable(@RequestParam(value = "goodsCode") String goodsCode,
                                                          @RequestParam(value = "limit", defaultValue = "20") int limit,
                                                          @RequestParam(value = "page", defaultValue = "1") int page);

    /*********************************************   yinshihao MerchantGoodsLog end   ********************************************************************/



    /********************************************* yifenrui 供应商  start******************************************************/

    /**
     * 供应商列表
     * @param code
     * @param name
     * @param status
     * @param limit
     * @param page
     * @return
     */
    @GetMapping("/api/merchant/supplier")
    PageTable<MerchantSupplier> searchSupplier(@RequestParam(name = "code",required = false) String code,
                                               @RequestParam(name = "name",required = false) String name,
                                               @RequestParam(name = "status",required = false) String status,
                                               @RequestParam(name = "limit",defaultValue = "20") Integer limit,
                                               @RequestParam(name = "page",defaultValue = "1") Integer page);


    /**
     *
     * 新增和更新供应商
     * @param merchantSupplier
     * @return
     */
    @PostMapping("/api/merchant/supplier/createOrUpdateMerchantSupplier")
    Response createOrUpdateMerchantSupplier(@RequestBody MerchantSupplier merchantSupplier);


    /**
     * 根据供应商编码获取供应商详情
     *
     * @param code 供应商编码
     * @return
     */
    @GetMapping("/api/merchant/supplier/supplierDetail")
    Response<MerchantSupplier> merchantMerchantSupplierDetailByCode(@RequestParam(value = "code") String code);


    /**
     * 操作供应商状态
     *
     * @param code       供应商编码
     * @param status     目标供应商状态
     * @param updateUser 更新用户
     * @return
     */
    @GetMapping("/api/merchant/supplier/operateStatus")
    Response merchantSupplierStatusOperate(@RequestParam(value = "code") String code, @RequestParam(value = "status") String status, @RequestParam(value = "updateUser") String updateUser, @RequestParam(value = "reason",required = false) String reason);

    /**
     * 供应商操作日志记录
     *
     * @param supplierCode 商户编码
     * @param limit     每页条数
     * @param page      页码
     * @return
     */
    @GetMapping("/api/merchant/supplier/ulog/queryPage")
    PageTable<MerchantSupplierUlog> merchantSupplierUlogPageTable(@RequestParam(value = "supplierCode") String supplierCode,
                                                               @RequestParam(value = "limit", defaultValue = "20") int limit,
                                                               @RequestParam(value = "page", defaultValue = "1") int page);
    /********************************************* yifenrui 供应商  end******************************************************/


    /********************************************* yifenrui 商品分类  start******************************************************/
    /**
     * 获取商品分类树.
     *
     * @return
     */
    @GetMapping("/api/merchant/goods/classification/getTree")
    Response<List<TreeDto<MerchantGoodsClassification>>> getMerchantGoodsClassificationTree();

    /**
     * 根据id查询商品分类.
     *
     * @param id the id
     * @return the by id
     */
    @GetMapping("/api/merchant/goods/classification/{id}")
    Response<MerchantGoodsClassification> getMerchantGoodsClassificationById(@PathVariable("id") Integer id);

    /**
     * 根据商品分类编码获取下级列表.
     *
     * @param goodsClassificationCode 服务分类编码
     * @return the response
     */
    @GetMapping("/api/merchant/goods/classification/listByGoodsClassificationCode")
    Response<List<MerchantGoodsClassification>> listByGoodsClassificationCode(@RequestParam("goodsClassificationCode") String goodsClassificationCode);


    @GetMapping("/api/merchant/goods/classification/deleteByGoodsClassCode")
    Response deleteByGoodsClassCode(@RequestParam("goodsClassificationCode") String goodsClassificationCode);

    /**
     * 分页查询商品分类列表
     * @param code
     * @param name
     * @param status
     * @param limit
     * @param page
     * @return
     */
    @GetMapping("/api/merchant/goods/classification")
    PageTable<MerchantGoodsClassification> searchMerchantGoodsClassification(@RequestParam(name = "code",required = false) String code,
                                               @RequestParam(name = "name",required = false) String name,
                                               @RequestParam(name = "status",required = false) String status,
                                               @RequestParam(name = "limit",defaultValue = "20") Integer limit,
                                               @RequestParam(name = "page",defaultValue = "1") Integer page);


    @PostMapping("/api/merchant/goods/classification/saveOrUpdate")
    Response saveOrUpdateMerchantGoodsClassification(@RequestBody MerchantGoodsClassification merchantGoodsClassification);


    /********************************************* yifenrui 商品分类  end******************************************************/


}

