package com.rocogz.merchant.comutil;

import com.rocogz.merchant.constant.Constant;
import org.springframework.cglib.core.Local;
import org.springframework.lang.NonNull;

import java.time.LocalDate;

/**
 * 计算有效期 工具类
 *
 * @author zhangmin
 */
public final class ValidityUtil {


    private ValidityUtil() {
    }


    /**
     * 1次 获取有效开始-结束日期， 注意：如果是延迟生效,生效开始日期=now(),生效结束日期=now()+delayDays
     * {@link Constant.DictValidityGenerateWay#DEFINITE}
     *
     * @param generateValidityWay 生成有效期方式：DEFINITE(指定日期) / DELAY（延迟生效）
     * @param specifiedStartDate  指定开始日期 【指定日期 方式传入参数】
     * @param specifiedEndDate    指定结束日期 【指定日期 方式传入参数】
     * @param delayDays           延迟多少天 【延迟生效 方式传入参数】
     * @return [有效开始日期, 有效结束日期]
     */
    @NonNull
    public static LocalDate[] getValidityRange(@NonNull String generateValidityWay,
                                               LocalDate specifiedStartDate,
                                               LocalDate specifiedEndDate,
                                               Integer delayDays) {


        if (Constant.DictValidityGenerateWay.DEFINITE.equals(generateValidityWay)) {

            if (specifiedStartDate == null || specifiedEndDate == null) {
                throw new IllegalArgumentException("指定日期方式：必须指定开始和结束日期");
            }

            if (specifiedEndDate.isBefore(specifiedStartDate)) {
                throw new IllegalArgumentException("指定的结束日期必须>=开始日期");
            }


            return new LocalDate[]{specifiedStartDate, specifiedEndDate};
        }

        delayDays = Math.max(0, delayDays);

        LocalDate startDate = LocalDate.now();

        LocalDate endDate = startDate.plusDays(delayDays);

        return new LocalDate[]{startDate, endDate};
    }


    /**
     * {@link Constant.DictValidityGenerateWay#DEFINITE}
     *
     * @param generateValidityWay 生成有效期方式：DEFINITE(指定日期) / DELAY（延迟生效）
     * @param specifiedStartDate  指定开始日期 【指定日期 方式传入参数】
     * @param delayDays           延迟多少天生效 【延迟生效 方式传入参数】
     * @return 有效开始日期
     */
    @NonNull
    public static LocalDate getValidityStartDate(@NonNull String generateValidityWay,  LocalDate specifiedStartDate, Integer delayDays) {
        return doGetValidityDate(generateValidityWay, specifiedStartDate, LocalDate.now(), delayDays);
    }

    @NonNull
    public static LocalDate getValidityStartDate(@NonNull String generateValidityWay,
                                                 LocalDate specifiedStartDate,
                                                 LocalDate baseDate,
                                                 Integer delayDays) {

        return doGetValidityDate(generateValidityWay, specifiedStartDate, baseDate, delayDays);
    }


    /**
     * {@link Constant.DictValidityGenerateWay#DEFINITE}
     *
     * @param generateValidityWay 生成有效期方式：DEFINITE(指定日期) / DELAY（延迟生效）
     * @param specifiedEndDate    指定结束日期 【指定日期 方式传入参数】
     * @param delayDays           延迟多少天失效 【延迟生效 方式传入参数】
     * @return 有效结束日期
     */
    @NonNull
    public static LocalDate getValidityEndDate(@NonNull String generateValidityWay, LocalDate specifiedEndDate,
                                               LocalDate validityStartDate, Integer delayDays) {

        if (delayDays != null && delayDays >= 1) {
            delayDays = delayDays - 1;
        }

        return doGetValidityDate(generateValidityWay, specifiedEndDate, validityStartDate,delayDays);
    }


    /**
     * 获取有效期状态
     *
     * @param validityStartDate 有效开始日期
     * @param validityEndDate   有效结束日期
     * @return 有效期状态
     */
    @NonNull
    public static String getValidityStatus(@NonNull LocalDate validityStartDate,  @NonNull LocalDate validityEndDate) {

        LocalDate now = LocalDate.now();

        if (now.isAfter(validityEndDate)) {
            return Constant.DictValidityStatus.EXPIRED;
        }

        if (now.isBefore(validityStartDate)) {
            return Constant.DictValidityStatus.INEFFECTIVE;
        }

        return Constant.DictValidityStatus.EFFECTIVE;
    }


    /**
     * @param validityStartDate 有效开始日期
     * @param validityEndDate   有效结束日期
     * @return 返回是否已过期
     */
    public static boolean isExpired(@NonNull LocalDate validityStartDate, @NonNull LocalDate validityEndDate) {
        return Constant.DictValidityStatus.EXPIRED.equals(getValidityStatus(validityStartDate, validityEndDate));
    }


    private static LocalDate doGetValidityDate(String generateValidityWay, LocalDate specifiedDate, LocalDate baseDate, Integer delayDays) {
        if(Constant.DictValidityGenerateWay.IMMEDIATE.equals(generateValidityWay)) {
            return baseDate;
        }

        if (Constant.DictValidityGenerateWay.DEFINITE.equals(generateValidityWay)) {
            if (specifiedDate == null) {
                throw new IllegalArgumentException("指定日期方式：必须指定生效/失效日期");
            }
            return specifiedDate;
        }

        delayDays = Math.max(0, delayDays);

        return baseDate.plusDays(delayDays);
    }

}
