package com.rocogz.merchant.entity.scm;

import com.rocogz.merchant.constant.Constant;
import com.rocogz.merchant.dto.scm.ScmSendDto;
import com.rocogz.merchant.entity.channel.product.MerchantChannelProduct;
import com.rocogz.merchant.entity.channel.product.MerchantDeductSubjectParams;
import com.rocogz.merchant.entity.electric.southern.MerchantElectricRechargeCardBatch;
import com.rocogz.merchant.entity.goods.MerchantGoods;
import com.rocogz.merchant.entity.goods.MerchantGoodsMeal;
import com.rocogz.merchant.entity.goods.MerchantGoodsValidityRule;
import com.rocogz.syy.common.entity.IdEntity;
import lombok.Data;
import lombok.EqualsAndHashCode;
import lombok.experimental.Accessors;

import java.math.BigDecimal;
import java.time.LocalDate;
import java.time.LocalDateTime;

/**
 * <p>
 * 明细订单
 * </p>
 *
 * @author liangyongtong
 * @since 2021-03-09
 */
@Data
@EqualsAndHashCode(callSuper = true)
@Accessors(chain = true)
public class MerchantScmOrderDetail extends IdEntity {

    private static final long serialVersionUID = 4042786962586393961L;
    /**
     * 订单号，供应链下游订单号
     */
    private String orderCode;

    /**
     * 二级订单分组编码,发放的主产品按照数量分拆的编号
     */
    private String itemGroupCode;

    /**
     * 明细订单编号,即供应链上游订单号,unique
     *
     * @see com.rocogz.syy.equity.entity.userCoupon.EquityUserCouponLaunchInfo#orderItemCode 发放用户券时设置到发放信息此字段上
     */
    private String orderItemCode;

    /**
     * 合作客户产品编码,发放的客户产品编号,如果是发放的 第三方套餐 客户产品编号有值
     * (1)如果是积分兑换的 则是NULL,
     * (2)如果是平台自定义套餐，则是NULL
     */
    private String customerProductCode;

    /**
     * 合作客户产品名称
     */
    private String customerProductName;

    /**
     * 代理商产品编码
     * 积分兑换的：服务商产品编号 有值
     * 发放的客户产品编号，服务商产品编号有值
     */
    private String agentProductCode;

    /**
     * 代理商产品名称
     */
    private String agentProductName;

    /**
     * 平台商品编码
     * 如果是平台自定义套餐, 该平台产品编号是套餐中子产品的平台产品编号
     */
    private String productCode;

    /**
     * 平台商品名称
     */
    private String productName;

    /**
     * 产品类别,上游订单中的产品类型,该产品类型用来决定走不同的上游订单Handler处理器类，非常重要
     * (1) 如果是不是平台自定义套餐,productType = MerchantScmProductInformation#productType
     * (2) 如果是平台自定义套餐，则 productType = 自定义平台套餐中子产品的产品类型
     *
     * @see Constant.DictData#PRODUCT_TYPE_DRIVING_SERVICE
     */
    private String productType;


    /**
     * 第三方产品编码,即上游产品编号,供应商产品编号
     * (1) 就是 在分销聚合服务系统-> 创建平台产品页面中 产品供应商信息中 填写的 供应商产品编号
     * (2) 同时也是 在分销聚合服务系统-> 创建通道产品页面中 填写的 第三方产品编号
     * （3）滴滴出行套餐券：配置的滴滴的套餐策略id
     * <p>
     * 如果是平台自定义套餐,则thirdProductCode = 子产品的平台供应商产品编码 = MerchantGoodsMeal#supplierGoodsCode
     *
     * @see MerchantScmProductInformation#thirdProductCode
     * @see MerchantChannelProduct#thirdProductCode
     * <p>
     * 第一步：刚开始创建供应链订单，还没有触发触发上游时：thirdProductCode= 平台产品上配置的供应商产品编号
     * 第二步：如果触发了上游,则thirdProductCode会更新为通道产品上的 thirdProductCode
     */
    private String thirdProductCode;

    /**
     * 上游订单重试次数
     * 定时任务重试上游查询 retry_num>=1 && retry_num<=6 && order_status = FAILURE
     *
     * @see com.rocogz.merchant.service.merchant.scm.service.impl.MerchantScmInformationServiceImpl#scheduleRetryFailedUpOrder()
     */
    private Integer retryNum;

    /**
     * 上游订单最后重试时间
     */
    private LocalDateTime lastestRetryTime;

    /**
     * 回调地址
     */
    private String callbackUrl;

    /**
     * 上游订单订单状态:调用dingju后更新上游订单状态
     * 更新上游订单入口：在com.rocogz.syy.message.server.scm.MerchantScmOrderHandler实现类中回调更新
     * （1）com.rocogz.syy.message.server.scm.consumer.ScmConsumer#handleReorderJms
     * （2）com.rocogz.syy.message.server.scm.consumer.ScmConsumer#handleOrderJms
     *
     * @see Constant.DictData#SCM_ORDER_STATUS_PENDING
     */
    private String orderStatus;

    /**
     * 第三方业务状态
     */
    private String thirdBusinessStatus;

    /**
     * 第三方业务号,例如：鼎聚的powerNo
     *
     * @see com.rocogz.syy.equity.entity.userCoupon.EquityUserCouponInfo#thirdCode
     * <p>
     * 如果电卡产品 thirdBusinessCode是电卡的流水号 例如：SPGS20230510000003
     * @see com.rocogz.syy.charge.entity.ChargeProcess#turnoverCode
     */
    private String thirdBusinessCode;

    /**
     * 第三方失败原因
     * （1）调用上游失败原因
     * （2）上游退款失败原因，例如：电卡过期退款失败原因
     */
    private String thirdReason;

    /**
     * 发放张数:下游订单张数 =1
     */
    private Integer grantNum;

    /**
     * ROCO车服用户券编号,在上游订单中 只有 goodsNature = Constant.DictData#GOODS_NATURE_ITEM 才有用户券编号
     *
     * @see com.rocogz.syy.equity.entity.userCoupon.EquityUserCouponInfo#code
     * @see com.rocogz.syy.equity.entity.userCoupon.EquityUserCouponLaunchInfo#userCouponCode
     */
    private String rocoUserCouponCode;

    /**
     * 用户券发放状态:
     * 待发放或已发放
     *
     * @see Constant.DictData#SCM_USER_COUPON_GRANT_STATUS_PENDING
     */
    private String userCouponGrantStatus;

    /**
     * 用户券发放异常原因
     */
    private String userCouponGrantErrorReason;

    /**
     * 用户券发放重试次数
     */
    private Integer userCouponGrantRetryNum;

    /**
     * 产品性质:单品/供应商定制套餐,上游订单中的产品性质只有：单品 / 供应商定制套餐 两种,
     * 因为平台自定义套餐会按照子产品 分拆成这2性质的上游订单
     *
     * @see Constant.DictData#GOODS_NATURE_ITEM
     */
    private String goodsNature;

    /**
     * 订单核销状态,车主单核销后,权益基础服务发消息通知更新此核销状态
     * (1)电卡产品： 上游发放成功时，就会更新为已核销
     *
     * @see 消息中心电卡处理器入口  com.rocogz.syy.message.server.scm.service.CommonElectricOrderHandler#order
     * <p>
     * (2)其他的更新为已核销入口是：
     * @see MerchantScmOrderInformationServiceImpl#updateWriteOffStatus(WriteOffDto)
     */
    private String writeOffStatus;

    /**
     * 订单核销时间
     */
    private LocalDateTime writeOffTime;

    /**
     * 单个明细订单面值/平台产品市场价, 不管是积分兑换，还是发放的,壳牌的都是 非固定面额的
     * 调用上游 鼎聚的油卡充值金额 传入的就是这个字段  <br/>
     * 调用上游 电卡充值金额,电卡卡中要充值的金额 传入的也是这个字段  <br/>
     * <p>
     * （1）非固定面额的油卡直充:        detailPrice = 充值金额 拆单后的面额 {@link ScmSendDto#unitFaceValue}
     * （2）如果是非固定面额电卡或积分：  detailPrice = 输入的充值金额 {@link ScmSendDto#unitFaceValue}
     * （3）其他产品类型 ：            detailPrice = 平台产品市场价 {@link MerchantGoods#marketPrice}
     * (4)如果是自定义平台套餐：      detailPrice =  {@link MerchantGoodsMeal#marketPrice}
     */
    private BigDecimal detailPrice;

    /**
     * 如果是C端加油的 discountAmt = {@link ScmSendDto#discountAmount} 拆单后的优惠抵扣金额
     */
    private BigDecimal discountAmt;

    /**
     * 本明细订单应该发放用户券张数，如果是套餐，则等于= sum(所有套餐子商品的com.rocogz.merchant.entity.goods.MerchantGoodsMeal#sum）
     * 如果是供应商套餐，itemGrantNum = 该orderItemCode下的所有merchant_scm_order_combo_detail
     * 如果是单品=1
     */
    private Integer itemGrantNum;

    /**
     * 本明细订单发放用户券成功张数
     */
    private Integer itemGrantSuccessNum;

    /**
     * 本明细订单用户券发放成功时间
     */
    private LocalDateTime itemGrantSuccessTime;

    /**
     * 第三方业务编号更新到权益中心状态
     */
    private String powerNoToEquityStatus;

    /**
     * 第三方业务编号更新到权益中心执行次数
     */
    private Integer powerNoToEquityNum;

    /**
     * 领取验证状态：用来标记：车主端客户是否点击了领取按钮
     */
    private String receiveValidateStatus;

    /**
     * 领取验证时间
     */
    private LocalDateTime receiveValidateTime;

    /**
     * 合作备付金扣减时点,上游订单上的此字段没用了，因为开始客户的备付金是按照上游订单扣减的，后来客户的备付金全部按照下游订单扣减了
     */
    @Deprecated
    private String dedutionTimePoint;

    /**
     * 备付金扣减状态：UNPAID、FAILURE、SUCCESS
     *
     * @see Constant.DictData#SCM_PAY_STATUS_FAILURE
     */
    @Deprecated
    private String coverPayStatus;

    /**
     * 备付金扣减成功时,备付金账户交易流水号
     */
    @Deprecated
    private String coverFlowCode;

    /**
     * 备付金扣减尝试次数
     */
    @Deprecated
    private Integer coverPayRetryNum;

    /**
     * 总备付金扣减金额
     */
    @Deprecated
    private BigDecimal coverPayPrice;

    /**
     * 备付金尝试扣减时间
     */
    @Deprecated
    private LocalDateTime coverPayTime;

    /**
     * 备付金领取状态：备付金账户中的领取金额是否增加成功
     */
    @Deprecated
    private String coverReceiveStatus;

    /**
     * 备付金退款状态
     */
    @Deprecated
    private String refundStatus;

    /**
     * 备付金退款金额 = payQuotaPrice
     */
    @Deprecated
    private BigDecimal refundPrice;

    /**
     * 备付金退款时间
     */
    @Deprecated
    private LocalDateTime refundTime;

    /**
     * 撤回时间
     */
    private LocalDateTime recallTime;

    /**
     * 订单类型,等于下游订单的orderType
     *
     * @see MerchantScmOrderInformation#orderType
     * @see Constant.DictData#SCM_ORDER_TYPE_SYS
     */
    private String orderType;

    /**
     * 备付金支付金额,备付金账号需要扣减的备付金金额
     */
    @Deprecated
    private BigDecimal payQuotaPrice;

    /**
     * 现金支付金额
     */
    @Deprecated
    private BigDecimal payCashPrice;

    /**
     * 上游订单发放成功时间
     */
    private LocalDateTime thirdGrantSuccessTime;


    /**
     * 平台产品采购单价
     *
     * @see MerchantGoodsMeal#costPrice (平台自定义套餐)
     * @see MerchantGoods#costPrice
     * <p>
     * 1. 如果是 非电卡产品，就是要扣减的南网电卡的金额
     * (1.1) platProdPurchasePrice =  MerchantGoods#costPrice   （不是平台自定义套餐）
     * (1.2) platProdPurchasePrice  = MerchantGoodsMeal#costPrice (平台自定义套餐)
     * <p>
     * 2. 如果是电卡产品
     * （2.1) platProdPurchasePrice =   MerchantGoods#costPrice 拆单后的采购节 (固定面额电卡)
     * (2.2) platProdPurchasePrice =  merchantGoods.costPrice * unitFaceValue/MerchantGoods.marketPrice 后的拆单 采购价 (非固定面额电卡)
     */
    private BigDecimal platProdPurchasePrice;

    /**
     * 平台产品 市场单价/面额
     *
     * @see MerchantGoods#marketPrice
     * @see MerchantGoodsMeal#marketPrice  如果是平台自定义套餐中的子产品
     */
    private BigDecimal platProdMarketPrice;


    /**
     * 调用上游接口 触发方式：自动/手工
     *
     * @see Constant.DictTriggerRemoteApiMode#MANUAL
     * @see MerchantGoodsValidityRule#remoteTriggerMode
     */
    private String triggerRemoteApiMode;

    /**
     * 使用有效期设置时点：发放时设置，领取时设置
     *
     * @see Constant.DictValiditySettingTimePoint#ISSUE
     * @see MerchantGoodsValidityRule#settingTimePoint
     */
    private String useValiditySetTimePoint;

    /**
     * 使用有效期规则Json
     *
     * @see MerchantGoodsValidityRule
     */
    private String useValidityRuleJson;

    /**
     * 平台产品延迟多少天失效【失效类型是 延迟生效】
     *
     * @see MerchantGoods#useValidity
     */
    private Integer useValidityInvalidDelayDays;

    /**
     * 使用有效期：开始日期
     */
    private LocalDate useValidityStartDate;

    /**
     * 使用有效期：结束日期
     */
    private LocalDate useValidityEndDate;

    /**
     * 使用有效期状态：未生效，已生效，已过期
     *
     * @see Constant.DictValidityStatus#INEFFECTIVE
     * (1)过期： 更新为过期的条件：使用有效期到了过期时间 && userStatus in ("未使用","部分使用") and order_status not in ('CANCEL','BACK')
     *   也就是说：userStatus=已使用不会更新为过期
     * (2)滴滴产品：如果使用有效更新为过期后，会再次查询上游滴滴券是否使用，如果滴滴券已使用，则会把滴滴券用户券更新为核销，
     *  滴滴的MerchantScmOrderComboDetail的核销状态也核销状态，通过更新上游订单的useStatus使用状态为 部分使用或已使用
     *  定时任务更新入口如下：com.rocogz.business.scm.service.ScmOrderUseValidityService#fetchUpateDidiCouponStatusWhenUsed
     */
    private String useValidityStatus;

    /**
     * 已过期标志位：只要领取有效期或使用有效期过期了，就算过期
     * 进入到过期补发列表的订单是：expireFlag=1 and use_status in ('UN_USED','PART_USED')
     */
    private Integer expireFlag;

    /**
     * 使用状态：未使用，部分使用、已使用
     * 1. 产品类型是：油卡(包括直冲和非直充油卡) 或 积分类型，并且使用有效期过期后才会更新此字段
     * 现在的业务：只是使用有效期过期后,定时任务调用interface查询油卡余额接口更新此字段
     * @see MerchantScmOrderDetailServiceImpl#scheduleFindToUseOilCardOrderItemList
     * @see MerchantScmOrderDetailServiceImpl#scheduleBatchUpdateUseStatusOilCardOrderItemList
     *
     * <p>
     * 2.滴滴产品： 更新核销状态时就会 更新为部分使用或已使用。如果全部核销完毕,更新为已使用，否则更新为部分使用
     *  (1) 核销入口：MerchantScmOrderInformationServiceImpl#updateWriteOffStatus(WriteOffDto)
     *  (2) 滴滴产品使用有效期过期后，会查询滴滴产品上游使用状态，如果上游已使用，则会更新核销和使用状态
     *     入口如下：MerchantScmOrderInformationServiceImpl#updateDidiWriteOffStatusWhenExpired
     * <p>
     * 3. 南网电卡产品： 更新为部分使用或已使用
     * 电卡消费后会回调interface接口
     * interface回调入口如下： com.rocogz.third.electric.southern.services.ApiCallbackSouthernElectricService#userChargeCallback
     * 商户基础服务更新使用状态入口是：MerchantScmOrderDetailController#updateUseStatus(UpdateUseStatusDto)
     * <p>

       <p>
        4. 星星充电，根据查询的余额，更新部分使用/已使用
          更新入口：车主端星星充电充电记录列表查询，用户券查询上游剩余电卡余额，更新使用状态， 使用有效期过期时，查询余额，更新使用状态
       @see com.rocogz.merchant.service.merchant.scm.controller.MerchantScmOrderDetailController#batchUpdateUseStatus
        </p>

     * 5. 支付宝红包：更新为核销时，就会使用状态更新为已使用
     * 入口是 MerchantScmOrderInformationServiceImpl#updateWriteOffStatus(WriteOffDto)
     * <p>
     *
     * <p>
     *   5.服务券：核销时更新为已核销时 并且把用使用状态更新为已使用
     *   （1）admin端核销入口 com.rocogz.business.equity.writeoff.service.CouponWriteOffService#doWriteOffOrder
     *    (2)管理端小程序核销入口：com.rocogz.service.agentb.BAgentCouponWriteOffService#doWriteOffOrder
     * </p>
     *
      6. 如果是ROCO牌的抵扣券（油卡产品,包括直充，非直充）,核销时会把该抵扣券的上游更新为已使用
     * 更新入口如下：MerchantScmOrderInformationServiceImpl#updateWriteOffStatus(WriteOffDto)
     * @see Constant.DictUseStatus
     * @see MerchantScmOrderInformationServiceImpl#updateWriteOffStatus(WriteOffDto)
     */
    private String useStatus;


    /**
     * 使用状态更新时间,只要更新为 部分使用，或已使用，都需要更新此字段
     */
    private LocalDateTime useUpdateTime;

    /**
     * 过期处理状态
     *
     * @see Constant.DictValidityExpiredHandleStatus
     */
    private String expiredHandleStatus;

    /**
     * 通道产品-编码
     *
     * @see MerchantChannelProduct#code
     */
    private String channelProductCode;

    /**
     * 通道产品-名称
     */
    private String channelProductName;

    /**
     * 通道产品-第三方平台编码，例如：DJ_GDZSY（"广东中石油"）
     *
     * @see MerchantChannelProduct#platformCode
     */
    private String channelProductPlatformCode;

    /**
     * 通道产品-第三方平台名称
     */
    private String channelProductPlatformName;

    /**
     * 通道产品-扣款主体编号
     *
     * @see MerchantChannelProduct#deductSubjectCode
     * @see MerchantDeductSubjectParams#subjectCode  通道扣款主体配置的参数，例如：appId,appsecret
     */
    private String channelProductDeductSubjectCode;

    /**
     * 通道产品-扣款主体名称
     */
    private String channelProductDeductSubjectName;

    /**
     * 通道产品-成本价
     */
    private BigDecimal channelProductPurchasePrice;

    /**
     * 通道产品-市场价
     */
    private BigDecimal channelProductMarketPrice;

    /**
     * 通道产品-过期操作类型
     *
     * @see MerchantChannelProduct#orderDelayReissue
     */
    private String channelProductOrderDelayReissue;

    /**
     * 第三方订单号,
     * (1)如果是电卡产品，则保存的是电卡产品的批次号
     *
     * @see MerchantElectricRechargeCardBatch#batchNo
     */
    private String thirdOrderCode;


    /**
     * 发放的实物供应链订单(非积分兑换的),对应的商品订单创建状态
     *
     * @see Constant.DictData#SCM_GOODS_ORDER_CREATE_RESULT_PENDING
     * 实物订单创建结果
     */
    private String goodsOrderCreateResult;

    /**
     * 实物订单创建商品订单重试次数
     * 定时重试入口：重试次数：[1-6]的会重试
     * @see com.rocogz.merchant.service.merchant.scm.service.impl.ScmGoodsOrderService#scheduleRetryCreateGoodsOrder()
     */
    private Integer goodsOrderCreateRetryNum;

    /**
     * 实物订单创建失败原因
     */
    private String goodsOrderCreateFailReason;

    /**
     * 实物订单创建时间
     */
    private LocalDateTime goodsOrderCreateTime;


    /**
     * 是否已完成补发 （Y/N）
     *
     * @see Constant.DictData#COMMON_STATUS_YES
     */
    private String reissued;

    /**
     * 过期订单对应的补发下游订单号【过期的上游订单上有值】,例如：SCM20210312000010
     * 该值保存的新下游补发订单号，该字段的值保存在过期上游订单记录上
     */
    private String reissueBy;

    /**
     * 补发订单对应的原上游订单号，为哪个上游订单补发[新补发的订单有值]，例如：SCM20210705001641_1_1
     * 保存的值是原上游订单号,该值记录在新补发上游订单上
     */
    private String reissueFor;

    /**
     * 补发金额 【被补发订单有值,即过期的上游订单上有值】
     * @see MerchantScmReissueApplyDetail#reissueAmount
     */
    private BigDecimal reissuedAmount;

    /**
     * 过期金额
     * 目前还没有地方更新和使用该字段
     * (1) interface 查询 上游余额接口是：  /api/order/remote/detail
     * @see com.rocogz.client.InterfaceAggregateClientService#queryOrderRemoteDetail
     */
    private BigDecimal expiredAmount;


    /**
     * 已延期次数，默认为0
     * 使用有效期过期，或领取有效期过期后，做了延期操作
     * @see  com.rocogz.merchant.service.merchant.scm.controller.MerchantScmInformationController#dateDefer
     */
    private Integer deferCnt;


    //=== 下面的属性用来传值


    //该上游订单产品的品牌,而不是整个下游订单发放产品的品牌，即可能是供应商套餐中子产品的品牌
    private transient String brandCode;

    //产品订单发放平台信息
    private transient MerchantScmInformation scmGrantInfo;

    //产品订单 目标用户表
    private transient MerchantScmTargetUser scmGrantTargetUser;

    private transient MerchantScmProductInformation scmProductInfo;

    private transient MerchantScmOrderInformation orderInformation;

}
