package com.rocogz.merchant.dto.scm.order;

import com.rocogz.merchant.constant.Constant;
import com.rocogz.merchant.entity.brand.MerchantBrand;
import com.rocogz.merchant.entity.channel.product.MerchantChannelProduct;
import com.rocogz.merchant.entity.channel.product.MerchantThirdPlatform;
import com.rocogz.merchant.entity.scm.MerchantScmOrderDetail;
import com.rocogz.merchant.entity.scm.MerchantScmProductInformation;
import lombok.Getter;
import lombok.NoArgsConstructor;
import lombok.Setter;
import lombok.experimental.Accessors;
import org.springframework.util.StringUtils;

import java.util.ArrayList;
import java.util.List;

/**
 * 撤回下游订单中,包括的上游订单
 *
 * @author zhangmin
 */
@Setter
@Getter
@NoArgsConstructor
@Accessors(chain = true)
public class RecallScmOrderItem {



    /**
     * 上游订单编号
     */
    private String scmOrderItemCode;


    /**
     * 上游订单状态
     *
     * @see MerchantScmOrderDetail#orderStatus
     * @see Constant.DictData#SCM_ORDER_STATUS_PENDING
     */
    private String orderItemStatus;

    /**
     * 使用状态
     *
     * @see MerchantScmOrderDetail#useStatus
     */
    private String useStatus;

    /**
     * 上游订单中的产品 产品性质：单品还是套餐
     *
     * @see Constant.DictData#GOODS_NATURE_ITEM
     * @see MerchantScmProductInformation#productNature
     */
    private String productNature;


    /**
     * 订单类型
     *
     * @see MerchantScmOrderDetail#orderType
     */
    private String orderType;


    /**
     * 上游订单的对应的 第三方业务单号
     *
     * @see MerchantScmOrderDetail#thirdBusinessCode
     */
    private String thirdBusinessCode;


    /**
     * 上游订单中的产品的产品类型
     *
     * @see MerchantScmOrderDetail#productType
     */
    private String productType;


    /**
     * 通道产品-第三方平台编码，例如：DJ_GDZSY（"广东中石油"）
     *
     * @see MerchantChannelProduct#platformCode
     */
    private String channelProductPlatformCode;

    /**
     * 通道产品-扣款主体编号
     *
     * @see MerchantChannelProduct#deductSubjectCode
     * @see MerchantDeductSubjectParams#subjectCode  通道扣款主体配置的参数，例如：appId,appsecret
     */
    private String channelProductDeductSubjectCode;

    /**
     * 上游订单的 产品的品牌
     *
     * @see MerchantBrand#DIDI
     */
    private String brandCode;


    public static RecallScmOrderItem of(MerchantScmOrderDetail orderDetail) {
        if (orderDetail == null) {
            return null;
        }

        RecallScmOrderItem orderItem = new RecallScmOrderItem();

        orderItem.setOrderItemStatus(orderDetail.getOrderStatus())
                .setScmOrderItemCode(orderDetail.getOrderItemCode())
                .setProductNature(orderDetail.getGoodsNature())
                .setProductType(orderDetail.getProductType())
                .setOrderType(orderDetail.getOrderType())
                .setUseStatus(orderDetail.getUseStatus())
                .setThirdBusinessCode(orderDetail.getThirdBusinessCode())
                .setBrandCode(orderDetail.getBrandCode())
                .setChannelProductPlatformCode(orderDetail.getChannelProductPlatformCode())
                .setChannelProductDeductSubjectCode(orderDetail.getChannelProductDeductSubjectCode());

        return orderItem;
    }


    /**
     * 是否是要走 广东中石油通道的上游订单
     *
     * @return
     */
    public boolean isGdzsyOrderItem() {

        if (!(Constant.DictData.PRODUCT_TYPE_OIL_CARD_DIGITAL_YES.equalsIgnoreCase(productType) ||
                Constant.DictData.PRODUCT_TYPE_OIL_CARD_DIGITAL.equalsIgnoreCase(productType))) {
            //如果不是 油卡产品类型，则直接返回false
            return false;
        }

        //如果是油卡产品类型

        //1. 如果上游通道是 DJ_GDZSY(广东中石油)
        //2. 如果上游订单通道为空，并且品牌是走广东中石油的品牌 （这种情况，说明还没有触发上游）
        return MerchantThirdPlatform.DJ_GDZSY.equalsIgnoreCase(channelProductPlatformCode) ||
                (StringUtils.isEmpty(channelProductPlatformCode) && MerchantThirdPlatform.GDZSY_CHANNEL_BRANDS.contains(brandCode));

    }




    /**
     * 是否是要走 四川中石油通道的上游订单
     *
     * @return
     */
    public boolean isSczsyOrderItem() {

        if (!(Constant.DictData.PRODUCT_TYPE_OIL_CARD_DIGITAL_YES.equalsIgnoreCase(productType) ||
                Constant.DictData.PRODUCT_TYPE_OIL_CARD_DIGITAL.equalsIgnoreCase(productType))) {
            //如果不是 油卡产品类型，则直接返回false
            return false;
        }

        //如果是油卡产品类型

        //1. 如果上游通道是 DJ_SCZSY(四川中石油)
        //2. 如果上游订单通道为空，并且品牌是走广东中石油的品牌 （这种情况，说明还没有触发上游）
        return MerchantThirdPlatform.DJ_SCZSY.equalsIgnoreCase(channelProductPlatformCode) ||
                (StringUtils.isEmpty(channelProductPlatformCode) && MerchantBrand.ZGSY.equalsIgnoreCase(brandCode));

    }


}
