package com.rocogz.common.service;

import com.alibaba.fastjson.JSON;
import com.rocogz.common.api.enums.SerialNoRuleEnum;
import com.rocogz.common.dao.SerialNoMapper;
import com.rocogz.common.entity.SerialNo;
import com.rocogz.common.lock.DistributedLock;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.dao.DataAccessException;
import org.springframework.data.redis.connection.RedisConnection;
import org.springframework.data.redis.core.RedisCallback;
import org.springframework.data.redis.core.StringRedisTemplate;
import org.springframework.data.redis.serializer.RedisSerializer;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;
import tk.mybatis.mapper.entity.Example;

import java.time.Duration;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.time.temporal.TemporalAdjusters;
import java.util.List;


/**
 * <dl>
 * <dd>Description: 发号器</dd>
 * <dd>Company: 广州大城若谷信息技术有限公司</dd>
 * <dd>@date：2020/02/13</dd>
 * <dd>@author：andrew</dd>
 * </dl>
 */
//@ConditionalOnBean({RedisDistributedLock.class, SerialNoMapper.class})
@Service
@Slf4j
public class SerialNoService {

    /**
     * 前缀标志
     */
    private static final String FLAG_PREFIX = "GENERATE_NO";

    /**
     * 扩容阈值， 小于则发起扩容，  必须大于并发数 否则可能出现获取失败的情况
     */
    private static final int CAPACITY_THRESHOLD = 500;

    /**
     * 每批次取数， 必须大于程序并发数， 否则可能出现获取失败的情况
     */
    private static final Long BATCH_SIZE = 1000L;

    /**
     * 重试最大次数
     */
    private static final int RETRY_LIMIT = 3;

    /**
     * 重试间隔时间
     */
    private static final long RETRY_TIME = 3000L;

    enum RedisKey {
        /**
         * 当前序号
         */
        CUR_SEQ_NO_PREFIX,

        /**
         * 序号配置
         */
        SEQ_NO_CONF_PREFIX,

        /**
         * 初始化序号
         */
        INIT_SEQ_NO_PREFIX,

        /**
         * 重置序号
         */
        REST_SEQ_NO_PREFIX,

        /**
         * 扩容序号
         */
        CAPACITY_SEQ_NO_PREFIX
    }

    private SerialNoMapper serialNoMapper;

    private StringRedisTemplate redisTemplate;

    private DistributedLock distributedLock;


    public SerialNoService(SerialNoMapper serialNoMapper, StringRedisTemplate redisTemplate, DistributedLock distributedLock) {
        log.info("初始化组件:[发号器]");
        this.serialNoMapper = serialNoMapper;
        this.redisTemplate = redisTemplate;
        this.distributedLock = distributedLock;
    }

    @Transactional(propagation = Propagation.NOT_SUPPORTED)
    public String generateCode(String typeEnum) {
        //重试次数
        int retryNum = 0;

        //1.获取可用的缓存配置
        SerialNo cacheSerialNo = getCacheAvailableSerialNo(typeEnum, retryNum);

        Assert.notNull(cacheSerialNo, "系统错误，没有可用的序号");

        //2.序号++
        String seqKey = getRedisKey(typeEnum, RedisKey.CUR_SEQ_NO_PREFIX);
        Long seqNo = redisTemplate.opsForValue().increment(seqKey);

        //3.校验序号
        checkSeq(seqNo, cacheSerialNo, typeEnum);

        //4.扩容判断
        checkCapacity(seqNo, cacheSerialNo, typeEnum);

        //5.拼接序号返回
        return aggregateResult(seqNo, cacheSerialNo);
    }


    private String aggregateResult(Long seqNo, SerialNo cacheSerialNo) {
        StringBuilder result = new StringBuilder();
        //拼接前缀
        result.append(cacheSerialNo.getSeqPrefix());
        //拼接日期
        result.append(formatPattern(cacheSerialNo.getLastestDate(), cacheSerialNo.getFormatPattern()));
        //拼接序号
        result.append(StringUtils.leftPad(seqNo.toString(), cacheSerialNo.getSeqLpadLength(), "0"));

        String resultStr = result.toString();
        log.info("generateCode:{}", resultStr);
        return resultStr;
    }


    /**
     * %Y
     * %m
     * %d
     *
     * @param pattern
     * @return
     */
    private String formatPattern(LocalDate lastDate, String pattern) {
        if (StringUtils.isNotBlank(pattern)) {
            return pattern.replace("%Y", String.valueOf(lastDate.getYear())).
                    replace("%m", StringUtils.leftPad(String.valueOf(lastDate.getMonthValue()), 2, "0")).
                    replace("%d", StringUtils.leftPad(String.valueOf(lastDate.getDayOfMonth()), 2, "0"));
        } else {
            return StringUtils.EMPTY;
        }
    }


    /**
     * 扩容校验
     *
     * @param seqNo
     * @param cacheSerialNo
     * @param typeEnum
     */
    private void checkCapacity(Long seqNo, SerialNo cacheSerialNo, String typeEnum) {
        long size = cacheSerialNo.getSeqNo() - seqNo;
        log.info("[{}]:缓存上限:[{}],剩余数:[{}]", typeEnum, cacheSerialNo.getSeqNo(), size);
        //如果剩余号小于阈值，则发起扩容
        if (size < CAPACITY_THRESHOLD) {
            SerialNo serialNo = serialNoMapper.selectByPrimaryKey(typeEnum);
            Assert.notNull(serialNo, "获取序号配置失败，请检查数据库配置：" + typeEnum);
            capacitySerialNo(serialNo, typeEnum);
        }
    }

    /**
     * 扩容序号
     *
     * @param typeEnum
     */
    private SerialNo capacitySerialNo(SerialNo serialNo, String typeEnum) {
        String lockKey = getRedisKey(typeEnum, RedisKey.CAPACITY_SEQ_NO_PREFIX);
        //1.加锁, 15秒， 不重试
        boolean tryLock = distributedLock.lock(lockKey, 1500L, 0);
        //如果拿到锁则进行扩容，否则跳过
        if (tryLock) {
            log.info("开始扩容");
            try {
                Long originVersion = serialNo.getVersion();
                Long originSeqNo = serialNo.getSeqNo() == null ? 0 : serialNo.getSeqNo();
                //新的
                Long newSeqNo = originSeqNo + BATCH_SIZE;
                Example example = new Example(SerialNo.class);
                example.createCriteria()
                        .andEqualTo("version", originVersion)
                        .andEqualTo("seqCode", serialNo.getSeqCode());
                serialNo.setVersion(originVersion + 1);
                serialNo.setSeqNo(newSeqNo);
                serialNo.setLastestDate(LocalDate.now());
                serialNo.setSeqVal(aggregateResult(newSeqNo, serialNo));
                int count = serialNoMapper.updateByExample(serialNo, example);
                //如果修改成功
                if (count > 0) {
                    //更新配置
                    redisTemplate.execute(new RedisCallback<List<Object>>() {
                        @Override
                        public List<Object> doInRedis(RedisConnection connection) throws DataAccessException {
                            connection.openPipeline();
                            //key 不存在时才设置, 默认为
                            connection.setNX(RedisSerializer.string().serialize(getRedisKey(typeEnum, RedisKey.CUR_SEQ_NO_PREFIX)), RedisSerializer.string().serialize(originSeqNo.toString()));
                            //设置配置
                            setRedisConf(connection, serialNo, typeEnum);
                            return connection.closePipeline();
                        }
                    });
                    log.info("[{}]:缓存上限：[{}], 扩容完毕", serialNo.getSeqCode(), serialNo.getSeqNo());
                    return serialNo;
                } else {
                    log.error("扩容修改序号失败");
                }
            } finally {
                distributedLock.releaseLock(lockKey);
            }
        } else {
            log.info("获锁失败，跳过扩容");
        }
        return null;
    }


    /**
     * 校验序号
     *
     * @param seq
     * @param serialNo
     */
    private void checkSeq(Long seq, SerialNo serialNo, String typeEnum) {
        try {
            Assert.notNull(seq, "生成序号为null");
            Assert.isTrue(seq.longValue() > 0, "生成序号为负数");
            Assert.isTrue(seq.longValue() <= serialNo.getSeqNo().longValue(), "生成序号超过上限");
            Assert.isTrue(seq.toString().length() <= serialNo.getSeqLpadLength().intValue(), "生成序号超过限制长度");
        } catch (Exception e) {
            clearCacheSerialNo(typeEnum);
            throw e;
        }
    }


    /**
     * 初始化序号
     *
     * @param typeEnum
     * @return
     */
    private SerialNo initSerialNo(String typeEnum, int retryNum) {
        log.info("初始化");
        //1.锁住
        String lockKey = getRedisKey(typeEnum, RedisKey.INIT_SEQ_NO_PREFIX);
        //锁住15秒， 不重试
        boolean tryLock = distributedLock.lock(lockKey, 15000L, 0);
        if (tryLock) {
            try {
                //查询出配置
                SerialNo serialNo = serialNoMapper.selectByPrimaryKey(typeEnum);
                Assert.notNull(serialNo, "获取序号配置失败，请检查数据库配置：" + typeEnum);
                //看是否需要重置
                SerialNo check = checkRestGet(serialNo, typeEnum, retryNum);
                if (check != null) {
                    //如果重置成功则直接返回
                    return check;
                } else {
                    //进行初始化
                    return capacitySerialNo(serialNo, typeEnum);
                }
            } finally {
                distributedLock.releaseLock(lockKey);
            }
        } else {
            //没有获取到锁，则重试
            return retry(typeEnum, retryNum);
        }
    }

    /**
     * 清理缓存的序号配置
     *
     * @param typeEnum
     */
    private void clearCacheSerialNo(String typeEnum) {
        redisTemplate.execute(new RedisCallback<List<Object>>() {
            @Override
            public List<Object> doInRedis(RedisConnection connection) throws DataAccessException {
                connection.openPipeline();
                connection.del(RedisSerializer.string().serialize(getRedisKey(typeEnum, RedisKey.SEQ_NO_CONF_PREFIX)));
                connection.del(RedisSerializer.string().serialize(getRedisKey(typeEnum, RedisKey.CUR_SEQ_NO_PREFIX)));
                return connection.closePipeline();
            }
        });

    }

    /**
     * 重置序号
     *
     * @param typeEnum
     * @return
     */
    private SerialNo resetSerialNo(String typeEnum, int retryNum) {
        log.info("重置序号");
        //1.锁住
        String lockKey = getRedisKey(typeEnum, RedisKey.REST_SEQ_NO_PREFIX);
        //锁住15秒， 不重试
        boolean tryLock = distributedLock.lock(lockKey, 15000L, 0);
        if (tryLock) {
            //如果获取到锁
            try {
                //查询出配置
                SerialNo serialNo = serialNoMapper.selectByPrimaryKey(typeEnum);
                Assert.notNull(serialNo, "获取序号配置失败，请检查数据库配置：" + typeEnum);

                //原来第版本号
                Long originVersion = serialNo.getVersion();
                serialNo.setVersion(originVersion + 1);
                //重置序号
                serialNo.setSeqNo(BATCH_SIZE);
                serialNo.setLastestDate(LocalDate.now());
                serialNo.setSeqVal(aggregateResult(BATCH_SIZE, serialNo));

                Example example = new Example(SerialNo.class);

                example.createCriteria().
                        andEqualTo("seqCode", serialNo.getSeqCode()).
                        //版本号一致
                                andEqualTo("version", originVersion);

                example.and().andNotEqualTo("lastestDate", LocalDate.now()).orIsNull("lastestDate");

                int count = serialNoMapper.updateByExample(serialNo, example);
                if (count > 0) {
                    //设置当前值为0 , 配置更新, 通过管道确保事务
                    redisTemplate.execute(new RedisCallback<List<Object>>() {
                        @Override
                        public List<Object> doInRedis(RedisConnection connection) throws DataAccessException {
                            connection.openPipeline();
                            //覆盖当前值
                            connection.set(RedisSerializer.string().serialize(getRedisKey(typeEnum, RedisKey.CUR_SEQ_NO_PREFIX)), RedisSerializer.string().serialize("0"));
                            //设置配置
                            setRedisConf(connection, serialNo, typeEnum);
                            return connection.closePipeline();
                        }
                    });
                    return serialNo;
                } else {
                    log.info("重置序号更新数据库失败");
                }
            } finally {
                distributedLock.releaseLock(lockKey);
            }
        } else {
            //没有获取到锁，则重试
            return retry(typeEnum, retryNum);
        }
        return null;
    }


    private void setRedisConf(RedisConnection connection, SerialNo serialNo, String typeEnum) {
        byte[] conf = RedisSerializer.string().serialize(JSON.toJSONString(serialNo));
        Long expire = getExpireSeconds(serialNo.getSeqNoResetRule());
        if (expire != null) {
            log.info(" rule:{},  expire:{}", serialNo.getSeqNoResetRule(), expire);
            connection.setEx(RedisSerializer.string().serialize(getRedisKey(typeEnum, RedisKey.SEQ_NO_CONF_PREFIX)), expire, conf);
        } else {
            connection.set(RedisSerializer.string().serialize(getRedisKey(typeEnum, RedisKey.SEQ_NO_CONF_PREFIX)), conf);
        }
    }

    /**
     * 获取截止过期时间， 秒
     *
     * @param restRuleStr
     * @return
     */
    private Long getExpireSeconds(String restRuleStr) {
        SerialNoRuleEnum restRule = SerialNoRuleEnum.getByName(restRuleStr);
        if (restRule != null) {
            LocalDateTime now = LocalDateTime.now();
            //当天 23：59：59
            LocalDateTime lastDayTime = LocalDateTime.of(LocalDate.now(), LocalTime.MAX);
            LocalDateTime end = null;
            switch (restRule) {
                case DAY:
                    end = LocalDateTime.of(LocalDate.now(), LocalTime.MAX);
                    break;
                case MONTH:
                    end = lastDayTime.with(TemporalAdjusters.lastDayOfMonth());
                    break;
                case YEAR:
                    end = lastDayTime.with(TemporalAdjusters.lastDayOfYear());
                    break;
                default:
                    return null;
            }
            return Duration.between(now, end).toMillis() / 1000;
        }
        return null;
    }


    /**
     * 重试
     *
     * @param typeEnum
     * @return
     */
    private SerialNo retry(String typeEnum, int retryNum) {
        //如果没有获取到锁
        try {
            //休眠五秒 后再次获取
            log.info("进入休眠等待重试");
            Thread.sleep(RETRY_TIME);
            //如果重试次数小于最大阈值，重试获取
            if (retryNum < RETRY_LIMIT) {
                return getCacheAvailableSerialNo(typeEnum, retryNum);
            } else {
                log.info("超过最大重试次数");
            }
        } catch (InterruptedException e) {
            log.error("休眠出现异常", e);
        }
        return null;
    }


    /**
     * 获取可用的缓存序号
     *
     * @param typeEnum
     * @return
     */
    private SerialNo getCacheAvailableSerialNo(String typeEnum, int retryNum) {
        retryNum++;
        log.info("第{}次获取配置", retryNum);
        String confKey = getRedisKey(typeEnum, RedisKey.SEQ_NO_CONF_PREFIX);
        String jsonConfStr = redisTemplate.opsForValue().get(confKey);
        if (StringUtils.isBlank(jsonConfStr)) {
            //如果配置为空, 则初始化
            return initSerialNo(typeEnum, retryNum);
        } else {
            try {
                SerialNo serialNo = JSON.parseObject(jsonConfStr, SerialNo.class);
                //查看是否需要重置,
                SerialNo check = checkRestGet(serialNo, typeEnum, retryNum);
                //如果重置了则返回重置的
                if (check != null) {
                    return check;
                }
                //不需要重置，直接返回
                return serialNo;
            } catch (Exception e) {
                log.error("解析序号配置异常", e);
                //解析异常，删除缓存
                clearCacheSerialNo(typeEnum);
                //重新初始化
                return initSerialNo(typeEnum, retryNum);
            }
        }
    }


    private SerialNo checkRestGet(SerialNo serialNo, String typeEnum, int retryNum) {
        SerialNoRuleEnum resetRule = SerialNoRuleEnum.getByName(serialNo.getSeqNoResetRule());
        if (resetRule != null) {
            int nowYear = LocalDate.now().getYear();
            int nowMonth = LocalDate.now().getMonthValue();
            int nowDay = LocalDate.now().getDayOfMonth();
            switch (resetRule) {
                case YEAR: {
                    //如果按年重置
                    if (serialNo.getLastestDate() == null || nowYear != serialNo.getLastestDate().getYear()) {
                        return resetSerialNo(typeEnum, retryNum);
                    }
                    break;
                }
                case MONTH: {
                    //如果是按月重置
                    if (serialNo.getLastestDate() == null || nowYear != serialNo.getLastestDate().getYear()
                            || nowMonth != serialNo.getLastestDate().getMonthValue()) {
                        return resetSerialNo(typeEnum, retryNum);
                    }
                    break;
                }
                case DAY: {
                    //如果是按天重置
                    if (serialNo.getLastestDate() == null || nowYear != serialNo.getLastestDate().getYear()
                            || nowMonth != serialNo.getLastestDate().getMonthValue()
                            || nowDay != serialNo.getLastestDate().getDayOfMonth()) {
                        return resetSerialNo(typeEnum, retryNum);
                    }
                    break;
                }
                case FOREVER: {
                    //不重置
                    break;
                }
                default:
                    //默认不重置
                    log.info("默认不重置");
                    break;
            }
        }
        return null;
    }


    private String getRedisKey(String type, RedisKey redisKey) {
        return String.join(":", FLAG_PREFIX, redisKey.name(), type);
    }


}
