package com.rocogz.common.lock;

import cn.hutool.core.lang.Assert;
import cn.hutool.core.util.IdUtil;
import com.google.common.collect.Sets;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.data.redis.core.StringRedisTemplate;
import org.springframework.stereotype.Component;

import java.util.Set;
import java.util.concurrent.TimeUnit;

/**
 * @author <a href="mailto:zhouyanjie666666@gmail.com">zyj</a>
 * @date 2019/7/24
 */

@Component
@Slf4j
public class RedisDistributedLock extends AbstractDistributedLock {
    /**
     * 锁的value 分割符
     */
    private static final String LOCK_VAL_DELIMITER = "@";


    private StringRedisTemplate redisTemplate;

    private ThreadLocal<Set<String>> lockFlag = new ThreadLocal<>();

    public RedisDistributedLock(StringRedisTemplate redisTemplate) {
        log.info("初始化组件:[分布式锁]");
        this.redisTemplate = redisTemplate;
    }

    @Override
    public boolean lock(final String key, final long expire, int retryCount, final long sleepMillis) {
        log.info("【Lock start: [{}]】 >>  expire:[{} ms], retryCount:[{}], interval:[{} ms]", key, expire, retryCount, sleepMillis);
        //获取一个标志
        String flag = setAndGetFlag(key);
        boolean result = false;
        try {
            //设置redis值
            result = setRedis(key, expire, flag);

            //如果线程没有中断， 并且获取锁是失败的 ，开始重试
            while (!Thread.currentThread().isInterrupted() && !result && retryCount-- > 0) {
                log.info("【Lock sleep: [{} ms]】] ",  sleepMillis);
                try {
                    Thread.sleep(sleepMillis);
                } catch (InterruptedException e) {
                    log.error("redis锁异常:[{}]", key, e);
                    break;
                }
                log.info("【Lock retry: [{}]】>> remain:[{}]", key, retryCount);
                result = setRedis(key, expire, flag);
            }
        } finally {
            //如果没有锁成功， 则删除标志
            if (!result) {
                removeFlag(flag);
            }
            log.info("【Lock end: [{}]】 >>  expire:[{} ms], result:[{}]", key, expire, result);
        }
        return result;
    }

    @Override
    public boolean releaseLock(final String key) {
        log.info("【Release Lock start: [{}]】", key);
        Assert.notBlank(key, "key不能为空");
        boolean result = false;
        Set<String> flags = lockFlag.get();
        if (CollectionUtils.isEmpty(flags)) {
            log.info("当前线程没有锁");
            result = true;
        } else {
            String flag = redisTemplate.opsForValue().get(key);
            if (StringUtils.isBlank(flag)) {
                log.info("【Lock Expired : [{}]】", key);
                result = removeFlagByRedisKey(key);
            } else {
                if (removeFlag(flag)) {
                    result = redisTemplate.delete(key);
                } else {
                    log.error("非本线程锁，不可删除");
                }
            }
        }

        log.info("【Release Lock end: [{}]】 >> result:[{}]", key, result);
        return result;
    }


    private boolean setRedis(final String key, final long expire, final String value) {
        boolean result;
        if (expire > 0) {
            result = redisTemplate.opsForValue().setIfAbsent(key, value, expire, TimeUnit.MILLISECONDS);
        } else {
            result = redisTemplate.opsForValue().setIfAbsent(key, value);
        }
        return result;
    }





    private boolean removeFlagByRedisKey(String redisKey) {
        Set<String> flags = lockFlag.get();
        if (CollectionUtils.isNotEmpty(flags) && StringUtils.isNotBlank(redisKey)) {
            for (String flag: flags) {
                String key = flag.split(LOCK_VAL_DELIMITER)[0];
                if (key.equals(redisKey)) {
                    return removeFlag(flag);
                }
            }
        }
        return false;
    }


    private synchronized void initLockFlag() {
        if (lockFlag.get() == null) {
            lockFlag.set(Sets.newHashSet());
        }
    }


    private String setAndGetFlag(String redisKey) {
        Assert.notBlank(redisKey, "redisKey 不能为空");
        Assert.isTrue(!redisKey.contains(LOCK_VAL_DELIMITER), "暂不支持key名称携带符号:{}", LOCK_VAL_DELIMITER);
        initLockFlag();
        String flag = String.join(LOCK_VAL_DELIMITER, redisKey, IdUtil.fastSimpleUUID());
        boolean result = lockFlag.get().add(flag);
        log.info("[ThreadLocal] add flag:[{}], result:[{}] -> [{}]", flag, result, lockFlag.get());
        return flag;
    }

    private synchronized boolean removeFlag(String flag) {
        boolean result = false;
        Set<String> flagSet  = lockFlag.get();
        if (flagSet != null) {
            result = flagSet.remove(flag);
            //如果都清空了
            if (flagSet.isEmpty()) {
                lockFlag.remove();
            }
        }
        log.info("[ThreadLocal] remove flag:[{}], result:[{}] -> [{}]", flag, result, lockFlag.get());
        return result;
    }
}
