package com.rocogz.syy.order.client;


import com.rocogz.syy.common.PageTable;
import com.rocogz.syy.common.response.Response;
import com.rocogz.syy.order.constant.EvaluateStatisOption;
import com.rocogz.syy.order.constant.OrderStatisOption;
import com.rocogz.syy.order.constant.StatisTimeRangeEnum;
import com.rocogz.syy.order.constant.OtherInfoOption;
import com.rocogz.syy.order.constant.trace.BussiSystemEnum;
import com.rocogz.syy.order.constant.trace.BussiTypeEnum;
import com.rocogz.syy.order.constant.trace.CallApiEnum;
import com.rocogz.syy.order.dto.AdminSearchParamDto;
import com.rocogz.syy.order.dto.OrderSummaryStatistic;
import com.rocogz.syy.order.dto.OrderWriteOffDto;
import com.rocogz.syy.order.dto.appoint.AppointAdminSearchParamDto;
import com.rocogz.syy.order.dto.appoint.AppointWriteOffDto;
import com.rocogz.syy.order.dto.appoint.AppointWriteOffPreparedDataDto;
import com.rocogz.syy.order.dto.appoint.CreateAppointOrderDto;
import com.rocogz.syy.order.dto.evaluate.AdminSearchEvaluationParamDto;
import com.rocogz.syy.order.dto.evaluate.OrderEvaluateSummaryStatistic;
import com.rocogz.syy.order.dto.oil.*;
import com.rocogz.syy.order.dto.pay.WxPayUnifiedOrderParamDto;
import com.rocogz.syy.order.dto.pay.WxPayUnifiedOrderResultDto;
import com.rocogz.syy.order.entity.evaluate.OrderEvaluate;
import com.rocogz.syy.order.entity.oil.OilDepositRecord;
import com.rocogz.syy.order.entity.oil.OrderOilDeposit;
import com.rocogz.syy.order.entity.orders.OrderLog;
import com.rocogz.syy.order.entity.orders.OrderPayment;
import com.rocogz.syy.order.entity.orders.SaleOrder;
import com.rocogz.syy.order.entity.pay.WxPayRecord;
import com.rocogz.syy.order.entity.trace.OrderErrorTrace;
import com.rocogz.syy.order.entity.trace.OrderThirdTrace;
import org.springframework.cloud.openfeign.SpringQueryMap;
import org.springframework.web.bind.annotation.*;

import java.time.LocalDateTime;
import java.util.List;
import java.util.Map;

/**
 * 订单接口
 *
 * @author：zhangin
 * @since：2020/3/25
 */
public interface IOrderClientService {

    /**
     * 创建订单
     *
     * @param order
     */
    @PostMapping("/api/order/create")
    Response createOrder(@RequestBody SaleOrder order);

    /**
     * 后台管理->订单搜索
     *
     * @param searchParamDto
     */
    @GetMapping("/api/order/adminSearch")
    PageTable<SaleOrder> adminSearch(@SpringQueryMap AdminSearchParamDto searchParamDto);

    /**
     * 获取订单主信息
     *
     * @param orderCode 订单编号
     */
    @GetMapping("/api/order/getOrderMainInfo/{orderCode}")
    Response<SaleOrder> getOrderMainInfo(@PathVariable("orderCode") String orderCode);

    /**
     * 查询订单主信息,如果需要其他信息也一起加载
     *
     * @param ordercode     订单编号
     * @param otherInfoList 需要加载的其他信息选项,用来实现按需加载,如果需要全部其他信息,则传入 ALL
     **/
    @GetMapping("/api/order/getOrderWithOtherInfo/{orderCode}")
    Response<SaleOrder> getOrderWithOtherInfo(@PathVariable("orderCode") String orderCode,
                                              @RequestParam("otherInfoList") List<OtherInfoOption> otherInfoList);


    /**
     * 车主端小程序 搜索订单
     *
     * @param searchParamDto
     * @return
     */
    @GetMapping("/api/order/weappSearch")
    PageTable<SaleOrder> weappSearch(@RequestParam("userCode") String userCode,
                                     @RequestParam(value = "status", required = false) String status,
                                     @RequestParam(value = "limit", defaultValue = "10") Integer limit,
                                     @RequestParam(value = "page", defaultValue = "1") Integer page);


    /**
     * 商户端小程序 搜索订单
     * @param storeCode          门店编号[必须]
     * @param keyword：订单号/车牌号/券码 [选传]
     * @param orderTypeList 订单类型列表[选传]
     * @param statusList   订单状态列表 [选传]
     * @param appointStartTime 预约开始时间[选传]
     * @param appointEndTime   预约开始结束时间[选传]
     */
    @GetMapping("/api/order/merchantWeappSearch")
    PageTable<SaleOrder> merchantWeappSearch(@RequestParam(value = "storeCode") String storeCode,
                                             @RequestParam(value = "keyword", required = false) String keyword,
                                             @RequestParam(value = "orderTypeList",required = false) List<String> orderTypeList,
                                             @RequestParam(value = "statusList", required = false) List<String> statusList,
                                             @RequestParam(value = "appointStartTime",required = false) String appointStartTime,
                                             @RequestParam(value = "appointEndTime",required = false) String appointEndTime,
                                             @RequestParam(value = "limit", defaultValue = "10") Integer limit,
                                             @RequestParam(value = "page", defaultValue = "1") Integer page);

    /**
     * 逻辑删除订单
     *
     * @param orderCode 订单号
     * @param userCode  如果是weapp端调用,则需要传入用户编号,用来判断是否是车主自己的订单
     * @return
     */
    @GetMapping("/api/order/delete/{orderCode}")
    Response<String> deleteLogical(@PathVariable("orderCode") String orderCode,
                                   @RequestParam(value = "userCode", required = false) String userCode);


    /**
     * 关闭 超时未支付的订单
     * @param orderCode 订单号
     * @return 如果关单成功, 则返回需要退回的用户券码
     */
    @GetMapping("/api/order/closeOrder/{orderCode}")
    Response<List<String>> closeTimeOutUnpayOrder(@PathVariable("orderCode") String orderCode);


    /**
     * 订单的操作日志列表
     *
     * @param orderCode 订单编号
     * @return 返回结果按照id倒序排列
     */
    @GetMapping("/api/order/log/{orderCode}")
    Response<List<OrderLog>> findLogList(@PathVariable("orderCode") String orderCode);


    /**
     * 定时把过了过期时间的订单设置为已过期
     * @return
     */
    @GetMapping("/api/order/scheduleExpireOrder")
    Response<String> scheduleExpireOrder();



    // ======  订单统计 相关接口 start   ===========>

    /**
     * 统计 门店/车主 订单数量, 统计结果 按照订单状态分组
     *
     * @param storeCode 门店编号 [可选]
     * @param userCode  用户编号 [可选]
     * @return Map<订单状态   ,   订单数量>
     */
    @GetMapping("/api/order/statis/countGroupByStatus")
    Response<Map<String, Long>> countOrderNumGroupByStatus(@RequestParam(value = "storeCode", required = false) String storeCode,
                                                           @RequestParam(value = "userCode", required = false) String userCode);

    /**
     * 商户端 统计销售业绩
     *
     * @param statisOptionList
     */
    @PostMapping("/api/order/statis/summarize")
    Response<Map<StatisTimeRangeEnum, OrderSummaryStatistic>> summarizeOrderByStoreCode(@RequestBody List<OrderStatisOption> statisOptionList);

    /**
     * 商户端: 统计今日最新：未核销订单数量、预约数、评论数
     *
     * @param statisOptionList
     */
    @GetMapping("/api/order/statis/countNewest/{storeCode}")
    Response<Map<String,Long>> countNewestOrderByStoreCode(@PathVariable("storeCode") String storeCode);

    // ======  订单核销 相关接口 start   ===========>

    /**
     * 第二阶段：核销之前数据预校验,只有校验通过后才能进入核销页面
     *
     * @param orderDto
     */
    @PostMapping("/api/order/writeOff/validateBeforeWriteOff")
    Response<String> validateBeforeWriteOff(@RequestBody OrderWriteOffDto orderDto);

    /**
     * 第三阶段：真正提交核销 & 创建订单：（包括保存订单前数据校验,生成订单、保存订单）
     *
     * @param orderDto
     * @return 订单创建成功后，会返回订单对象 SaleOrder
     */
    @PostMapping("/api/order/writeOff/createOrder")
    Response<SaleOrder> createOrderWhenWriteOff(@RequestBody OrderWriteOffDto orderDto);

    /**
     * 更新订单状态 & 保存操作日志
     * 注意：此接口只针对单个订单的状态更新
     *
     * @param log （orderCode 与 afterStatus 必须传入）
     */
    @PostMapping("/api/order/writeOff/updateOrderStatus")
    Response<String> updateStatusWithSaveLog(@RequestBody List<OrderLog> logList);


    // ======  订单评价 相关接口 start   ===========>

    /**
     * @param evaluate(传入参数：订单号、评价来源,评价人名,评价内容,评分,印象标签，图片，kpi)
     * @return 返回创建成功的评价对象
     */
    @PostMapping("/api/order/evaluate/create")
    Response<OrderEvaluate> createEvaluate(@RequestBody OrderEvaluate evaluate);

    /**
     * 获取评价信息
     *
     * @param orderCode     订单号
     * @param loadOtherFlag null或0: 不加载其他评价信息
     *                      1: 加载评价其他信息（标签、kpi、前端可见图片）
     *                      2: 加载评价其他信息（标签、kpi、全部图片）
     */
    @GetMapping("/api/order/evaluate/{orderCode}")
    Response<OrderEvaluate> getEvaluateByOrderCode(@PathVariable("orderCode") String orderCode,
                                                   @RequestParam(value = "loadOtherFlag", required = false) Integer loadOtherFlag);


    /**
     * 删除评价
     *
     * @param orderCode  订单号
     * @param updateUser 删除操作人
     * @return
     */
    @GetMapping("/api/order/evaluate/delete/{orderCode}")
    Response<String> deleteEvaluate(@PathVariable("orderCode") String orderCode,
                                    @RequestParam(value = "updateUser", required = false) String updateUser);

    /**
     * 精选/取消精选
     *
     * @param orderCode  订单号
     * @param updateUser 删除操作人
     * @return
     */
    @GetMapping("/api/order/evaluate/togglePerfect/{orderCode}")
    Response<String> toggleEvaluatePerfect(@PathVariable("orderCode") String orderCode,
                                           @RequestParam(value = "updateUser", required = false) String updateUser);

    /**
     * 设置 可见/ 不可见
     *
     * @param orderCode  订单号
     * @param updateUser 删除操作人
     * @return
     */
    @GetMapping("/api/order/evaluate/toggleVisibility/{orderCode}")
    Response<String> toggleEvaluateVisibility(@PathVariable("orderCode") String orderCode,
                                              @RequestParam(value = "updateUser", required = false) String updateUser);


    /**
     * 管理后台 搜索评价
     *
     * @param searchParamDto
     */
    @GetMapping("/api/order/evaluate/adminSearch")
    PageTable<OrderEvaluate> adminSearchEvaluation(@SpringQueryMap AdminSearchEvaluationParamDto searchParamDto);


    /**
     * 车主端小程序 搜索评价
     *
     * @param code            门店编号或 服务项目编号 等
     * @param statisWay       统计方式，按什么维度统计
     * @param visible         是否会员可见,null 表示不限制
     * @param perfect         是否精选，null表示不限制
     * @param impressionLabel 要查询的印象标签,空表示不限制
     * @param
     */
    @GetMapping("/api/order/evaluate/weappSearch")
    PageTable<OrderEvaluate> weappSearchEvaluation(@RequestParam(value = "code", required = false) String code,
                                                   @RequestParam(value = "statisWay", required = false) EvaluateStatisOption.EvaluateStatisWay statisWay,
                                                   @RequestParam(value = "visible", required = false) Boolean visible,
                                                   @RequestParam(value = "perfect", required = false) Boolean perfect,
                                                   @RequestParam(value = "impressionLabel", required = false) String impressionLabel,
                                                   @RequestParam(value = "limit", defaultValue = "20") int limit,
                                                   @RequestParam(value = "page", defaultValue = "1") int page);


    /**
     * 车主端小程序 搜索评价,统计评价数量
     *
     * @param code      门店编号或 服务项目编号 等
     * @param statisWay 统计方式，按什么维度统计
     * @param visible   是否会员可见,null 表示不限制
     * @param perfect   是否精选，null表示不限制
     */
    @GetMapping("/api/order/evaluate/weappSearchCount")
    Response<Integer> weappSearchCountEvaluation(@RequestParam(value = "code", required = false) String code,
                                                 @RequestParam(value = "statisWay", required = false) EvaluateStatisOption.EvaluateStatisWay statisWay,
                                                 @RequestParam(value = "visible", required = false) Boolean visible,
                                                 @RequestParam(value = "perfect", required = false) Boolean perfect,
                                                 @RequestParam(value = "impressionLabel", required = false) String impressionLabel);


    // 评价统计相关
    @PostMapping("/api/order/evaluate/statis/summarize")
    Response<Map<String, OrderEvaluateSummaryStatistic>> summarizeEvaluation(@RequestBody List<EvaluateStatisOption> statisOptionList);


    //电子加油券 相关接口 start   ===========>

    /**
     * 加油电子券下订单
     *
     * @return 创建成功时返回订单对象
     */
    @PostMapping("/api/order/oil/createOrder")
    Response<SaleOrder> createOilTicketOrder(@RequestBody OilOrderCreateDto orderDto);

    /**
     * 第一步：代金券+现金  加油电子券下订单
     * @param orderDto
     */
    @PostMapping("/api/order/oil/createWithCashOrder")
    Response<SaleOrder> createWithCashOilTicketOrder(@RequestBody OilOrderWithCashCreateDto orderDto);


    /**
     * 使用电子券支付 加油电子券订单
     *
     * @return 支付成功时返回OilOrderPayedResultDto对象, 对象中每个属性都有值
     */
    @PostMapping("/api/order/oil/payOrder")
    Response<OilOrderPayedResultDto> payOilTicketOrder(@RequestBody OilOrderPayDto payDto);

    /**
     * 使用现金支付 加油订单
     * @param payDto
     * @return
     */
    @PostMapping("/api/order/oil/cashPayOrder")
    Response payOilTickOrderByCash(@RequestBody OilOrderCashPayDto payDto);

    /**
     * 当系统5次自动尝试以后还是不能创建油卡交易单，则可以等油卡系统恢复正常后，手工尝试再去创建油卡订单
     *
     * @param orderCode 车服订单号
     */
    @GetMapping("/api/order/oil/manualCreateOilTrade/{orderCode}")
    Response<String> manualIssueCreateOilTrade(@PathVariable("orderCode") String orderCode);


    /**
     * 当油卡系统充值交易订单创建成功后,调用此方法更新此系统状态
     *
     * @param orderCode 车服订单号
     * @param tradeNo   油卡系统充值交易号
     * @return
     */
    @PostMapping("/api/order/oil/deposit/updateDepositOrder")
    Response<String> updateDepositWhenOilCreatedOrder(@RequestParam("orderCode") String orderCode,
                                                      @RequestParam("tradeNo") String tradeNo);

    /**
     * 查找待补发尝试的交易单
     *
     * @return
     */
    @GetMapping("/api/order/oil/deposit/findPendingRetryDeposit")
    Response<List<OrderOilDeposit>> findPendingRetryDeposit();


    /**
     * 获得充值单详情
     *
     * @param tradeNo       交易号
     * @param isLoadRecords 是否加载充值记录流水
     * @return
     */
    @GetMapping("/api/order/oil/deposit/getDepositOrderDetail/{tradeNo}")
    Response<OrderOilDeposit> getDepositOrderDetailByTradeNo(@PathVariable("tradeNo") String tradeNo,
                                                             @RequestParam(value = "isLoadRecords", required = false) boolean isLoadRecords);

    /**
     * 更新尝试次数和失败原因
     *
     * @param deposit
     * @return
     */
    @PostMapping("/api/order/oil/deposit/updateRetryTimes")
    Response<String> updateRetryTimesAndReason(@RequestBody OrderOilDeposit deposit);

    /**
     * 保存成功充值的 充值记录
     *
     * @param depositRecord 充值记录
     * @return 返回成功处理的交易流水号bizNo和状态码
     * （1）普通处理成功返回状态码： ResponseCode.SUCCEED
     * （2）充值完成处理成功返回状态码：OrderOilDeposit.DEPOSIT_FINISHED_RESP_CODE
     */
    @PostMapping("/api/order/oil/depositRecord/save")
    Response<OilDepositRecord> saveDepositRecord(@RequestBody OilDepositRecord depositRecord);

    /**
     * 车主端小程序 搜索充值记录,按照时间倒序排列
     *
     * @param userCode      会员编号
     * @param orderCode     车服订单号
     * @param depositMobile 充值手机号
     */
    @GetMapping("/api/order/oil/depositRecord/weappSearch")
    PageTable<OilDepositRecord> weappSearchDepositRecord(@RequestParam(value = "userCode", required = true) String userCode,
                                                         @RequestParam(value = "orderCode", required = false) String orderCode,
                                                         @RequestParam(value = "depositMobile", required = false) String depositMobile,
                                                         @RequestParam(value = "limit", defaultValue = "20") int limit,
                                                         @RequestParam(value = "page", defaultValue = "1") int page);

    /**
     * 车主端小程序 合计充值到账金额
     * @param userCode 会员编号
     * @param orderCode 车服订单号
     * @param depositMobile 充值手机号
     */
    @GetMapping("/api/order/oil/depositRecord/summarize")
    Response<DepositSummaryStatistic> summarizeDeposit(@RequestParam(value = "userCode",required = true) String userCode,
                                                      @RequestParam(value = "orderCode",required = false) String orderCode,
                                                      @RequestParam(value = "depositMobile",required = false) String depositMobile);



    /**
     * 后台管理->电子劵订单搜索
     *
     * @param searchParamDto
     */
    @GetMapping("/api/order/oil/adminSearch")
    PageTable<SaleOrder> adminOilOrderSearch(@SpringQueryMap OilAdminSearchParamDto searchParamDto);


    /**
     * 后台管理->电子劵订单支付信息信息
     * @param orderCode
     * @return
     */
    @GetMapping("/api/order/oil/getOrderOilPaymentInfo/{orderCode}")
    Response<OrderPayment> getOrderOilPaymentInfo(@PathVariable("orderCode") String orderCode);

    /**
     * 后台管理->电子劵订单支付信息信息
     * @param orderCode
     * @return
     */
    @GetMapping("/api/order/oil/getOrderOilPaymentLogList/{orderCode}")
    Response<OrderPayment> getOrderOilPaymentLogList(@PathVariable("orderCode") String orderCode);

    /**
     * 后台管理->电子劵订单充值记录列表
     * @param orderCode
     * @return
     */
    @GetMapping("/api/order/oil/getOilDepositRecordList/{orderCode}")
    Response<List<OilDepositRecord>> getOilDepositRecordList(@PathVariable("orderCode") String orderCode);





    //预约订单 相关接口 start   ===========>

    /**
     *统计门店服务项目在各时间段已预约订单数量
     *@param storeServiceCode 门店服务项目编号
     *@param timeRangeList，时间范围格式：2010-11-12 09:01-11:30
     */
    @GetMapping("/api/order/appoint/countAppintNum")
    Response<Map<String,Integer>> countAppintNum(@RequestParam("storeServiceCode") String storeServiceCode,
                                                 @RequestParam("timeRangeList") List<String> timeRangeList);

    /**
     * 获取已预约的数量
     * @param storeServiceCode 门店服务编号
     * @param startTime 预约开始时间 时间格式：YYYY-MM-dd HH:mm:ss
     * @param endTime 预约结束时间 时间格式：YYYY-MM-dd HH:mm:ss
     */
    @GetMapping("/api/order/appoint/getAppointedNum")
    Response<Integer> getAppointedNum(@RequestParam("storeServiceCode") String storeServiceCode,
                                      @RequestParam("startTime") String startTime,
                                      @RequestParam("endTime")  String endTime);


    /**
     * 更新预约订单的会员账号信息 （当绑定车主端触发）
     * @param userCode 会员编号
     * @param openId  会员Openid
     * @param userMobile 会员手机号
     */
    @GetMapping("/api/order/appoint/updateAppointUserInfo")
    Response<String> updateUserInfoByAppoitMobileWhenBound(@RequestParam("userCode") String userCode,
                                                                  @RequestParam(value = "openId") String openId,
                                                                  @RequestParam(value = "userMobile") String userMobile);


    /**
     * 创建预约订单
     * @param orderDto
     * @return
     */
    @PostMapping("/api/order/appoint/createAppoint")
    Response<String> createAppointOrder(@RequestBody CreateAppointOrderDto orderDto);

    /**
     * 预约订单核销之前 准备数据
     * @param thirdWriteOffCode 第三方订单核销码
     */
    @GetMapping("/api/order/appoint/prepareDataBeforeWriteOff/{thirdWriteOffCode}")
    Response<AppointWriteOffPreparedDataDto> prepareDataBeforeWriteOffAppointment(@PathVariable("thirdWriteOffCode") String thirdWriteOffCode);

    /**
     *核销预约订单
     * @param writeOffDto
     * @return
     */
    @PostMapping("/api/order/appoint/doWriteOff")
    Response<SaleOrder> writeOffAppointOrder(@RequestBody AppointWriteOffDto writeOffDto);


    /**
     * 内部取消订单：【车主端/管理端】取消订单
     *
     * @param orderCode  roco车服订单号
     * @param actionUser 操作人
     * @param channel     操作渠道 {@link com.rocogz.syy.order.constant.OrderConstant.DictChannel}
     */
    @GetMapping("/api/order/appoint/internelCancelOrder/{orderCode}")
    Response<String> internelCancelAppointOrder(@PathVariable("orderCode") String orderCode,
                                                @RequestParam(value = "actionUser",required = false) String actionUser,
                                                @RequestParam(value = "channel") String channel);


    /**
     * 外部调用取消订单接口：例如：车务平台发起的取消订单
     * @param thirdOrderCode 人保车务平台订单号
     * @param thirdWriteOffCode 人保车务平台的订单核销码
     */
    @GetMapping("/api/order/appoint/externelCancelOrder/{thirdOrderCode}")
    Response<String> externelCancelAppointOrder(@PathVariable("thirdOrderCode") String thirdOrderCode,@RequestParam("thirdWriteOffCode") String thirdWriteOffCode);


    /**
     * 后台搜索 预约订单
     * @param searchParamDto
     * @return
     */
    @GetMapping("/api/order/appoint/adminSearch")
    PageTable<SaleOrder> adminSearchAppointOrder(@SpringQueryMap AppointAdminSearchParamDto searchParamDto);


    //服务调用 跟踪   相关接口 start   ===========>

    @GetMapping("/api/order/trace/error")
    PageTable<OrderErrorTrace> searchErrorTrace(@RequestParam(value = "coreCode", required = false) String coreCode,
                                                @RequestParam(value = "bussType", required = false) BussiTypeEnum bussType,
                                                @RequestParam(value = "callApi", required = false) CallApiEnum callApi,
                                                @RequestParam(value = "system", required = false) BussiSystemEnum system,
                                                @RequestParam(value = "createStartTime", required = false) LocalDateTime createStartTime,
                                                @RequestParam(value = "createEndTime", required = false) LocalDateTime createEndTime,
                                                @RequestParam(value = "limit", defaultValue = "20") int limit,
                                                @RequestParam(value = "page", defaultValue = "1") int page);


    @PostMapping("/api/order/trace/error/save")
    Response<String> saveErrorTrace(@RequestBody OrderErrorTrace callTrace);


    @GetMapping("/api/order/trace/thirdApi")
    PageTable<OrderThirdTrace> searchThirdCallTrace(@RequestParam(value = "coreCode", required = false) String coreCode,
                                                    @RequestParam(value = "callApiPath", required = false) String callApiPath,
                                                    @RequestParam(value = "sourcePlatform", required = false) String sourcePlatform,
                                                    @RequestParam(value = "targetPlatform", required = false) String targetPlatform,
                                                    @RequestParam(value = "createStartTime", required = false) LocalDateTime createStartTime,
                                                    @RequestParam(value = "createEndTime", required = false) LocalDateTime createEndTime,
                                                    @RequestParam(value = "limit", defaultValue = "20") int limit,
                                                    @RequestParam(value = "page", defaultValue = "1") int page);

    @PostMapping("/api/order/trace/thirdApi/save")
    Response<String> saveThirdCallTrace(@RequestBody OrderThirdTrace callTrace);

    //微信支付记录 相关接口 start   ===========>
    /**
     * Query wx pay record by out trade no and status response.
     *
     * @param outTradeNo   the out trade no
     * @param wxTradeState the wx trade state
     * @return the response
     */
    @GetMapping("/api/order/wx-pay-record/queryByOutTradeNoAndStatus")
    Response<WxPayRecord> queryWxPayRecordByOutTradeNoAndStatus(@RequestParam("outTradeNo") String outTradeNo,
                                                                @RequestParam(value = "wxTradeState", required = false) String wxTradeState);

    /**
     * Update wx pay record by id response.
     *
     * @param wxPayRecord the wx pay record
     * @return the response
     */
    @PostMapping("/api/order/wx-pay-record/updateById")
    Response updateWxPayRecordById(@RequestBody WxPayRecord wxPayRecord);

    //微信支付记录 相关接口 end   ===========>

    //微信支付API 相关接口 start   ===========>

    /**
     * 统一下单(详见https://pay.weixin.qq.com/wiki/doc/api/jsapi.php?chapter=9_1)
     * 在发起微信支付前，需要调用统一下单接口，获取"预支付交易会话标识"
     * 接口地址：https://api.mch.weixin.qq.com/pay/unifiedorder
     *
     * @param paramDto the param dto
     * @return the response
     */
    @PostMapping("/api/order/pay/wx/unifiedOrder")
    Response<WxPayUnifiedOrderResultDto> unifiedWxPayOrder(@RequestBody WxPayUnifiedOrderParamDto paramDto);

    /**
     * 微信支付下单成功后未立即支付，再次调起支付接口
     *
     * @param outTradeNo roco车服的订单编号
     * @return response
     */
    @GetMapping("/api/order/pay/wx/repayOrder")
    Response<WxPayUnifiedOrderResultDto> repayWxPayOrder(@RequestParam("outTradeNo") String outTradeNo);

    /**
     * <pre>
     * 关闭订单
     * 应用场景
     * 以下情况需要调用关单接口：
     * 1. 商户订单支付失败需要生成新单号重新发起支付，要对原订单号调用关单，避免重复支付；
     * 2. 系统下单后，用户支付超时，系统退出不再受理，避免用户继续，请调用关单接口。
     * 注意：订单生成后不能马上调用关单接口，最短调用时间间隔为5分钟。
     * 接口地址：https://api.mch.weixin.qq.com/pay/closeorder
     * 是否需要证书：   不需要。
     * </pre>
     *
     * @param outTradeNo 商户系统内部的订单号
     * @return the response
     */
    @GetMapping("/api/order/pay/wx/closeOrder/{outTradeNo}")
    Response closeWxPayOrder(@PathVariable("outTradeNo") String outTradeNo);

    //微信支付API 相关接口 end   ===========>

}

