package com.rocogz.syy.order.util;

import com.google.common.collect.Lists;
import com.rocogz.syy.order.dto.evaluate.EvaluatedOrderDto;
import com.rocogz.syy.order.entity.evaluate.OrderEvaluate;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.math.NumberUtils;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.text.DecimalFormat;
import java.text.NumberFormat;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Optional;

import static java.util.stream.Collectors.toMap;

/**
 * 数字处理工具类
 **/
public final class ArithUtil {

    private static final int DEF_DIV_SCALE = 10;

    /**
     * 提供精确的加法运算。
     *
     * @param v1 被加数
     * @param v2 加数
     * @return 两个参数的和
     */
    public static double add(double v1, double v2) {
        return v1 + v2;
    }

    /**
     * 提供精确的减法运算。
     *
     * @param v1 被减数
     * @param v2 减数
     * @return v1 - v2 的差
     */
    public static double sub(double v1, double v2) {
        return v1 - v2;
    }

    /**
     * 提供精确的乘法运算
     *
     * @param v1 被减数
     * @param v2 减数
     * @return 两个参数的乘积
     */
    public static double mutiply(double v1, double v2) {
        return v1 * v2;
    }

    /**
     * 提供相对精确的除法运算,当除不净时，保留10小数，小数点10位后的数字四舍五入
     *
     * @param v1 被除数
     * @param v2 除数
     * @return 返回v1/v2的商，最多保留10小数,小数点10位后的数字四舍五入
     */
    public static double div(double v1, double v2) {
        return div(v1, v2, DEF_DIV_SCALE);
    }

    /**
     * 提供相对精确的除法运算,当除不净时，保留scale位小数，小数点scale位后的数字四舍五入
     *
     * @param v1    被除数
     * @param v2    除数
     * @param scale 表示要精确到小数点后几位
     * @return 返回v1/v2的商，最多保留scale小数,小数点scale位后的数字四舍五入
     */
    public static double div(double v1, double v2, int scale) {
        if (scale < 0) {
            throw new IllegalArgumentException("The scale must be a positive integer or zero");
        }
        BigDecimal d1 = new BigDecimal(Double.toString(v1));
        BigDecimal d2 = new BigDecimal(Double.toString(v2));
        return d1.divide(d2, scale, BigDecimal.ROUND_HALF_UP).doubleValue();
    }

    public static String divideToPercent(BigDecimal a, BigDecimal b) {
        BigDecimal result =  bigDecimalDivide(a,b,4).multiply(new BigDecimal("100"));
        return formatMoney(result) +"%";
    }

    /**
     * 提供精确的小数位四舍五入处理。
     *
     * @param v     需要四舍五入的数字
     * @param scale 小数点后保留几位
     * @return 四舍五入后的结果
     */
    public static double round(double v, int scale) {
        return roundToBigDecimal(v, scale).doubleValue();
    }


    /**
     * 提供精确的小数位四舍五入处理。
     *
     * @param value     需要四舍五入的数字
     * @param scale 小数点后保留几位
     * @return 四舍五入后的结果
     */
    public static BigDecimal roundToBigDecimal(double value, int scale) {
        return roundToBigDecimal(new BigDecimal(String.valueOf(value)), scale);
    }


    /**
     * 提供精确的小数位四舍五入处理。
     * @param value
     * @param scale 小数点后保留几位
     * @return
     */
    public static BigDecimal roundToBigDecimal(BigDecimal value, int scale) {
        if (scale < 0) {
            throw new IllegalArgumentException("The scale must be a positive integer or zero");
        }

        return value.divide(BigDecimal.ONE, scale, BigDecimal.ROUND_HALF_UP);
    }


    /**
     * value舍入 成 保留两位小数
     * 12.4494=>12.45
     */
    public static BigDecimal roundToTwoScaleBigDecimal(BigDecimal value) {
        if (value == null) return null;
        return value.setScale(2, BigDecimal.ROUND_HALF_UP);
    }


    /**
     * 取一半的值，即值除以2 如：传100 返回50
     *
     * @param num 被除数
     * @return
     */
    public static BigDecimal halfValue(BigDecimal num) {
        if (num == null){
            return null;
        }
        return num.divide(new BigDecimal(2), 2, BigDecimal.ROUND_HALF_UP);
    }

    /**
     * 向下取整数 12.9 =>12
     */
    public static int roundDown(double v) {
        BigDecimal b = new BigDecimal(Double.toString(v));
        return b.divide(BigDecimal.ONE, 0, BigDecimal.ROUND_DOWN).intValue();
    }

    /**
     * 向上取整数 12.1 =>13
     */
    public static int roundUp(double v) {
        BigDecimal b = new BigDecimal(Double.toString(v));
        return b.divide(BigDecimal.ONE, 0, BigDecimal.ROUND_UP).intValue();
    }

    /**
     * 四舍五入取整
     */
    public static int roundHalfUp(double v) {
        return new BigDecimal(v).setScale(0, BigDecimal.ROUND_HALF_UP).intValue();
    }

    /**
     * 返回现金表示形式保留两位小数
     * 如果money是100.53600,返回￥100.54; 100 = >￥100.00
     */
    public static String getCurrency(double money) {
        NumberFormat formatter = NumberFormat.getCurrencyInstance(Locale.CHINA);
        return formatter.format(money);
    }

    /**
     * 返回现金表示形式保留两位小数(去掉人民币符号 ¥)
     * 如果money是100.53600,返回￥100.54;
     * 100 = >100.00,
     * 13131.2= >13,131.20
     */
    public static String getNoSymbolCurrency(double money) {
        String fmtMoney = getCurrency(money);
        return fmtMoney.substring(1,fmtMoney.length());
    }


    public static double format(BigDecimal num, int len) {
        if (num == null) {
            return 0d;
        }
        len = len < 0 ? 2 : len;
        String pattern = "#." + StringUtils.rightPad("", len, '0');
        DecimalFormat df = new DecimalFormat(pattern);
        return Double.valueOf(df.format(num));
    }


    /***
     * 处理科学计数法表示的金额 <br/>
     * 100.0 =>100 <br/>
     * 1.24e2 =>124
     * 12.4494=>12.45
     * @param amount
     * @return
     */
    public static String formatMoney(double amount) {
        String money;
        int intPart = (int) amount;
        if (amount != intPart) {
            NumberFormat fmt = NumberFormat.getInstance();
            fmt.setMaximumFractionDigits(2);
            money = fmt.format(amount);
            money = money.replace(",", StringUtils.EMPTY);
        } else {
            money = String.valueOf(intPart);
        }
        return money;
    }

    /***
     * 处理科学计数法表示的金额 <br/>
     * 100.0 =>100 <br/>
     * 1.24e2 =>124
     * 12.4494=>12.45
     * @param amount 金额
     * @return 返回字符串
     */
    public static String formatMoney(BigDecimal amount) {
        if (amount == null) {
            return "0";
        }
        return formatMoney(amount.doubleValue());
    }



    /**
     * 百分数 转成 小数 例如： 12.3% => 0.123
     * @param percent 分数 转换成小数
     * @return
     */
    public static double percentToDouble(String percent) {
        String floatNum = percent.substring(0, percent.length() - 1);
        double n = NumberUtils.toDouble(floatNum);
        return n = n / 100;
    }

    /**
     * BigDecimal类型的乘法
     *
     * @param v1
     * @param v2
     * @return
     */
    public static BigDecimal bigDecimalMultiply(BigDecimal v1, BigDecimal... v2) {
        if (v1 == null || v2 == null) {
            return new BigDecimal(0);
        }
        if (v2.length == 0) {
            return v1;
        }
        BigDecimal product = v1;
        for (BigDecimal v : v2) {
            product = product.multiply(v);
        }
        return product;
    }

    /**
     * BigDecimal类型的除法
     *
     * @param v1 除数
     * @param v2 被除数
     * @return
     */
    public static BigDecimal bigDecimalDivide(BigDecimal v1, BigDecimal v2,int scale) {
        if (v1 == null || v2.compareTo(BigDecimal.ZERO) == 0) {
            return BigDecimal.ZERO;
        }
        return v1.divide(v2, scale, RoundingMode.HALF_UP);
    }

    /**
     * BigDecimal类型相加
     *
     * @param vs 不定长数值
     * @return
     */
    public static BigDecimal bigDecimalAdd(BigDecimal... vs) {
        if (vs == null || vs.length == 0) {
            return BigDecimal.ZERO;
        }
        BigDecimal sum = BigDecimal.ZERO;
        for (BigDecimal v : vs) {
            if (v != null) {
                sum = sum.add(v);
            }
        }
        return sum;
    }

    /**
     * BigDecimal类型减法
     *
     * @param v  减数
     * @param vs 被减数
     * @return 值
     */
    public static BigDecimal bigDecimalSubtract(BigDecimal v, BigDecimal... vs) {
        if (v == null) {
            throw new RuntimeException("减数不能为空！");
        }
        if (vs == null || vs.length == 0) {
            return v;
        }
        for (BigDecimal val : vs) {
            if (val != null) {
                v = v.subtract(val);
            }
        }
        return v;
    }

    //判断值value 是否在某个范围内
    public static boolean betweenSection(BigDecimal value, BigDecimal lower, BigDecimal upper) {
        return value.compareTo(lower) >= 0 && value.compareTo(upper) <= 0;
    }

    //v1的v2次幂:比如x的n次幂 其中v1,v2都不能为0
    public static BigDecimal pow(double v1, double v2) throws Exception {
        if (v1 == 0 || v2 == 0) {
            throw new Exception("底数或次方数不能为零!");
        }

        return new BigDecimal(Math.pow(v1, v2));
    }
}